#!/usr/bin/env python3
#BEGIN_LEGAL
#
#Copyright (c) 2024 Intel Corporation
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#  
#END_LEGAL
import argparse
import ctypes
from pathlib import Path

# Define the root directory for the XED library
xed_root: Path = Path(__file__).parents[2].resolve()

# Import XED specific constants amd types (better to be autogenerated in the future)
XED_MACHINE_MODE_LONG_64 = 1
XED_ADDRESS_WIDTH_64b = 8

class xed_decoded_inst_t(ctypes.Structure):
    _fields_ = [
        ("_data", ctypes.c_ubyte * 512)  # Oversized buffer to prevent potential buffer overflows
    ]

class xed_state_t(ctypes.Structure):
    _fields_ = [
        ("_data", ctypes.c_ubyte * 512)  # Oversized buffer to prevent potential buffer overflows
    ]

xed_enum = int  # Represents an enum type in XED


def parse_arguments() -> argparse.Namespace:
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser(description='Python example using XED via ctypes')
    parser.add_argument('--xed-lib',
                        dest='lib_path',
                        help='Path to XED shared object library. Default to obj/xed.{so,dll}',
                        type=Path,
                        default=None)
    
    env = parser.parse_args()

    # Default to the expected library path
    if env.lib_path is None:
        env.lib_path = xed_root / 'obj/xed.dll'
        if not env.lib_path.exists():
            env.lib_path = env.lib_path.with_name('libxed.so')

    return env

def setup_ctypes_functions(lib: ctypes.CDLL) -> None:
    """Set up the ctypes function signatures for the required XED functions.
    (Preferable to be autogenerated)
    
    Args:
        lib (ctypes.CDLL): The loaded XED shared library.
    """
    lib.xed_iclass_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_iclass_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_category_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_category_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_extension_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_extension_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_isa_set_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_isa_set_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_error_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_error_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_iform_enum_t2str_py.restype = ctypes.c_char_p
    lib.xed_iform_enum_t2str_py.argtypes = [ctypes.c_int]

    lib.xed_decoded_inst_dump_py.argtypes = [
        ctypes.POINTER(xed_decoded_inst_t), ctypes.c_char_p, ctypes.c_int, ctypes.c_uint64
    ]
    lib.xed_decoded_inst_dump_py.restype = ctypes.c_void_p

def load_xed_library(lib_path: Path) -> ctypes.CDLL:
    """Load the XED shared library and set up ctypes function signatures.
    
    Args:
        lib_path (Path): Path to the XED shared object library.
    
    Returns:
        ctypes.CDLL: The loaded shared library.
    
    Raises:
        FileNotFoundError: If the shared library is not found at the given path.
    """
    if not lib_path.exists():
        raise FileNotFoundError(f"XED library not found at {lib_path}")
    
    # Load the shared library
    lib = ctypes.CDLL(str(lib_path.resolve()))
    
    # Set up function signatures for the library
    setup_ctypes_functions(lib)
    
    return lib

def initialize_xed(lib: ctypes.CDLL, xedd: xed_decoded_inst_t) -> None:
    """Initialize XED and set up the decoding environment.
    
    Args:
        lib (ctypes.CDLL): The loaded XED library.
        xedd (xed_decoded_inst_t): The decoded instruction object to be initialized.
    """
    lib.xed_tables_init_py()

    # Set the decoding mode to 64-bit
    dstate = xed_state_t()
    lib.xed_state_init2_py(ctypes.byref(dstate), XED_MACHINE_MODE_LONG_64, XED_ADDRESS_WIDTH_64b)
    lib.xed_operand_values_set_mode_py(ctypes.byref(xedd), ctypes.byref(dstate))

def decode_instruction(lib: ctypes.CDLL, xedd: xed_decoded_inst_t, instruction_bytes: bytes) -> None:
    """Decode a given instruction byte sequence using XED.
    
    Args:
        lib (ctypes.CDLL): The loaded XED library.
        xedd (xed_decoded_inst_t): The decoded instruction object.
        instruction_bytes (bytes): The raw instruction bytes to be decoded.
    
    Raises:
        RuntimeError: If decoding fails.
    """
    print(f'[DECODED BYTES] {instruction_bytes.hex()}')

    int_array = (ctypes.c_uint8 * len(instruction_bytes))(*instruction_bytes)
    err = lib.xed_decode_py(ctypes.byref(xedd), int_array, len(instruction_bytes))

    if err != 0:
        error_string = lib.xed_error_enum_t2str_py(err)
        raise ValueError(f"XED ERROR: {error_string.decode('utf-8')}")

def print_decoded_instruction_info(lib: ctypes.CDLL, xedd: xed_decoded_inst_t, buffer_size: int = 700) -> None:
    """Print detailed information about the decoded instruction.
    
    Args:
        lib (ctypes.CDLL): The loaded XED library.
        xedd (xed_decoded_inst_t): The decoded instruction object.
        buffer_size (int): The size of the buffer used for dumping the instruction format. Defaults to 700.
    """
    PAD = 12
    # Get and print the instruction ICLASS
    iclass: xed_enum = lib.xed_decoded_inst_get_iclass_py(ctypes.byref(xedd))
    iclass_str: str = lib.xed_iclass_enum_t2str_py(iclass)
    print(f'{"ICLASS":<{PAD}}: {iclass_str.decode("utf-8")}')

    # Get and print the instruction category
    category: xed_enum = lib.xed_decoded_inst_get_category_py(ctypes.byref(xedd))
    category_str: str = lib.xed_category_enum_t2str_py(category)
    print(f'{"CATEGORY":<{PAD}}: {category_str.decode("utf-8")}')

    # Get and print the instruction extension
    extension: xed_enum = lib.xed_decoded_inst_get_extension_py(ctypes.byref(xedd))
    extension_str: str = lib.xed_category_enum_t2str_py(extension)
    print(f'{"EXTENSION":<{PAD}}: {extension_str.decode("utf-8")}')

    # Get and print the instruction IFORM
    iform: xed_enum = lib.xed_decoded_inst_get_iform_enum_py(ctypes.byref(xedd))
    iform_str: str = lib.xed_iform_enum_t2str_py(iform)
    print(f'{"IFORM":<{PAD}}: {iform_str.decode("utf-8")}')

    # Get and print the instruction ISA set
    isa_set: xed_enum = lib.xed_decoded_inst_get_isa_set_py(ctypes.byref(xedd))
    isa_set_str: str = lib.xed_isa_set_enum_t2str_py(isa_set)
    print(f'{"ISA SET":<{PAD}}: {isa_set_str.decode("utf-8")}')

    # Check if it's an APX instruction
    apx: bool = lib.xed_classify_apx_py(ctypes.byref(xedd))
    if apx:
        print("[APX instruction]")

    # Dump the decoded instruction format
    buffer = ctypes.create_string_buffer(buffer_size)
    lib.xed_decoded_inst_dump_py(ctypes.byref(xedd), buffer, buffer_size, 0)

    # Print the formatted output
    print('\n==== xed_decoded_inst_dump_py() output: ====')
    print(buffer.value.decode())

def main() -> None:
    """Main function to handle library loading, instruction decoding, and output."""
    env = parse_arguments()

    try:
        # Load the XED library and initialize ctypes XED library definitions
        lib = load_xed_library(env.lib_path)

        # Initialize the XED decoder
        xedd = xed_decoded_inst_t()
        initialize_xed(lib, xedd)

        # Define the raw instruction bytes to decode
        instruction_bytes = bytes([0x66, 0xd5, 0x80, 0x12, 0x4C, 0x83, 0x00])

        # Decode the instruction
        decode_instruction(lib, xedd, instruction_bytes)

        # Print detailed information about the decoded instruction
        print_decoded_instruction_info(lib, xedd)

    except Exception as e:
        print(f"Error: {e}")
        exit(1)

if __name__ == '__main__':
    main()
