# SPDX-License-Identifier: MIT

include(CMakeDependentOption)

# Switch off all algs except for those passed in the alglist
function(filter_algs alglist)
  # Set every OQS_ENABLE_* variable =OFF unless one of the following conditions holds:
  #  1. the switch for one of the requested minimal build algorithm's family, e.g OQS_ENABLE_KEM_KYBER
  #  2. the switch for one of the requested algorithms, e.g. OQS_ENABLE_KEM_kyber_768.
  #  3. the switch for platform-specific ("_aesni" or "_avx2") implementation of
  #     one of the requested algorithms, e.g. OQS_ENABLE_KEM_kyber_768_avx2.

  get_cmake_property(_vars VARIABLES)
  foreach (_var ${_vars})
      if(_var MATCHES "^OQS_ENABLE_..._" AND NOT _var MATCHES "_AVAILABLE$" AND ${_var})
          set(${_var} OFF PARENT_SCOPE)
          # Case 1, family name
	  foreach (_alg ${ARGV0})
             string(TOUPPER ${_alg} upalg)
	     if("OQS_ENABLE_${upalg}" MATCHES "^${_var}")
                 set(${_var} ON PARENT_SCOPE)
             endif()
          endforeach()
          # Case 2, exact match
	  foreach (_alg ${ARGV0})
	     if(${_var}X STREQUAL "OQS_ENABLE_${_alg}X")
                 set(${_var} ON PARENT_SCOPE)
             endif()
          endforeach()
          # Case 3, platform specific
          string(REPLACE "_aesni" "" _var_base ${_var})
          string(REPLACE "_avx2" "" _var_base ${_var_base})
          string(REPLACE "_avx" "" _var_base ${_var_base})
          string(REPLACE "_x86_64" "" _var_base ${_var_base})
          string(REPLACE "_aarch64" "" _var_base ${_var_base})
	  foreach (_alg ${ARGV0})
            if(${_var}_AVAILABLE)
              if(${_var_base}X STREQUAL ${_alg}X)
                  set(${_var} ON PARENT_SCOPE)
              endif()
            endif()
          endforeach()
      endif()
  endforeach()
  message(STATUS "Algorithms filtered for ${ARGV0}")
endfunction()

if(DEFINED OQS_KEM_DEFAULT)
    message(WARNING "OQS_KEM_DEFAULT not longer supported")
endif()
if(DEFINED OQS_SIG_DEFAULT)
    message(WARNING "OQS_SIG_DEFAULT not longer supported")
endif()

# Only enable OpenSSL by default on not-Windows
if(NOT WIN32)
    option(OQS_USE_OPENSSL "Enable OpenSSL usage" ON)
else()
    option(OQS_USE_OPENSSL "Enable OpenSSL usage" OFF)
endif()

# Use OpenSSL's AES only if no AESNI and x86 dist build is used.
# Reason: AESNI implementation better fits our incremental API.
cmake_dependent_option(OQS_USE_AES_OPENSSL "" ON "OQS_USE_OPENSSL; NOT OQS_DIST_X86_64_BUILD; NOT OQS_USE_AES_INSTRUCTIONS" OFF)
cmake_dependent_option(OQS_USE_SHA2_OPENSSL "" ON "OQS_USE_OPENSSL" OFF)
# Disable OpenSSL's SHA3 by default. The implementation is not complete
# enough to support our incremental API.
cmake_dependent_option(OQS_USE_SHA3_OPENSSL "" OFF "OQS_USE_OPENSSL" OFF)

# sanity check: Disable OpenSSL if not a single OpenSSL component define is on
cmake_dependent_option(OQS_USE_OPENSSL "" ON "OQS_USE_AES_OPENSSL OR OQS_USE_SHA2_OPENSSL OR OQS_USE_SHA3_OPENSSL" OFF)

option(OQS_DLOPEN_OPENSSL "Enable OpenSSL through dlopen" OFF)

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR OQS_USE_AVX2_INSTRUCTIONS)
    cmake_dependent_option(OQS_ENABLE_SHA3_xkcp_low_avx2 "" ON "NOT OQS_USE_SHA3_OPENSSL" OFF)
endif()
endif()

# SHA3 AVX512VL only supported on Linux x86_64
if(CMAKE_SYSTEM_NAME STREQUAL "Linux" AND (OQS_DIST_X86_64_BUILD OR OQS_USE_AVX512_INSTRUCTIONS))
    cmake_dependent_option(OQS_USE_SHA3_AVX512VL "Enable SHA3 AVX512VL usage" ON "NOT OQS_USE_SHA3_OPENSSL" OFF)
else()
    option(OQS_USE_SHA3_AVX512VL "Enable SHA3 AVX512VL usage" OFF)
endif()

# BIKE is not supported on Windows, 32-bit ARM, X86, S390X (big endian) and PPC64 (big endian)
cmake_dependent_option(OQS_ENABLE_KEM_BIKE "Enable BIKE algorithm family" ON "NOT WIN32; NOT ARCH_ARM32v7; NOT ARCH_X86; NOT ARCH_S390X; NOT ARCH_PPC64" OFF)
# BIKE doesn't work on any 32-bit platform
if(CMAKE_SIZEOF_VOID_P MATCHES "4")
set(OQS_ENABLE_KEM_BIKE OFF)
endif()
cmake_dependent_option(OQS_ENABLE_KEM_bike_l1 "" ON "OQS_ENABLE_KEM_BIKE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_bike_l3 "" ON "OQS_ENABLE_KEM_BIKE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_bike_l5 "" ON "OQS_ENABLE_KEM_BIKE" OFF)

option(OQS_ENABLE_KEM_FRODOKEM "Enable Frodo algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_640_aes "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_640_shake "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_976_aes "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_976_shake "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_1344_aes "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_frodokem_1344_shake "" ON "OQS_ENABLE_KEM_FRODOKEM" OFF)

option(OQS_ENABLE_KEM_NTRUPRIME "Enable ntruprime algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_KEM_ntruprime_sntrup761 "" ON "OQS_ENABLE_KEM_NTRUPRIME" OFF)
if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_ntruprime_sntrup761_avx2 "" ON "OQS_ENABLE_KEM_ntruprime_sntrup761" OFF)
endif()
endif()

##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_ADD_ENABLE_BY_ALG_START
option(OQS_ENABLE_KEM_CLASSIC_MCELIECE "Enable classic_mceliece algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_348864 "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_348864f "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_460896 "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_460896f "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6688128 "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6688128f "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6960119 "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6960119f "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_8192128 "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_8192128f "" ON "OQS_ENABLE_KEM_CLASSIC_MCELIECE" OFF)

option(OQS_ENABLE_KEM_HQC "Enable hqc algorithm family" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_hqc_128 "" ON "OQS_ENABLE_KEM_HQC" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_hqc_192 "" ON "OQS_ENABLE_KEM_HQC" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_hqc_256 "" ON "OQS_ENABLE_KEM_HQC" OFF)

option(OQS_ENABLE_KEM_KYBER "Enable kyber algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_KEM_kyber_512 "" ON "OQS_ENABLE_KEM_KYBER" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_kyber_768 "" ON "OQS_ENABLE_KEM_KYBER" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_kyber_1024 "" ON "OQS_ENABLE_KEM_KYBER" OFF)

option(OQS_ENABLE_KEM_ML_KEM "Enable ml_kem algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_512 "" ON "OQS_ENABLE_KEM_ML_KEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_768 "" ON "OQS_ENABLE_KEM_ML_KEM" OFF)
cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_1024 "" ON "OQS_ENABLE_KEM_ML_KEM" OFF)

option(OQS_ENABLE_SIG_DILITHIUM "Enable dilithium algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_dilithium_2 "" ON "OQS_ENABLE_SIG_DILITHIUM" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_dilithium_3 "" ON "OQS_ENABLE_SIG_DILITHIUM" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_dilithium_5 "" ON "OQS_ENABLE_SIG_DILITHIUM" OFF)

option(OQS_ENABLE_SIG_ML_DSA "Enable ml_dsa algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_44 "" ON "OQS_ENABLE_SIG_ML_DSA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_65 "" ON "OQS_ENABLE_SIG_ML_DSA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_87 "" ON "OQS_ENABLE_SIG_ML_DSA" OFF)

option(OQS_ENABLE_SIG_FALCON "Enable falcon algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_falcon_512 "" ON "OQS_ENABLE_SIG_FALCON" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_falcon_1024 "" ON "OQS_ENABLE_SIG_FALCON" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_512 "" ON "OQS_ENABLE_SIG_FALCON" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_1024 "" ON "OQS_ENABLE_SIG_FALCON" OFF)

option(OQS_ENABLE_SIG_SPHINCS "Enable sphincs algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_128f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_128s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_192f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_192s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_256f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_256s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_128f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_128s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_192f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_192s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_256f_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_256s_simple "" ON "OQS_ENABLE_SIG_SPHINCS" OFF)

option(OQS_ENABLE_SIG_MAYO "Enable mayo algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_mayo_1 "" ON "OQS_ENABLE_SIG_MAYO" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_mayo_2 "" ON "OQS_ENABLE_SIG_MAYO" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_mayo_3 "" ON "OQS_ENABLE_SIG_MAYO" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_mayo_5 "" ON "OQS_ENABLE_SIG_MAYO" OFF)

option(OQS_ENABLE_SIG_CROSS "Enable cross algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_balanced "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_fast "" ON "OQS_ENABLE_SIG_CROSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_small "" ON "OQS_ENABLE_SIG_CROSS" OFF)

option(OQS_ENABLE_SIG_UOV "Enable uov algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc_skc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc_skc "" ON "OQS_ENABLE_SIG_UOV" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc_skc "" ON "OQS_ENABLE_SIG_UOV" OFF)

option(OQS_ENABLE_SIG_SNOVA "Enable snova algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_esk "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_17_2 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_25_8_3 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_56_25_2 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_49_11_3 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_8_4 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_5 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_60_10_4 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_29_6_5 "" ON "OQS_ENABLE_SIG_SNOVA" OFF)
##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_ADD_ENABLE_BY_ALG_END

##### OQS_COPY_FROM_LIBJADE_FRAGMENT_ADD_ENABLE_BY_ALG_START
if ((OQS_LIBJADE_BUILD STREQUAL "ON"))

option(OQS_ENABLE_LIBJADE_KEM_KYBER "Enable libjade implementation of kyber algorithm family" ON)
cmake_dependent_option(OQS_ENABLE_LIBJADE_KEM_kyber_512 "" ON "OQS_ENABLE_LIBJADE_KEM_KYBER" OFF)
cmake_dependent_option(OQS_ENABLE_LIBJADE_KEM_kyber_768 "" ON "OQS_ENABLE_LIBJADE_KEM_KYBER" OFF)
endif()
##### OQS_COPY_FROM_LIBJADE_FRAGMENT_ADD_ENABLE_BY_ALG_END

if((OQS_MINIMAL_BUILD STREQUAL "ON"))
   message(FATAL_ERROR "OQS_MINIMAL_BUILD option ${OQS_MINIMAL_BUILD} no longer supported")
endif()

if(NOT DEFINED OQS_ALGS_ENABLED OR OQS_ALGS_ENABLED STREQUAL "")
	set(OQS_ALGS_ENABLED "All")
endif()

if(NOT ((OQS_MINIMAL_BUILD STREQUAL "") OR (OQS_MINIMAL_BUILD STREQUAL "OFF")))
	filter_algs("${OQS_MINIMAL_BUILD}")
elseif (${OQS_ALGS_ENABLED} STREQUAL "STD")
##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_LIST_STANDARDIZED_ALGS_START
	filter_algs("KEM_ml_kem_512;KEM_ml_kem_768;KEM_ml_kem_1024;SIG_ml_dsa_44;SIG_ml_dsa_65;SIG_ml_dsa_87;SIG_falcon_512;SIG_falcon_1024;SIG_falcon_padded_512;SIG_falcon_padded_1024;SIG_sphincs_sha2_128f_simple;SIG_sphincs_sha2_128s_simple;SIG_sphincs_sha2_192f_simple;SIG_sphincs_sha2_192s_simple;SIG_sphincs_sha2_256f_simple;SIG_sphincs_sha2_256s_simple;SIG_sphincs_shake_128f_simple;SIG_sphincs_shake_128s_simple;SIG_sphincs_shake_192f_simple;SIG_sphincs_shake_192s_simple;SIG_sphincs_shake_256f_simple;SIG_sphincs_shake_256s_simple")
##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_LIST_STANDARDIZED_ALGS_END
elseif(${OQS_ALGS_ENABLED} STREQUAL "NIST_R4")
	filter_algs("KEM_classic_mceliece_348864;KEM_classic_mceliece_348864f;KEM_classic_mceliece_460896;KEM_classic_mceliece_460896f;KEM_classic_mceliece_6688128;KEM_classic_mceliece_6688128f;KEM_classic_mceliece_6960119;KEM_classic_mceliece_6960119f;KEM_classic_mceliece_8192128;KEM_classic_mceliece_8192128f;KEM_hqc_128;KEM_hqc_192;KEM_hqc_256;KEM_bike_l1;KEM_bike_l3;KEM_bike_l5")
elseif(${OQS_ALGS_ENABLED} STREQUAL "NIST_SIG_ONRAMP")
	filter_algs("SIG_mayo_1;SIG_mayo_2;SIG_mayo_3;SIG_mayo_5;SIG_cross_rsdp_128_balanced;SIG_cross_rsdp_128_fast;SIG_cross_rsdp_128_small;SIG_cross_rsdp_192_balanced;SIG_cross_rsdp_192_fast;SIG_cross_rsdp_192_small;SIG_cross_rsdp_256_balanced;SIG_cross_rsdp_256_fast;SIG_cross_rsdp_256_small;SIG_cross_rsdpg_128_balanced;SIG_cross_rsdpg_128_fast;SIG_cross_rsdpg_128_small;SIG_cross_rsdpg_192_balanced;SIG_cross_rsdpg_192_fast;SIG_cross_rsdpg_192_small;SIG_cross_rsdpg_256_balanced;SIG_cross_rsdpg_256_fast;SIG_cross_rsdpg_256_small;SIG_uov_ov_Ip;SIG_uov_ov_Is;SIG_uov_ov_III;SIG_uov_ov_V;SIG_uov_ov_Ip_pkc;SIG_uov_ov_Is_pkc;SIG_uov_ov_III_pkc;SIG_uov_ov_V_pkc;SIG_uov_ov_Ip_pkc_skc;SIG_uov_ov_Is_pkc_skc;SIG_uov_ov_III_pkc_skc;SIG_uov_ov_V_pkc_skc;SNOVA_24_5_4;SNOVA_24_5_4_SHAKE;SNOVA_24_5_4_esk;SNOVA_24_5_4_SHAKE_esk;SNOVA_37_17_2;SNOVA_25_8_3;SNOVA_56_25_2;SNOVA_49_11_3;SNOVA_37_8_4;SNOVA_24_5_5;SNOVA_60_10_4;SNOVA_29_6_5")
else()
	message(STATUS "Alg enablement unchanged")
endif()


##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_ADD_ENABLE_BY_ALG_CONDITIONAL_START
if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_348864_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_348864" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS AND OQS_USE_BMI1_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_348864f_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_348864f" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_460896_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_460896" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS AND OQS_USE_BMI1_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_460896f_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_460896f" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6688128_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_6688128" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS AND OQS_USE_BMI1_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6688128f_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_6688128f" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6960119_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_6960119" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS AND OQS_USE_BMI1_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_6960119f_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_6960119f" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_8192128_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_8192128" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS AND OQS_USE_BMI1_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_classic_mceliece_8192128f_avx2 "" ON "OQS_ENABLE_KEM_classic_mceliece_8192128f" OFF)
endif()
endif()



if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_512_avx2 "" ON "OQS_ENABLE_KEM_kyber_512" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))
if(((CMAKE_C_COMPILER_ID STREQUAL "GNU") AND (CMAKE_C_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((CMAKE_CXX_COMPILER_ID STREQUAL "GNU") AND (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((NOT (CMAKE_C_COMPILER_ID STREQUAL "GNU")) AND (NOT (CMAKE_CXX_COMPILER_ID STREQUAL "GNU"))))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_512_aarch64 "" ON "OQS_ENABLE_KEM_kyber_512" OFF)
else()
    message(WARNING "  ARM optimizations are not fully supported on this compiler version.")
endif()
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_768_avx2 "" ON "OQS_ENABLE_KEM_kyber_768" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))
if(((CMAKE_C_COMPILER_ID STREQUAL "GNU") AND (CMAKE_C_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((CMAKE_CXX_COMPILER_ID STREQUAL "GNU") AND (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((NOT (CMAKE_C_COMPILER_ID STREQUAL "GNU")) AND (NOT (CMAKE_CXX_COMPILER_ID STREQUAL "GNU"))))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_768_aarch64 "" ON "OQS_ENABLE_KEM_kyber_768" OFF)
else()
    message(WARNING "  ARM optimizations are not fully supported on this compiler version.")
endif()
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_1024_avx2 "" ON "OQS_ENABLE_KEM_kyber_1024" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))
if(((CMAKE_C_COMPILER_ID STREQUAL "GNU") AND (CMAKE_C_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((CMAKE_CXX_COMPILER_ID STREQUAL "GNU") AND (CMAKE_CXX_COMPILER_VERSION VERSION_GREATER_EQUAL "9.4.0")) OR ((NOT (CMAKE_C_COMPILER_ID STREQUAL "GNU")) AND (NOT (CMAKE_CXX_COMPILER_ID STREQUAL "GNU"))))
    cmake_dependent_option(OQS_ENABLE_KEM_kyber_1024_aarch64 "" ON "OQS_ENABLE_KEM_kyber_1024" OFF)
else()
    message(WARNING "  ARM optimizations are not fully supported on this compiler version.")
endif()
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_512_x86_64 "" ON "OQS_ENABLE_KEM_ml_kem_512" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))

    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_512_aarch64 "" ON "OQS_ENABLE_KEM_ml_kem_512" OFF)

endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_USE_CUPQC)
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_512_cuda "" ON "OQS_ENABLE_KEM_ml_kem_512" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_768_x86_64 "" ON "OQS_ENABLE_KEM_ml_kem_768" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))

    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_768_aarch64 "" ON "OQS_ENABLE_KEM_ml_kem_768" OFF)

endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_USE_CUPQC)
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_768_cuda "" ON "OQS_ENABLE_KEM_ml_kem_768" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_1024_x86_64 "" ON "OQS_ENABLE_KEM_ml_kem_1024" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if((OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS)))

    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_1024_aarch64 "" ON "OQS_ENABLE_KEM_ml_kem_1024" OFF)

endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_USE_CUPQC)
    cmake_dependent_option(OQS_ENABLE_KEM_ml_kem_1024_cuda "" ON "OQS_ENABLE_KEM_ml_kem_1024" OFF)
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_2_avx2 "" ON "OQS_ENABLE_SIG_dilithium_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_2_aarch64 "" ON "OQS_ENABLE_SIG_dilithium_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_3_avx2 "" ON "OQS_ENABLE_SIG_dilithium_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_3_aarch64 "" ON "OQS_ENABLE_SIG_dilithium_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_5_avx2 "" ON "OQS_ENABLE_SIG_dilithium_5" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_dilithium_5_aarch64 "" ON "OQS_ENABLE_SIG_dilithium_5" OFF)
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_44_avx2 "" ON "OQS_ENABLE_SIG_ml_dsa_44" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_65_avx2 "" ON "OQS_ENABLE_SIG_ml_dsa_65" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_ml_dsa_87_avx2 "" ON "OQS_ENABLE_SIG_ml_dsa_87" OFF)
endif()
endif()


if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_512_avx2 "" ON "OQS_ENABLE_SIG_falcon_512" OFF)
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_512_aarch64 "" ON "OQS_ENABLE_SIG_falcon_512" OFF)
endif()
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_1024_avx2 "" ON "OQS_ENABLE_SIG_falcon_1024" OFF)
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_1024_aarch64 "" ON "OQS_ENABLE_SIG_falcon_1024" OFF)
endif()
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_512_avx2 "" ON "OQS_ENABLE_SIG_falcon_padded_512" OFF)
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_512_aarch64 "" ON "OQS_ENABLE_SIG_falcon_padded_512" OFF)
endif()
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_1024_avx2 "" ON "OQS_ENABLE_SIG_falcon_padded_1024" OFF)
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_falcon_padded_1024_aarch64 "" ON "OQS_ENABLE_SIG_falcon_padded_1024" OFF)
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_128f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_128f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_128s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_128s_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_192f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_192f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_192s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_192s_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_256f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_256f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_sha2_256s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_sha2_256s_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_128f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_128f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_128s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_128s_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_192f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_192f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_192s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_192s_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_256f_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_256f_simple" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_sphincs_shake_256s_simple_avx2 "" ON "OQS_ENABLE_SIG_sphincs_shake_256s_simple" OFF)
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_1_avx2 "" ON "OQS_ENABLE_SIG_mayo_1" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_1_neon "" ON "OQS_ENABLE_SIG_mayo_1" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_2_avx2 "" ON "OQS_ENABLE_SIG_mayo_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_2_neon "" ON "OQS_ENABLE_SIG_mayo_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_3_avx2 "" ON "OQS_ENABLE_SIG_mayo_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_3_neon "" ON "OQS_ENABLE_SIG_mayo_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_5_avx2 "" ON "OQS_ENABLE_SIG_mayo_5" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_mayo_5_neon "" ON "OQS_ENABLE_SIG_mayo_5" OFF)
endif()
endif()


if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_128_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_128_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_128_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_128_small" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_192_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_192_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_192_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_192_small" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_256_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_256_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdp_256_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdp_256_small" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_128_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_128_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_128_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_128_small" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_192_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_192_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_192_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_192_small" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_balanced_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_256_balanced" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_fast_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_256_fast" OFF)
endif()

if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_cross_rsdpg_256_small_avx2 "" ON "OQS_ENABLE_SIG_cross_rsdpg_256_small" OFF)
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_neon "" ON "OQS_ENABLE_SIG_uov_ov_Is" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Is" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_neon "" ON "OQS_ENABLE_SIG_uov_ov_Ip" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Ip" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_neon "" ON "OQS_ENABLE_SIG_uov_ov_III" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_III" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_neon "" ON "OQS_ENABLE_SIG_uov_ov_V" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_V" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc_neon "" ON "OQS_ENABLE_SIG_uov_ov_Is_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Is_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc_neon "" ON "OQS_ENABLE_SIG_uov_ov_Ip_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Ip_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc_neon "" ON "OQS_ENABLE_SIG_uov_ov_III_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_III_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc_neon "" ON "OQS_ENABLE_SIG_uov_ov_V_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_V_pkc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc_skc_neon "" ON "OQS_ENABLE_SIG_uov_ov_Is_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Is_pkc_skc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Is_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc_neon "" ON "OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_Ip_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc_skc_neon "" ON "OQS_ENABLE_SIG_uov_ov_III_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_III_pkc_skc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_III_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc_skc_neon "" ON "OQS_ENABLE_SIG_uov_ov_V_pkc_skc" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_uov_ov_V_pkc_skc_avx2 "" ON "OQS_ENABLE_SIG_uov_ov_V_pkc_skc" OFF)
endif()
endif()


if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_esk" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_esk_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_esk" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_esk_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_4_SHAKE_esk" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_17_2_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_37_17_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_17_2_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_37_17_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_25_8_3_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_25_8_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_25_8_3_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_25_8_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_56_25_2_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_56_25_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_56_25_2_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_56_25_2" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_49_11_3_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_49_11_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_49_11_3_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_49_11_3" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_8_4_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_37_8_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_37_8_4_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_37_8_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_5_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_5" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_24_5_5_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_24_5_5" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_60_10_4_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_60_10_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_60_10_4_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_60_10_4" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_29_6_5_avx2 "" ON "OQS_ENABLE_SIG_snova_SNOVA_29_6_5" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Darwin|Linux")
if(OQS_DIST_ARM64_V8_BUILD OR (OQS_USE_ARM_NEON_INSTRUCTIONS AND OQS_USE_ARM_NEON_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_SIG_snova_SNOVA_29_6_5_neon "" ON "OQS_ENABLE_SIG_snova_SNOVA_29_6_5" OFF)
endif()
endif()

##### OQS_COPY_FROM_UPSTREAM_FRAGMENT_ADD_ENABLE_BY_ALG_CONDITIONAL_END

##### OQS_COPY_FROM_LIBJADE_FRAGMENT_ADD_ENABLE_BY_ALG_CONDITIONAL_START
if ((OQS_LIBJADE_BUILD STREQUAL "ON"))

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_LIBJADE_KEM_kyber_512_avx2 "" ON "OQS_ENABLE_LIBJADE_KEM_kyber_512" OFF)
endif()
endif()

if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
if(OQS_DIST_X86_64_BUILD OR (OQS_USE_AVX2_INSTRUCTIONS AND OQS_USE_BMI2_INSTRUCTIONS AND OQS_USE_POPCNT_INSTRUCTIONS))
    cmake_dependent_option(OQS_ENABLE_LIBJADE_KEM_kyber_768_avx2 "" ON "OQS_ENABLE_LIBJADE_KEM_kyber_768" OFF)
endif()
endif()

endif()
##### OQS_COPY_FROM_LIBJADE_FRAGMENT_ADD_ENABLE_BY_ALG_CONDITIONAL_END

option(OQS_ENABLE_SIG_STFL_XMSS "Enable XMSS algorithm family" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h10 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h16 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h20 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake128_h10 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake128_h16 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake128_h20 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha512_h10 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha512_h16 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha512_h20 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h10 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h16 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h20 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h10_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h16_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_sha256_h20_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h10_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h16_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h20_192 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h10_256 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h16_256 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmss_shake256_h20_256 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)

cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h20_2 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h20_4 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h40_2 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h40_4 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h40_8 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h60_3 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h60_6 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_sha256_h60_12 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h20_2 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h20_4 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h40_2 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h40_4 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h40_8 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h60_3 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h60_6 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_xmssmt_shake128_h60_12 "" ON "OQS_ENABLE_SIG_STFL_XMSS" OFF)


option(OQS_ENABLE_SIG_STFL_LMS "Enable LMS algorithm family" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h5_w1 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h5_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h5_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w1 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w1 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w1 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h25_w1 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h25_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h25_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h25_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h5_w8_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w4_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w8_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w2_h10_w2 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w4_h10_w4 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h10_w8_h10_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w8_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w8_h10_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h15_w8_h15_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w8_h5_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w8_h10_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w8_h15_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)
cmake_dependent_option(OQS_ENABLE_SIG_STFL_lms_sha256_h20_w8_h20_w8 "" ON "OQS_ENABLE_SIG_STFL_LMS" OFF)

option(OQS_HAZARDOUS_EXPERIMENTAL_ENABLE_SIG_STFL_KEY_SIG_GEN "Enable stateful key and signature generation for research and experimentation" OFF)
cmake_dependent_option(OQS_ALLOW_STFL_KEY_AND_SIG_GEN "" ON "OQS_HAZARDOUS_EXPERIMENTAL_ENABLE_SIG_STFL_KEY_SIG_GEN" OFF)

if (${OQS_HAZARDOUS_EXPERIMENTAL_ENABLE_SIG_STFL_KEY_SIG_GEN} AND ${OQS_ENABLE_SIG_STFL_XMSS})
    set(OQS_ALLOW_XMSS_KEY_AND_SIG_GEN ON)
else()
    set(OQS_ALLOW_XMSS_KEY_AND_SIG_GEN OFF)
endif()

if (${OQS_HAZARDOUS_EXPERIMENTAL_ENABLE_SIG_STFL_KEY_SIG_GEN} AND ${OQS_ENABLE_SIG_STFL_LMS})
    set(OQS_ALLOW_LMS_KEY_AND_SIG_GEN ON)
else()
    set(OQS_ALLOW_LMS_KEY_AND_SIG_GEN OFF)
endif()

if(OQS_ALLOW_STFL_KEY_AND_SIG_GEN  STREQUAL "ON")
    message(STATUS "Experimental stateful key and signature generation is enabled. Ensure secret keys are securely stored to prevent multiple simultaneous sign operations.")
endif()

# Set XKCP (Keccak) required for Sphincs and SNOVA AVX2 code even if OpenSSL3 SHA3 is used:
if (${OQS_ENABLE_SIG_SPHINCS} OR ${OQS_ENABLE_SIG_SNOVA} OR NOT ${OQS_USE_SHA3_OPENSSL})
    set(OQS_ENABLE_SHA3_xkcp_low ON)
else()
    set(OQS_ENABLE_SHA3_xkcp_low OFF)
endif()
if(CMAKE_SYSTEM_NAME MATCHES "Linux|Darwin")
    if(OQS_DIST_X86_64_BUILD OR OQS_USE_AVX2_INSTRUCTIONS)
        set(OQS_ENABLE_SHA3_xkcp_low_avx2 ON)
    else()
        set(OQS_ENABLE_SHA3_xkcp_low_avx2 OFF)
    endif()
endif()
