/*
 * Copyright © 2024 Igalia S.L.
 * SPDX-License-Identifier: MIT
 */

#include "util/u_math.h"
#include "adreno_common.xml.h"
#include "adreno_pm4.xml.h"
#include "a6xx.xml.h"

#include "freedreno_perfcntr.h"

#define REG6(_x) REG_A6XX_##_x
#define REG7(_x) REG_A7XX_##_x

#define FD7_COUNTER(_sel, _offset) COUNTER_BASE(_sel, _offset+0, _offset+1)
#define FD7_COUNTABLE(_sel) COUNTABLE_BASE(#_sel, A7XX_##_sel, UINT64, AVERAGE)

static const struct fd_perfcntr_counter cp_counters[] = {
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(0)), REG7(RBBM_PERFCTR_CP(0))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(1)), REG7(RBBM_PERFCTR_CP(1))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(2)), REG7(RBBM_PERFCTR_CP(2))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(3)), REG7(RBBM_PERFCTR_CP(3))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(4)), REG7(RBBM_PERFCTR_CP(4))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(5)), REG7(RBBM_PERFCTR_CP(5))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(6)), REG7(RBBM_PERFCTR_CP(6))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(7)), REG7(RBBM_PERFCTR_CP(7))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(8)), REG7(RBBM_PERFCTR_CP(8))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(9)), REG7(RBBM_PERFCTR_CP(9))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(10)), REG7(RBBM_PERFCTR_CP(10))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(11)), REG7(RBBM_PERFCTR_CP(11))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(12)), REG7(RBBM_PERFCTR_CP(12))),
      FD7_COUNTER(REG6(CP_PERFCTR_CP_SEL(13)), REG7(RBBM_PERFCTR_CP(13))),
};

static const struct fd_perfcntr_countable cp_countables[] = {
      FD7_COUNTABLE(PERF_CP_NEVER_COUNT),
      FD7_COUNTABLE(PERF_CP_ALWAYS_COUNT),
      FD7_COUNTABLE(PERF_CP_BUSY_GFX_CORE_IDLE),
      FD7_COUNTABLE(PERF_CP_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_CP_NUM_PREEMPTIONS),
      FD7_COUNTABLE(PERF_CP_PREEMPTION_REACTION_DELAY),
      FD7_COUNTABLE(PERF_CP_PREEMPTION_SWITCH_OUT_TIME),
      FD7_COUNTABLE(PERF_CP_PREEMPTION_SWITCH_IN_TIME),
      FD7_COUNTABLE(PERF_CP_DEAD_DRAWS_IN_BIN_RENDER),
      FD7_COUNTABLE(PERF_CP_PREDICATED_DRAWS_KILLED),
      FD7_COUNTABLE(PERF_CP_MODE_SWITCH),
      FD7_COUNTABLE(PERF_CP_ZPASS_DONE),
      FD7_COUNTABLE(PERF_CP_CONTEXT_DONE),
      FD7_COUNTABLE(PERF_CP_CACHE_FLUSH),
      FD7_COUNTABLE(PERF_CP_LONG_PREEMPTIONS),
      FD7_COUNTABLE(PERF_CP_SQE_I_CACHE_STARVE),
      FD7_COUNTABLE(PERF_CP_SQE_IDLE),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_RB),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_IB1),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_IB2),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_IB3),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_FSDT),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_STARVE_SDS),
      FD7_COUNTABLE(PERF_CP_SQE_MRB_STARVE),
      FD7_COUNTABLE(PERF_CP_SQE_RRB_STARVE),
      FD7_COUNTABLE(PERF_CP_SQE_VSD_STARVE),
      FD7_COUNTABLE(PERF_CP_VSD_DECODE_STARVE),
      FD7_COUNTABLE(PERF_CP_SQE_PIPE_OUT_STALL),
      FD7_COUNTABLE(PERF_CP_SQE_SYNC_STALL),
      FD7_COUNTABLE(PERF_CP_SQE_PM4_WFI_STALL),
      FD7_COUNTABLE(PERF_CP_SQE_SYS_WFI_STALL),
      FD7_COUNTABLE(PERF_CP_WAIT_ON_OTHER_PIPE),
      FD7_COUNTABLE(PERF_CP_OUTPUT_BLOCKED),
      FD7_COUNTABLE(PERF_CP_SQE_T4_EXEC),
      FD7_COUNTABLE(PERF_CP_SQE_LOAD_STATE_EXEC),
      FD7_COUNTABLE(PERF_CP_SQE_SAVE_SDS_STATE),
      FD7_COUNTABLE(PERF_CP_SQE_DRAW_EXEC),
      FD7_COUNTABLE(PERF_CP_SQE_CTXT_REG_BUNCH_EXEC),
      FD7_COUNTABLE(PERF_CP_SQE_EXEC_PROFILED),
      FD7_COUNTABLE(PERF_CP_MEMORY_POOL_EMPTY),
      FD7_COUNTABLE(PERF_CP_MEMORY_POOL_SYNC_STALL),
      FD7_COUNTABLE(PERF_CP_MEMORY_POOL_ABOVE_THRESH),
      FD7_COUNTABLE(PERF_CP_MEMORY_POOL_BELOW_THRESH),
      FD7_COUNTABLE(PERF_CP_AHB_WR_STALL_PRE_DRAWS),
      FD7_COUNTABLE(PERF_CP_AHB_STALL_SQE_GMU),
      FD7_COUNTABLE(PERF_CP_AHB_STALL_SQE_WR_OTHER),
      FD7_COUNTABLE(PERF_CP_AHB_STALL_SQE_RD_OTHER),
      FD7_COUNTABLE(PERF_CP_CLUSTER_FE_U_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_FE_S_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_SP_VS_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_US_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_VS_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_GRAS_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_SP_PS_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_PS_EMPTY),
      FD7_COUNTABLE(PERF_CP_CLUSTER_PS_EMPTY),
      FD7_COUNTABLE(PERF_CP_PM4_DATA),
      FD7_COUNTABLE(PERF_CP_PM4_HEADERS),
      FD7_COUNTABLE(PERF_CP_VBIF_READ_BEATS),
      FD7_COUNTABLE(PERF_CP_VBIF_WRITE_BEATS),
      FD7_COUNTABLE(PERF_CP_SQE_INSTR_COUNTER),
      FD7_COUNTABLE(PERF_CP_CLUSTER_FE_US_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_FE_S_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_SP_VS_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_US_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_VS_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_GRAS_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_SP_PS_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_VPC_PS_FULL),
      FD7_COUNTABLE(PERF_CP_CLUSTER_PS_FULL),
      FD7_COUNTABLE(PERF_CP_ICACHE_MISSES),
      FD7_COUNTABLE(PERF_CP_ICACHE_HITS),
      FD7_COUNTABLE(PERF_CP_ICACHE_STALL),
      FD7_COUNTABLE(PERF_CP_DCACHE_MISSES),
      FD7_COUNTABLE(PERF_CP_DCACHE_HITS),
      FD7_COUNTABLE(PERF_CP_DCACHE_STALLS),
      FD7_COUNTABLE(PERF_CP_AQE_SQE_STALL),
      FD7_COUNTABLE(PERF_CP_SQE_AQE_STARVE),
      FD7_COUNTABLE(PERF_CP_ISR_CYCLES),
      FD7_COUNTABLE(PERF_CP_SQE_MD8_STALL_CYCLES),
      FD7_COUNTABLE(PERF_CP_SQE_MESH_EXEC_CYCLES),
      FD7_COUNTABLE(PERF_CP_AQE_NUM_AS_CHUNKS),
      FD7_COUNTABLE(PERF_CP_AQE_NUM_MS_CHUNKS),
      FD7_COUNTABLE(PERF_CP_S_SKEW_BUFFER_FULL),
      FD7_COUNTABLE(PERF_CP_S_SKEW_BUFFER_ABOVE_THRESH),
};

static const struct fd_perfcntr_counter rbbm_counters[] = {
      FD7_COUNTER(REG6(RBBM_PERFCTR_RBBM_SEL(0)), REG7(RBBM_PERFCTR_RBBM(0))),
      FD7_COUNTER(REG6(RBBM_PERFCTR_RBBM_SEL(1)), REG7(RBBM_PERFCTR_RBBM(1))),
      FD7_COUNTER(REG6(RBBM_PERFCTR_RBBM_SEL(2)), REG7(RBBM_PERFCTR_RBBM(2))),
      FD7_COUNTER(REG6(RBBM_PERFCTR_RBBM_SEL(3)), REG7(RBBM_PERFCTR_RBBM(3))),
};

static const struct fd_perfcntr_countable rbbm_countables[] = {
      FD7_COUNTABLE(PERF_RBBM_NEVER_COUNT),
      FD7_COUNTABLE(PERF_RBBM_US_ALWAYS_COUNT),
      FD7_COUNTABLE(PERF_RBBM_US_ALWAYS_ON),
      FD7_COUNTABLE(PERF_RBBM_US_STATUS_MASKED),
      FD7_COUNTABLE(PERF_RBBM_US_PC_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_COM_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_DCOM_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_VBIF_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_VSC_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_UCHE_BUSY),
      FD7_COUNTABLE(PERF_RBBM_US_HLSQ_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_HLSQ_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_PC_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_TESS_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_TSEFE_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_TSEBE_BUSY),
      FD7_COUNTABLE(PERF_RBBM_S_RAS_BUSY),
};

static const struct fd_perfcntr_counter pc_counters[] = {
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(0)), REG7(RBBM_PERFCTR_PC(0))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(1)), REG7(RBBM_PERFCTR_PC(1))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(2)), REG7(RBBM_PERFCTR_PC(2))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(3)), REG7(RBBM_PERFCTR_PC(3))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(4)), REG7(RBBM_PERFCTR_PC(4))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(5)), REG7(RBBM_PERFCTR_PC(5))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(6)), REG7(RBBM_PERFCTR_PC(6))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(7)), REG7(RBBM_PERFCTR_PC(7))),
};

static const struct fd_perfcntr_countable pc_countables[] = {
      FD7_COUNTABLE(PERF_PC_NEVER_COUNT),
      FD7_COUNTABLE(PERF_PC_US_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_PC_US_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_PC_US_UCHE_OUTSTANDING_TRANS),
      FD7_COUNTABLE(PERF_PC_US_PASS1_TF_STALL_CYCLES),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_FOR_INDEX),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_FOR_TF),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_FOR_VIZ_STREAM),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_DI),
      FD7_COUNTABLE(PERF_PC_US_VIS_STREAMS_LOADED),
      FD7_COUNTABLE(PERF_PC_US_INSTANCES),
      FD7_COUNTABLE(PERF_PC_US_DEAD_PRIM),
      FD7_COUNTABLE(PERF_PC_US_SLICE_LIVE_PRIM),
      FD7_COUNTABLE(PERF_PC_US_3D_DRAWCALLS),
      FD7_COUNTABLE(PERF_PC_US_2D_DRAWCALLS),
      FD7_COUNTABLE(PERF_PC_US_NON_DRAWCALL_GLOBAL_EVENTS),
      FD7_COUNTABLE(PERF_PC_US_MESH_DRAWS),
      FD7_COUNTABLE(PERF_PC_US_MESH_DEAD_DRAWS),
      FD7_COUNTABLE(PERF_PC_US_MESH_MVIS_EN_DRAWS),
      FD7_COUNTABLE(PERF_PC_US_MESH_DEAD_PRIM),
      FD7_COUNTABLE(PERF_PC_US_MESH_LIVE_PRIM),
      FD7_COUNTABLE(PERF_PC_US_MESH_PA_EN_PRIM),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_FOR_MVIS_STREAM),
      FD7_COUNTABLE(PERF_PC_US_STARVE_CYCLES_PREDRAW),
      FD7_COUNTABLE(PERF_PC_US_STALL_CYCLES_COMPUTE_GFX),
      FD7_COUNTABLE(PERF_PC_US_STALL_CYCLES_GFX_COMPUTE),
      FD7_COUNTABLE(PERF_PC_US_PREDRAW_STALLS),
      FD7_COUNTABLE(PERF_PC_US_DP0_INPUT_STALLS),
      FD7_COUNTABLE(PERF_PC_US_DP1_INPUT_STALLS),
      FD7_COUNTABLE(PERF_PC_US_BR_STALLS_BV_WORKLOAD),
      FD7_COUNTABLE(PERF_PC_US_BV_STALLS_BR_WORKLOAD),
      FD7_COUNTABLE(PERF_PC_US_PASSPAIR_STALL),
      FD7_COUNTABLE(PERF_PC_US_STALL_CYCLES_UCHE0),
      FD7_COUNTABLE(PERF_PC_US_STALL_CYCLES_UCHE1),
      FD7_COUNTABLE(PERF_PC_US_UCHE_0_TRANS),
      FD7_COUNTABLE(PERF_PC_US_UCHE_1_TRANS),
      FD7_COUNTABLE(PERF_PC_US_BV_STALLED_BY_ATTR),
      FD7_COUNTABLE(PERF_PC_US_BV_STARVED_BY_RARB),
      FD7_COUNTABLE(PERF_PC_US_VPC_PRIM_COUNT_STALLS_BR),
      FD7_COUNTABLE(PERF_PC_US_VPC_PRIM_COUNT_STALLS_BV),
      FD7_COUNTABLE(PERF_PC_US_BV_STALLED_BY_UCHE_FEEDBACK),
      FD7_COUNTABLE(PERF_PC_US_VSD_RARB_DVIZ_FULL),
      FD7_COUNTABLE(PERF_PC_US_VSD_RARB_PVIZ_FULL),
      FD7_COUNTABLE(PERF_PC_US_VSD_RARB_TVIZ_FULL),
      FD7_COUNTABLE(PERF_PC_US_DP0_RARB_FULL),
      FD7_COUNTABLE(PERF_PC_US_DP1_RARB_FULL),
      FD7_COUNTABLE(PERF_PC_US_DP0_LIVE_PRIM),
      FD7_COUNTABLE(PERF_PC_US_DP1_LIVE_PRIM),
      FD7_COUNTABLE(PERF_PC_US_BV2BR_SWITCH),
      FD7_COUNTABLE(PERF_PC_US_BR2BV_SWITCH),
      FD7_COUNTABLE(PERF_PC_US_STALL_CYCLES_PC_S),
      FD7_COUNTABLE(PERF_PC_S_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_PC_S_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_PC_S_STALL_CYCLES_VFD),
      FD7_COUNTABLE(PERF_PC_S_STALL_CYCLES_VPC_FE),
      FD7_COUNTABLE(PERF_PC_S_STALL_CYCLES_TESS),
      FD7_COUNTABLE(PERF_PC_S_STALL_CYCLES_VFD_ONLY),
      FD7_COUNTABLE(PERF_PC_S_STALL_CYCLES_VPC_ONLY),
      FD7_COUNTABLE(PERF_PC_S_VPC_PRIMITIVES),
      FD7_COUNTABLE(PERF_PC_S_VERTEX_HITS),
      FD7_COUNTABLE(PERF_PC_S_IA_VERTICES),
      FD7_COUNTABLE(PERF_PC_S_IA_PRIMITIVES),
      FD7_COUNTABLE(PERF_PC_S_HS_INVOCATIONS),
      FD7_COUNTABLE(PERF_PC_S_DS_INVOCATIONS),
      FD7_COUNTABLE(PERF_PC_S_VS_INVOCATIONS),
      FD7_COUNTABLE(PERF_PC_S_GS_INVOCATIONS),
      FD7_COUNTABLE(PERF_PC_S_DS_PRIMITIVES),
      FD7_COUNTABLE(PERF_PC_S_TESS_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_PC_S_TESS_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_PC_S_TESS_STALL_CYCLES_PC),
      FD7_COUNTABLE(PERF_PC_S_TESS_STARVE_CYCLES_PC),
      FD7_COUNTABLE(PERF_PC_S_TESS_SETUP_ACTIVE),
      FD7_COUNTABLE(PERF_PC_S_TESS_PID_ACTIVE),
      FD7_COUNTABLE(PERF_PC_S_TESS_PRIM_GEN_ACTIVE),
      FD7_COUNTABLE(PERF_PC_S_TESS_FACTOR_TRANS),
      FD7_COUNTABLE(PERF_PC_S_TESS_PC_UV_TRANS),
      FD7_COUNTABLE(PERF_PC_S_TESS_PC_UV_PATCHES),
      FD7_COUNTABLE(PERF_PC_S_MESH_VS_WAVES),
};

static const struct fd_perfcntr_counter vfd_counters[] = {
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(0)), REG7(RBBM_PERFCTR_VFD(0))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(1)), REG7(RBBM_PERFCTR_VFD(1))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(2)), REG7(RBBM_PERFCTR_VFD(2))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(3)), REG7(RBBM_PERFCTR_VFD(3))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(4)), REG7(RBBM_PERFCTR_VFD(4))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(5)), REG7(RBBM_PERFCTR_VFD(5))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(6)), REG7(RBBM_PERFCTR_VFD(6))),
      FD7_COUNTER(REG6(VFD_PERFCTR_VFD_SEL(7)), REG7(RBBM_PERFCTR_VFD(7))),
};

static const struct fd_perfcntr_countable vfd_countables[] = {
      FD7_COUNTABLE(PERF_VFD_NEVER_COUNT),
      FD7_COUNTABLE(PERF_VFD_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_VPC_ALLOC),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_SP_INFO),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_SP_ATTR),
      FD7_COUNTABLE(PERF_VFD_STARVE_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_VFD_RBUFFER_FULL),
      FD7_COUNTABLE(PERF_VFD_ATTR_INFO_FIFO_FULL),
      FD7_COUNTABLE(PERF_VFD_DECODED_ATTRIBUTE_BYTES),
      FD7_COUNTABLE(PERF_VFD_NUM_ATTRIBUTES),
      FD7_COUNTABLE(PERF_VFD_UPPER_SHADER_FIBERS),
      FD7_COUNTABLE(PERF_VFD_LOWER_SHADER_FIBERS),
      FD7_COUNTABLE(PERF_VFD_MODE_0_FIBERS),
      FD7_COUNTABLE(PERF_VFD_MODE_1_FIBERS),
      FD7_COUNTABLE(PERF_VFD_MODE_2_FIBERS),
      FD7_COUNTABLE(PERF_VFD_MODE_3_FIBERS),
      FD7_COUNTABLE(PERF_VFD_MODE_4_FIBERS),
      FD7_COUNTABLE(PERF_VFD_TOTAL_VERTICES),
      FD7_COUNTABLE(PERF_VFDP_STALL_CYCLES_VFD),
      FD7_COUNTABLE(PERF_VFDP_STALL_CYCLES_VFD_INDEX),
      FD7_COUNTABLE(PERF_VFDP_STALL_CYCLES_VFD_PROG),
      FD7_COUNTABLE(PERF_VFDP_STARVE_CYCLES_PC),
      FD7_COUNTABLE(PERF_VFDP_VS_STAGE_WAVES),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_PRG_END_FE),
      FD7_COUNTABLE(PERF_VFD_STALL_CYCLES_CBSYNC),
};

static const struct fd_perfcntr_counter hlsq_counters[] = {
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(0)), REG7(RBBM_PERFCTR_HLSQ(0))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(1)), REG7(RBBM_PERFCTR_HLSQ(1))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(2)), REG7(RBBM_PERFCTR_HLSQ(2))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(3)), REG7(RBBM_PERFCTR_HLSQ(3))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(4)), REG7(RBBM_PERFCTR_HLSQ(4))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(5)), REG7(RBBM_PERFCTR_HLSQ(5))),
};

static const struct fd_perfcntr_countable hlsq_countables[] = {
      FD7_COUNTABLE(PERF_HLSQ_NEVER_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_STALL_CYCLES_SP_STATE),
      FD7_COUNTABLE(PERF_HLSQ_STALL_CYCLES_SP_FS_STAGE),
      FD7_COUNTABLE(PERF_HLSQ_UCHE_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_UCHE_LATENCY_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_HLSQ_COMPUTE_DRAWCALLS),
      FD7_COUNTABLE(PERF_HLSQ_FS_DATA_WAIT_PROGRAMMING),
      FD7_COUNTABLE(PERF_HLSQ_DUAL_FS_PROG_ACTIVE),
      FD7_COUNTABLE(PERF_HLSQ_DUAL_VS_PROG_ACTIVE),
      FD7_COUNTABLE(PERF_HLSQ_FS_BATCH_COUNT_ZERO),
      FD7_COUNTABLE(PERF_HLSQ_VS_BATCH_COUNT_ZERO),
      FD7_COUNTABLE(PERF_HLSQ_WAVE_PENDING_NO_QUAD),
      FD7_COUNTABLE(PERF_HLSQ_WAVE_PENDING_NO_PRIM_BASE),
      FD7_COUNTABLE(PERF_HLSQ_STALL_CYCLES_VPC_BE),
      FD7_COUNTABLE(PERF_HLSQ_VSBR_STALL_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_FS_STALL_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_LPAC_STALL_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_BV_STALL_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_VSBR_DEREF_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_FS_DEREF_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_LPAC_DEREF_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_BV_DEREF_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_VSBR_S2W_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_FS_S2W_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_LPAC_S2W_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_BV_S2W_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_VSBR_WAIT_FS_S2W),
      FD7_COUNTABLE(PERF_HLSQ_FS_WAIT_VS_S2W),
      FD7_COUNTABLE(PERF_HLSQ_LPAC_WAIT_VS_S2W),
      FD7_COUNTABLE(PERF_HLSQ_BV_WAIT_FS_S2W),
      FD7_COUNTABLE(PERF_HLSQ_FS_WAIT_SAME_VS_S2W),
      FD7_COUNTABLE(PERF_HLSQ_FS_STARVING_SP),
      FD7_COUNTABLE(PERF_HLSQ_VS_DATA_WAIT_PROGRAMMING),
      FD7_COUNTABLE(PERF_HLSQ_BV_DATA_WAIT_PROGRAMMING),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXTS_VS),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXT_CYCLES_VS),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXTS_FS),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXT_CYCLES_FS),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXTS_BV),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXT_CYCLES_BV),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXTS_LPAC),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_WAVE_CONTEXT_CYCLES_LPAC),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_WARMUP_INC_VS),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_WARMUP_INC_FS),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_WARMUP_INC_BV),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_WARMUP_INC_LPAC),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_MISS_INC_VS),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_MISS_INC_FS),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_MISS_INC_BV),
      FD7_COUNTABLE(PERF_HLSQ_SPTROC_STCHE_MISS_INC_LPAC),
      FD7_COUNTABLE(PERF_HLSQ_VSBR_S2W_CYCLES_SP),
      FD7_COUNTABLE(PERF_HLSQ_FS_S2W_CYCLES_SP),
      FD7_COUNTABLE(PERF_HLSQ_LPAC_S2W_CYCLES_SP),
      FD7_COUNTABLE(PERF_HLSQ_BV_S2W_CYCLES_SP),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_HLSQ),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_HLSQ_HIT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_SP),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_SP_HIT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_INS_HLSQ),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_INS_HLSQ_HIT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_INS_SP),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_INS_SP_HIT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_REQ_UCHE),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_LATENCY_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_STALL_SP_MISS_REQ),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_BANK0_REPLACEMENT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_BANK1_REPLACEMENT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_BANK2_REPLACEMENT),
      FD7_COUNTABLE(PERF_HLSQ_L2STC_BANK3_REPLACEMENT),
      FD7_COUNTABLE(PERF_HLSQ_S2W_STALL_BY_MISS_RETURN),
      FD7_COUNTABLE(PERF_HLSQ_MISS_RETURN_STALL_BY_S2W),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_L0_STALL_INS_RD),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_L0_INS_MISS),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_L0_INS_HIT),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_L0_INS_LATENCY_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_L0_INS_LATENCY_CYCLE),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_DPS_RUN_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_STPROC_DPS_RUN_CYCLE),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BR_QUERY_REQ),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BV_QUERY_REQ),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BR_QUERY_REQ_WHEN_BV_PENDING),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BR_QUERY_BUSY),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BV_QUERY_BUSY),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BR_QUERY_FAIL),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BV_QUERY_FAIL),
      FD7_COUNTABLE(PERF_HLSQ_VS_CTXT_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_FS_CTXT_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_BV_CTXT_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_VS_CONST_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_FS_CONST_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_BV_CONST_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_VS_INS_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_FS_INS_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_BV_INS_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_VS_DES_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_FS_DES_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_BV_DES_BUF_FULL_BLOCK_CPI),
      FD7_COUNTABLE(PERF_HLSQ_PRIMITIVE_COUNT),
      FD7_COUNTABLE(PERF_HLSQ_LPAC2BV_SWITCH_CNT),
      FD7_COUNTABLE(PERF_HLSQ_BV2LPAC_SWITCH_CNT),
      FD7_COUNTABLE(PERF_HLSQ_LPAC2BV_SWITCH_CYC),
      FD7_COUNTABLE(PERF_HLSQ_BV2LPAC_SWITCH_CYC),
      FD7_COUNTABLE(PERF_HLSQ_VSDP_BV2BR_SWITCH_CYC),
};

static const struct fd_perfcntr_counter vpc_counters[] = {
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(0)), REG7(RBBM_PERFCTR_VPC(0))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(1)), REG7(RBBM_PERFCTR_VPC(1))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(2)), REG7(RBBM_PERFCTR_VPC(2))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(3)), REG7(RBBM_PERFCTR_VPC(3))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(4)), REG7(RBBM_PERFCTR_VPC(4))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(5)), REG7(RBBM_PERFCTR_VPC(5))),
};

static const struct fd_perfcntr_countable vpc_countables[] = {
      FD7_COUNTABLE(PERF_VPC_NEVER_COUNT),
      FD7_COUNTABLE(PERF_VPC_FE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_VFD_WACK),
      FD7_COUNTABLE(PERF_VPC_FE_STARVE_CYCLES_SP),
      FD7_COUNTABLE(PERF_VPC_FE_PC_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_FE_SP_COMPONENTS),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_VPCRAM_POS),
      FD7_COUNTABLE(PERF_VPC_FE_VS_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_VS_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_NUM_VPCRAM_READ_POS),
      FD7_COUNTABLE(PERF_VPC_FE_WIT_FULL_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_VPCRAM_FULL_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_NUM_VPCRAM_WRITE),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_TSE_FE),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_VPC_US),
      FD7_COUNTABLE(PERF_VPC_FE_TSE_FE_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_FE_GS_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_FE_TSE_FE_TRANSACTIONS),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_CCU),
      FD7_COUNTABLE(PERF_VPC_FE_NUM_WM_HIT),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_DQ_WACK),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_PRG_END_FE),
      FD7_COUNTABLE(PERF_VPC_FE_STALL_CYCLES_PRG_END_VPCVS),
      FD7_COUNTABLE(PERF_VPC_FE_POSRAM_FULL_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_GMEM_NOP_FULL_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_GMEM_POS_FULL_CYCLES),
      FD7_COUNTABLE(PERF_VPC_FE_BOTTLENECK),
      FD7_COUNTABLE(PERF_VPC_US_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VPC_US_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VPC_US_STARVE_CYCLES_TSE_FE),
      FD7_COUNTABLE(PERF_VPC_US_PTUS_FULL),
      FD7_COUNTABLE(PERF_VPC_US_COMP_INVIS_PRIM_COUNT),
      FD7_COUNTABLE(PERF_VPC_US_STALL_CYCLES_VSC),
      FD7_COUNTABLE(PERF_VPC_US_STALL_CYCLES_VPC_BE),
      FD7_COUNTABLE(PERF_VPC_US_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_VPC_US_STREAMOUT_TRANSACTION),
      FD7_COUNTABLE(PERF_VPC_US_NUM_GMEM_READ_SO),
      FD7_COUNTABLE(PERF_VPC_US_STARVE_CYCLES_UCHE_RD),
      FD7_COUNTABLE(PERF_VPC_US_STALL_CYCLES_PRG_END_VPCUS),
      FD7_COUNTABLE(PERF_VPC_US_STARVE_CYCLES_REORDER),
      FD7_COUNTABLE(PERF_VPC_US_BOTTLENECK),
      FD7_COUNTABLE(PERF_VPC_BE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VPC_BE_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_TSE_BE),
      FD7_COUNTABLE(PERF_VPC_BE_TSE_BE_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_BE_TSE_BE_TRANSACTIONS),
      FD7_COUNTABLE(PERF_VPC_BE_STARVE_CYCLES_LRZ),
      FD7_COUNTABLE(PERF_VPC_BE_LRZ_ASSIGN_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_BE_RB_VISIBLE_PRIMITIVES),
      FD7_COUNTABLE(PERF_VPC_BE_STARVE_CYCLES_RB),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_HLSQ_PRIM_ALLOC),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_SP_LM),
      FD7_COUNTABLE(PERF_VPC_BE_NUM_PA_REQ),
      FD7_COUNTABLE(PERF_VPC_BE_NUM_LM_REQ_HIT),
      FD7_COUNTABLE(PERF_VPC_BE_NUM_ATTR_REQ_LM),
      FD7_COUNTABLE(PERF_VPC_BE_LM_TRANSACTION),
      FD7_COUNTABLE(PERF_VPC_BE_PS_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VPC_BE_PS_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_CCHE),
      FD7_COUNTABLE(PERF_VPC_BE_STARVE_CYCLES_CCHE),
      FD7_COUNTABLE(PERF_VPC_BE_LM_FULL_WAIT_FOR_INTP_END),
      FD7_COUNTABLE(PERF_VPC_BE_CCHE_REQBUF_FULL),
      FD7_COUNTABLE(PERF_VPC_BE_CCHE_NUM_POS_REQ),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_LM_ACK),
      FD7_COUNTABLE(PERF_VPC_BE_STALL_CYCLES_PRG_END_VPCPS),
      FD7_COUNTABLE(PERF_VPC_BE_POS_OVERFETCH_ATTR),
      FD7_COUNTABLE(PERF_VPC_BE_BOTTLENECK),
};

static const struct fd_perfcntr_counter tse_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(0)), REG7(RBBM_PERFCTR_TSE(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(1)), REG7(RBBM_PERFCTR_TSE(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(2)), REG7(RBBM_PERFCTR_TSE(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(3)), REG7(RBBM_PERFCTR_TSE(3))),
};

static const struct fd_perfcntr_countable tse_countables[] = {
      FD7_COUNTABLE(PERF_TSE_NEVER_COUNT),
      FD7_COUNTABLE(PERF_TSE_BE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_TSE_BE_CLIPPING_CYCLES),
      FD7_COUNTABLE(PERF_TSE_BE_STALL_CYCLES_RAS),
      FD7_COUNTABLE(PERF_TSE_BE_STALL_CYCLES_LRZ_BARYPLANE),
      FD7_COUNTABLE(PERF_TSE_BE_STALL_CYCLES_LRZ_ZPLANE),
      FD7_COUNTABLE(PERF_TSE_BE_STARVE_CYCLES_PC),
      FD7_COUNTABLE(PERF_TSE_BE_INPUT_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_INPUT_NULL_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_TRIVAL_REJ_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_CLIPPED_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_ZERO_AREA_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_FACENESS_CULLED_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_ZERO_PIXEL_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_OUTPUT_NULL_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_OUTPUT_VISIBLE_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_CINVOCATION),
      FD7_COUNTABLE(PERF_TSE_BE_CPRIMITIVES),
      FD7_COUNTABLE(PERF_TSE_BE_2D_INPUT_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_2D_ALIVE_CYCLES),
      FD7_COUNTABLE(PERF_TSE_BE_CLIP_PLANES),
      FD7_COUNTABLE(PERF_TSE_BE_EMPTY_BBOX_KILLED_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_ST1_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_BE_ST2_VPORT_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_BE_ST2_SCISSOR_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_BE_ILLEGAL_BOUNDING_BOX_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_VP_OUT_IS_NAN),
      FD7_COUNTABLE(PERF_TSE_BE_EXCLUDED_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_EARLY_CULL_CLIPPED_PRIM),
      FD7_COUNTABLE(PERF_TSE_BE_BR_STALLS_DUETO_BV_CLIP),
      FD7_COUNTABLE(PERF_TSE_BE_BR_STALLS_DUETO_BV_POLY),
      FD7_COUNTABLE(PERF_TSE_BE_BV_STALLS_DUETO_BR_CLIP),
      FD7_COUNTABLE(PERF_TSE_BE_BV_STALLS_DUETO_BR_POLY),
      FD7_COUNTABLE(PERF_TSE_BE_BV_STALLS_DUETO_BR),
      FD7_COUNTABLE(PERF_TSE_FE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_TSE_FE_STALL_CYCLES_VPC_US),
      FD7_COUNTABLE(PERF_TSE_FE_STARVE_CYCLES_PC),
      FD7_COUNTABLE(PERF_TSE_FE_INPUT_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_INPUT_NULL_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_TRIVAL_REJ_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_ZERO_AREA_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_FACENESS_CULLED_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_ZERO_PIXEL_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_OUTPUT_NULL_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_OUTPUT_VISIBLE_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_CINVOCATION),
      FD7_COUNTABLE(PERF_TSE_FE_CPRIMITIVES),
      FD7_COUNTABLE(PERF_TSE_FE_CLIP_PLANES),
      FD7_COUNTABLE(PERF_TSE_FE_EMPTY_BBOX_KILLED_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_ST1_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_FE_ST2_VPORT_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_FE_ST2_SCISSOR_VP_PARAMS_CACHE_MISS),
      FD7_COUNTABLE(PERF_TSE_FE_ILLEGAL_BOUNDING_BOX_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_VP_OUT_IS_NAN),
      FD7_COUNTABLE(PERF_TSE_FE_EXCLUDED_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_EARLY_CULL_CLIPPED_PRIM),
      FD7_COUNTABLE(PERF_TSE_FE_BR_STALLS_DUETO_BV_CLIP),
      FD7_COUNTABLE(PERF_TSE_FE_BR_STALLS_DUETO_BV_POLY),
      FD7_COUNTABLE(PERF_TSE_FE_BV_STALLS_DUETO_BR_CLIP),
      FD7_COUNTABLE(PERF_TSE_FE_BV_STALLS_DUETO_BR_POLY),
      FD7_COUNTABLE(PERF_TSE_FE_BV_STALLS_DUETO_BR),
      FD7_COUNTABLE(PERF_TSE_BE_STALL_CYCLES_LRZ_PRIM),
};

static const struct fd_perfcntr_counter ras_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(0)), REG7(RBBM_PERFCTR_RAS(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(1)), REG7(RBBM_PERFCTR_RAS(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(2)), REG7(RBBM_PERFCTR_RAS(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(3)), REG7(RBBM_PERFCTR_RAS(3))),
};

static const struct fd_perfcntr_countable ras_countables[] = {
      FD7_COUNTABLE(PERF_RAS_NEVER_COUNT),
      FD7_COUNTABLE(PERF_RAS_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_RAS_SUPERTILE_ACTIVE_CYCLES),
      FD7_COUNTABLE(PERF_RAS_STALL_CYCLES_LRZ),
      FD7_COUNTABLE(PERF_RAS_STARVE_CYCLES_TSE),
      FD7_COUNTABLE(PERF_RAS_SUPER_TILES),
      FD7_COUNTABLE(PERF_RAS_8X4_TILES),
      FD7_COUNTABLE(PERF_RAS_MASKGEN_ACTIVE),
      FD7_COUNTABLE(PERF_RAS_FULLY_COVERED_SUPER_TILES),
      FD7_COUNTABLE(PERF_RAS_FULLY_COVERED_8X4_TILES),
      FD7_COUNTABLE(PERF_RAS_PRIM_KILLED_INVISILBE),
      FD7_COUNTABLE(PERF_RAS_SUPERTILE_GEN_ACTIVE_CYCLES),
      FD7_COUNTABLE(PERF_RAS_LRZ_INTF_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_RAS_BLOCKS),
      FD7_COUNTABLE(PERF_RAS_FALSE_PARTIAL_STILE),
      FD7_COUNTABLE(PERF_RAS_SLICE_BLOCK_NONEMTPY),
      FD7_COUNTABLE(PERF_RAS_SLICE_BLOCK_EMPTY),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_0_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_1_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_2_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_3_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_4_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_5_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_6_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_7_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_8_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_9_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_10_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_11_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_12_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_13_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_14_WORKING_CC_L2),
      FD7_COUNTABLE(PERF_RAS_SAMPLE_MASK_GEN_LANE_15_WORKING_CC_L2),
};

static const struct fd_perfcntr_counter uche_counters[] = {
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(0)), REG7(RBBM_PERFCTR_UCHE(0))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(1)), REG7(RBBM_PERFCTR_UCHE(1))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(2)), REG7(RBBM_PERFCTR_UCHE(2))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(3)), REG7(RBBM_PERFCTR_UCHE(3))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(4)), REG7(RBBM_PERFCTR_UCHE(4))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(5)), REG7(RBBM_PERFCTR_UCHE(5))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(6)), REG7(RBBM_PERFCTR_UCHE(6))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(7)), REG7(RBBM_PERFCTR_UCHE(7))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(8)), REG7(RBBM_PERFCTR_UCHE(8))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(9)), REG7(RBBM_PERFCTR_UCHE(9))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(10)), REG7(RBBM_PERFCTR_UCHE(10))),
      FD7_COUNTER(REG6(UCHE_PERFCTR_UCHE_SEL(11)), REG7(RBBM_PERFCTR_UCHE(11))),
};

static const struct fd_perfcntr_countable uche_countables[] = {
      FD7_COUNTABLE(PERF_UCHE_NEVER_COUNT),
      FD7_COUNTABLE(PERF_UCHE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_UCHE_STALL_CYCLES_ARBITER),
      FD7_COUNTABLE(PERF_UCHE_VBIF_STALL_WRITE_DATA),
      FD7_COUNTABLE(PERF_UCHE_STARVED_CYCLES_VBIF_DECMP),
      FD7_COUNTABLE(PERF_UCHE_STALL_CYCLES_DECMP),
      FD7_COUNTABLE(PERF_UCHE_ARBITER_STALL_CYCLES_VBIF),
      FD7_COUNTABLE(PERF_UCHE_VBIF_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_UCHE_VBIF_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_UCHE_DCMP_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_UCHE_DCMP_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_SP),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_TP),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_TP_UBWC),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_TP_GBIF),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_TP_GMEM),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_VFD),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_VPC),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_HLSQ),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_LRZ),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_PC),
      FD7_COUNTABLE(PERF_UCHE_WRITE_REQUESTS_SP),
      FD7_COUNTABLE(PERF_UCHE_WRITE_REQUESTS_LRZ),
      FD7_COUNTABLE(PERF_UCHE_WRITE_REQUESTS_VPC),
      FD7_COUNTABLE(PERF_UCHE_WRITE_REQUESTS_VSC),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_SP),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_TP),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_VFD),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_VPC),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_HLSQ),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_LRZ),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_PC),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_CH0),
      FD7_COUNTABLE(PERF_UCHE_VBIF_READ_BEATS_CH1),
      FD7_COUNTABLE(PERF_UCHE_VBIF_WRITE_BEATS_CH0),
      FD7_COUNTABLE(PERF_UCHE_VBIF_WRITE_BEATS_CH1),
      FD7_COUNTABLE(PERF_UCHE_GMEM_READ_BEATS),
      FD7_COUNTABLE(PERF_UCHE_GMEM_WRITE_BEATS),
      FD7_COUNTABLE(PERF_UCHE_UBWC_READ_BEATS),
      FD7_COUNTABLE(PERF_UCHE_UBWC_WRITE_BEATS),
      FD7_COUNTABLE(PERF_UCHE_EVICTS),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ0),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ1),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ2),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ3),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ4),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ5),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ6),
      FD7_COUNTABLE(PERF_UCHE_BANK_REQ7),
      FD7_COUNTABLE(PERF_UCHE_TPH_REF_FULL),
      FD7_COUNTABLE(PERF_UCHE_TPH_VICTIM_FULL),
      FD7_COUNTABLE(PERF_UCHE_TPH_EXT_FULL),
      FD7_COUNTABLE(PERF_UCHE_RAM_READ_REQ),
      FD7_COUNTABLE(PERF_UCHE_RAM_WRITE_REQ),
      FD7_COUNTABLE(PERF_UCHE_LONG_LINE_ALL_EVICTS),
      FD7_COUNTABLE(PERF_UCHE_LONG_LINE_PARTIAL_EVICTS),
      FD7_COUNTABLE(PERF_UCHE_TPH_CONFLICT_CL_CCHE),
      FD7_COUNTABLE(PERF_UCHE_TPH_CONFLICT_CL_OTHER),
      FD7_COUNTABLE(PERF_UCHE_DBANK_CONFLICT_CL_CCHE),
      FD7_COUNTABLE(PERF_UCHE_DBANK_CONFLICT_CL_OTHER_CLIENTS),
      FD7_COUNTABLE(PERF_UCHE_CCHE_TPH_QUEUE_FULL),
      FD7_COUNTABLE(PERF_UCHE_CCHE_DPH_IO_QUEUE_FULL),
      FD7_COUNTABLE(PERF_UCHE_CCHE_DPH_CMDPOOL_FULL),
      FD7_COUNTABLE(PERF_UCHE_EVICTS_SP),
      FD7_COUNTABLE(PERF_UCHE_EVICTS_LRZ),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_VPCUS),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_VFD_BYPASS_BV),
      FD7_COUNTABLE(PERF_UCHE_READ_REQUESTS_VFD_BYPASS_BR),
      FD7_COUNTABLE(PERF_BYPC_FULL),
      FD7_COUNTABLE(PERF_BYPC_FULL_CCHE_STALL),
      FD7_COUNTABLE(PERF_BYPC_VHUB_STALL),
      FD7_COUNTABLE(PERF_BYPD_FULL),
      FD7_COUNTABLE(PERF_BYPD_FULL_GBIF_STALL),
      FD7_COUNTABLE(PERF_VHUB_PTABLE_FULL),
      FD7_COUNTABLE(PERF_DHUB_PTABLE_FULL),
      FD7_COUNTABLE(PERF_CCHE_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_CCHE_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_CCHE_UCHE_STALL_WRITE_DATA),
      FD7_COUNTABLE(PERF_CCHE_UCHE_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_CCHE_UCHE_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_SP_TOTAL),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_SP_UBWC),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_SP_GBIF),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_SP_GMEM),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_TP_TOTAL),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_TP_UBWC),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_TP_GBIF),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_TP_GMEM),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_VFD_TOTAL),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUEST_VFD_GMEM),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUEST_VFD_GBIF),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_LRZ),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_VPC),
      FD7_COUNTABLE(PERF_CCHE_WRITE_REQUESTS_SP),
      FD7_COUNTABLE(PERF_CCHE_WRITE_REQUESTS_LRZ),
      FD7_COUNTABLE(PERF_CCHE_READ_REQUESTS_GMEM),
      FD7_COUNTABLE(PERF_CCHE_WRITE_REQUESTS_GMEM),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_TP),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_VFD),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_SP),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_VPC),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_LRZ),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_CH0),
      FD7_COUNTABLE(PERF_CCHE_UCHE_READ_BEATS_CH1),
      FD7_COUNTABLE(PERF_CCHE_GMEM_READ_BEATS_VPC),
      FD7_COUNTABLE(PERF_CCHE_GMEM_READ_BEATS_TP),
      FD7_COUNTABLE(PERF_CCHE_GMEM_READ_BEATS_SP),
      FD7_COUNTABLE(PERF_CCHE_GMEM_READ_BEATS_VFD),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ0),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ1),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ2),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ3),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ4),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ5),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ6),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ7),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ8),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ9),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ10),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ11),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ12),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ13),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ14),
      FD7_COUNTABLE(PERF_CCHE_BANK_REQ15),
      FD7_COUNTABLE(PERF_CCHE_GBANK_REQ0),
      FD7_COUNTABLE(PERF_CCHE_GBANK_REQ1),
      FD7_COUNTABLE(PERF_CCHE_GBANK_REQ2),
      FD7_COUNTABLE(PERF_CCHE_GBANK_REQ3),
      FD7_COUNTABLE(PERF_CCHE_TPH_REF_FULL),
      FD7_COUNTABLE(PERF_CCHE_TPH_VICTIM_FULL),
      FD7_COUNTABLE(PERF_CCHE_TPH_EXT_FULL),
      FD7_COUNTABLE(PERF_CCHE_RAM_READ_REQ),
      FD7_COUNTABLE(PERF_CCHE_RAM_WRITE_REQ),
      FD7_COUNTABLE(PERF_CCHE_TPH_CONFLICT_CL),
      FD7_COUNTABLE(PERF_CCHE_DBANK_CONFLICT),
      FD7_COUNTABLE(PERF_CCHE_TPH_QUEUE_FULL),
      FD7_COUNTABLE(PERF_CCHE_DPH_QUEUE_FULL),
      FD7_COUNTABLE(PERF_CCHE_OPH_QUEUE_FULL),
      FD7_COUNTABLE(PERF_CCHE_WACK_QUEUE_FULL),
      FD7_COUNTABLE(PERF_CCHE_GMEM0_LOCAL_RD_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM0_LOCAL_WR_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM1_LOCAL_RD_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM1_LOCAL_WR_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM0_REMOTE_RD_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM0_REMOTE_WR_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM1_REMOTE_RD_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_GMEM1_REMOTE_WR_REQUEST),
      FD7_COUNTABLE(PERF_CCHE_STALL_CYCLES_TP),
};

static const struct fd_perfcntr_counter tp_counters[] = {
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(0)), REG7(RBBM_PERFCTR_TP(0))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(1)), REG7(RBBM_PERFCTR_TP(1))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(2)), REG7(RBBM_PERFCTR_TP(2))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(3)), REG7(RBBM_PERFCTR_TP(3))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(4)), REG7(RBBM_PERFCTR_TP(4))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(5)), REG7(RBBM_PERFCTR_TP(5))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(6)), REG7(RBBM_PERFCTR_TP(6))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(7)), REG7(RBBM_PERFCTR_TP(7))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(8)), REG7(RBBM_PERFCTR_TP(8))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(9)), REG7(RBBM_PERFCTR_TP(9))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(10)), REG7(RBBM_PERFCTR_TP(10))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(11)), REG7(RBBM_PERFCTR_TP(11))),
};

static const struct fd_perfcntr_countable tp_countables[] = {
      FD7_COUNTABLE(PERF_TP_NEVER_COUNT),
      FD7_COUNTABLE(PERF_TP_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_TP_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_TP_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_TP_LATENCY_TRANS),
      FD7_COUNTABLE(PERF_TP_FLAG_FIFO_DELAY_SAMPLES),
      FD7_COUNTABLE(PERF_TP_FLAG_FIFO_DELAY_CYCLES),
      FD7_COUNTABLE(PERF_TP_L1_CACHELINE_REQUESTS),
      FD7_COUNTABLE(PERF_TP_L1_CACHELINE_MISSES),
      FD7_COUNTABLE(PERF_TP_SP_TP_TRANS),
      FD7_COUNTABLE(PERF_TP_TP_SP_TRANS),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS),
      FD7_COUNTABLE(PERF_TP_FILTER_WORKLOAD_16BIT),
      FD7_COUNTABLE(PERF_TP_FILTER_WORKLOAD_32BIT),
      FD7_COUNTABLE(PERF_TP_QUADS_RECEIVED),
      FD7_COUNTABLE(PERF_TP_QUADS_OFFSET),
      FD7_COUNTABLE(PERF_TP_QUADS_SHADOW),
      FD7_COUNTABLE(PERF_TP_QUADS_ARRAY),
      FD7_COUNTABLE(PERF_TP_QUADS_GRADIENT),
      FD7_COUNTABLE(PERF_TP_QUADS_1D),
      FD7_COUNTABLE(PERF_TP_QUADS_2D),
      FD7_COUNTABLE(PERF_TP_QUADS_BUFFER),
      FD7_COUNTABLE(PERF_TP_QUADS_3D),
      FD7_COUNTABLE(PERF_TP_QUADS_CUBE),
      FD7_COUNTABLE(PERF_TP_DIVERGENT_QUADS_RECEIVED),
      FD7_COUNTABLE(PERF_TP_PRT_NON_RESIDENT_EVENTS),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS_POINT),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS_BILINEAR),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS_MIP),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS_ANISO),
      FD7_COUNTABLE(PERF_TP_OUTPUT_PIXELS_ZERO_LOD),
      FD7_COUNTABLE(PERF_TP_FLAG_CACHE_REQUESTS),
      FD7_COUNTABLE(PERF_TP_FLAG_CACHE_MISSES),
      FD7_COUNTABLE(PERF_TP_L1_5_L2_REQUESTS),
      FD7_COUNTABLE(PERF_TP_2D_OUTPUT_PIXELS),
      FD7_COUNTABLE(PERF_TP_2D_OUTPUT_PIXELS_POINT),
      FD7_COUNTABLE(PERF_TP_2D_OUTPUT_PIXELS_BILINEAR),
      FD7_COUNTABLE(PERF_TP_2D_FILTER_WORKLOAD_16BIT),
      FD7_COUNTABLE(PERF_TP_2D_FILTER_WORKLOAD_32BIT),
      FD7_COUNTABLE(PERF_TP_TPA2TPC_TRANS),
      FD7_COUNTABLE(PERF_TP_L1_MISSES_ASTC_1TILE),
      FD7_COUNTABLE(PERF_TP_L1_MISSES_ASTC_2TILE),
      FD7_COUNTABLE(PERF_TP_L1_MISSES_ASTC_4TILE),
      FD7_COUNTABLE(PERF_TP_L1_5_COMPRESS_REQS),
      FD7_COUNTABLE(PERF_TP_L1_5_L2_COMPRESS_MISS),
      FD7_COUNTABLE(PERF_TP_L1_BANK_CONFLICT),
      FD7_COUNTABLE(PERF_TP_L1_5_MISS_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_TP_L1_5_MISS_LATENCY_TRANS),
      FD7_COUNTABLE(PERF_TP_QUADS_CONSTANT_MULTIPLIED),
      FD7_COUNTABLE(PERF_TP_FRONTEND_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_L1_TAG_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_L1_DATA_WRITE_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_PRE_L1_DECOM_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_BACKEND_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_L1_5_CACHE_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_TP_STARVE_CYCLES_SP),
      FD7_COUNTABLE(PERF_TP_STARVE_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_TP_STALL_CYCLES_UFC),
      FD7_COUNTABLE(PERF_TP_FORMAT_DECOMP_POINT),
      FD7_COUNTABLE(PERF_TP_FILTER_POINT_FP16),
      FD7_COUNTABLE(PERF_TP_FILTER_POINT_FP32),
      FD7_COUNTABLE(PERF_TP_LATENCY_FIFO_FULL),
      FD7_COUNTABLE(PERF_TP_FORMAT_DECOMP_BILINEAR),
      FD7_COUNTABLE(PERF_TP_PACKED_POINT_BOTH_VALID_FP16),
      FD7_COUNTABLE(PERF_TP_PACKED_POINT_SINGLE_VALID_FP16),
      FD7_COUNTABLE(PERF_TP_PACKED_POINT_BOTH_VALID_FP32),
      FD7_COUNTABLE(PERF_TP_PACKED_POINT_SINGLE_VALID_FP32),
};

static const struct fd_perfcntr_counter sp_counters[] = {
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(0)), REG7(RBBM_PERFCTR_SP(0))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(1)), REG7(RBBM_PERFCTR_SP(1))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(2)), REG7(RBBM_PERFCTR_SP(2))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(3)), REG7(RBBM_PERFCTR_SP(3))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(4)), REG7(RBBM_PERFCTR_SP(4))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(5)), REG7(RBBM_PERFCTR_SP(5))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(6)), REG7(RBBM_PERFCTR_SP(6))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(7)), REG7(RBBM_PERFCTR_SP(7))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(8)), REG7(RBBM_PERFCTR_SP(8))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(9)), REG7(RBBM_PERFCTR_SP(9))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(10)), REG7(RBBM_PERFCTR_SP(10))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(11)), REG7(RBBM_PERFCTR_SP(11))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(12)), REG7(RBBM_PERFCTR_SP(12))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(13)), REG7(RBBM_PERFCTR_SP(13))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(14)), REG7(RBBM_PERFCTR_SP(14))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(15)), REG7(RBBM_PERFCTR_SP(15))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(16)), REG7(RBBM_PERFCTR_SP(16))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(17)), REG7(RBBM_PERFCTR_SP(17))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(18)), REG7(RBBM_PERFCTR_SP(18))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(19)), REG7(RBBM_PERFCTR_SP(19))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(20)), REG7(RBBM_PERFCTR_SP(20))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(21)), REG7(RBBM_PERFCTR_SP(21))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(22)), REG7(RBBM_PERFCTR_SP(22))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(23)), REG7(RBBM_PERFCTR_SP(23))),
};

static const struct fd_perfcntr_countable sp_countables[] = {
      FD7_COUNTABLE(PERF_SP_NEVER_COUNT),
      FD7_COUNTABLE(PERF_SP_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_SP_ALU_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_STALL_CYCLES_VPC_BE),
      FD7_COUNTABLE(PERF_SP_STALL_CYCLES_TP),
      FD7_COUNTABLE(PERF_SP_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_SP_STALL_CYCLES_RB),
      FD7_COUNTABLE(PERF_SP_NON_EXECUTION_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_CONTEXTS),
      FD7_COUNTABLE(PERF_SP_WAVE_CONTEXT_CYCLES),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_WAVE_CYCLES),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_WAVE_SAMPLES),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_WAVE_CYCLES),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_WAVE_SAMPLES),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_DURATION_CYCLES),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_DURATION_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_CTRL_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_LOAD_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_EMIT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_NOP_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_WAIT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_FETCH_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_IDLE_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_END_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_LONG_SYNC_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_SHORT_SYNC_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_JOIN_CYCLES),
      FD7_COUNTABLE(PERF_SP_LM_LOAD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_LM_STORE_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_LM_ATOMICS),
      FD7_COUNTABLE(PERF_SP_GM_LOAD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_GM_STORE_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_GM_ATOMICS),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_TEX_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_EFU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_FULL_ALU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_HALF_ALU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_TEX_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_CFLOW_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_EFU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_FULL_ALU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_HALF_ALU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_BARY_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_VS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_ADDR_LOCK_COUNT),
      FD7_COUNTABLE(PERF_SP_UCHE_READ_TRANS),
      FD7_COUNTABLE(PERF_SP_UCHE_WRITE_TRANS),
      FD7_COUNTABLE(PERF_SP_EXPORT_VPC_TRANS),
      FD7_COUNTABLE(PERF_SP_EXPORT_RB_TRANS),
      FD7_COUNTABLE(PERF_SP_PIXELS_KILLED),
      FD7_COUNTABLE(PERF_SP_ICL1_REQUESTS),
      FD7_COUNTABLE(PERF_SP_ICL1_MISSES),
      FD7_COUNTABLE(PERF_SP_HS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_DS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_GS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_CS_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_GPR_READ),
      FD7_COUNTABLE(PERF_SP_GPR_WRITE),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_HALF_EFU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_VS_STAGE_HALF_EFU_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_LM_BANK_CONFLICTS),
      FD7_COUNTABLE(PERF_SP_TEX_CONTROL_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_LOAD_CONTROL_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_FLOW_CONTROL_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_LM_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_DISPATCHER_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_SEQUENCER_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_LOW_EFFICIENCY_STARVED_BY_TP),
      FD7_COUNTABLE(PERF_SP_STARVE_CYCLES_HLSQ),
      FD7_COUNTABLE(PERF_SP_NON_EXECUTION_LS_CYCLES),
      FD7_COUNTABLE(PERF_SP_WORKING_EU),
      FD7_COUNTABLE(PERF_SP_ANY_EU_WORKING),
      FD7_COUNTABLE(PERF_SP_WORKING_EU_FS_STAGE),
      FD7_COUNTABLE(PERF_SP_ANY_EU_WORKING_FS_STAGE),
      FD7_COUNTABLE(PERF_SP_WORKING_EU_VS_STAGE),
      FD7_COUNTABLE(PERF_SP_ANY_EU_WORKING_VS_STAGE),
      FD7_COUNTABLE(PERF_SP_WORKING_EU_CS_STAGE),
      FD7_COUNTABLE(PERF_SP_ANY_EU_WORKING_CS_STAGE),
      FD7_COUNTABLE(PERF_SP_GPR_READ_PREFETCH),
      FD7_COUNTABLE(PERF_SP_GPR_READ_CONFLICT),
      FD7_COUNTABLE(PERF_SP_GPR_WRITE_CONFLICT),
      FD7_COUNTABLE(PERF_SP_GM_LOAD_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_SP_GM_LOAD_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_SP_EXECUTABLE_WAVES),
      FD7_COUNTABLE(PERF_SP_ICL1_MISS_FETCH_CYCLES),
      FD7_COUNTABLE(PERF_SP_BYPASS_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_SP_ANY_EU_WORKING_LPAC),
      FD7_COUNTABLE(PERF_SP_WAVE_ALU_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_EFU_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_INT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_CSP_CYCLES),
      FD7_COUNTABLE(PERF_SP_EWAVE_CONTEXTS),
      FD7_COUNTABLE(PERF_SP_EWAVE_CONTEXT_CYCLES),
      FD7_COUNTABLE(PERF_SP_LPAC_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_SP_LPAC_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_1X_WAVES),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_2X_WAVES),
      FD7_COUNTABLE(PERF_SP_QUADS),
      FD7_COUNTABLE(PERF_SP_CS_INVOCATIONS),
      FD7_COUNTABLE(PERF_SP_PIXELS),
      FD7_COUNTABLE(PERF_SP_LPAC_DRAWCALLS),
      FD7_COUNTABLE(PERF_SP_PI_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_INPUT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_OUTPUT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_HWAVE_WAIT_CYCLES),
      FD7_COUNTABLE(PERF_SP_WAVE_HWAVE_SYNC),
      FD7_COUNTABLE(PERF_SP_OUTPUT_3D_PIXELS),
      FD7_COUNTABLE(PERF_SP_FULL_ALU_MAD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_HALF_ALU_MAD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FULL_ALU_MUL_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_HALF_ALU_MUL_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_FULL_ALU_ADD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_HALF_ALU_ADD_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_BARY_FP32_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_ALU_GPR_READ_CYCLES),
      FD7_COUNTABLE(PERF_SP_ALU_DATA_FORWARDING_CYCLES),
      FD7_COUNTABLE(PERF_SP_LM_FULL_CYCLES),
      FD7_COUNTABLE(PERF_SP_TEXTURE_FETCH_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_SP_TEXTURE_FETCH_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_SP_FS_STAGE_PI_TEX_INSTRUCTION),
      FD7_COUNTABLE(PERF_SP_RAY_QUERY_INSTRUCTIONS),
      FD7_COUNTABLE(PERF_SP_RBRT_KICKOFF_FIBERS),
      FD7_COUNTABLE(PERF_SP_RBRT_KICKOFF_DQUADS),
      FD7_COUNTABLE(PERF_SP_RTU_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_SP_RTU_L0_HITS),
      FD7_COUNTABLE(PERF_SP_RTU_L0_MISSES),
      FD7_COUNTABLE(PERF_SP_RTU_L0_HIT_ON_MISS),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_WAVE_QUEUE),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_L0_HIT_QUEUE),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_L0_MISS_QUEUE),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_L0D_IDX_QUEUE),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_L0DATA),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_REPLACE_CNT),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_MRG_CNT),
      FD7_COUNTABLE(PERF_SP_RTU_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_SP_RTU_OPERAND_FETCH_STALL_CYCLES_L0),
      FD7_COUNTABLE(PERF_SP_RTU_OPERAND_FETCH_STALL_CYCLES_INS_FIFO),
      FD7_COUNTABLE(PERF_SP_RTU_BVH_FETCH_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_SP_RTU_BVH_FETCH_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_SP_STCHE_MISS_INC_VS),
      FD7_COUNTABLE(PERF_SP_STCHE_MISS_INC_FS),
      FD7_COUNTABLE(PERF_SP_STCHE_MISS_INC_BV),
      FD7_COUNTABLE(PERF_SP_STCHE_MISS_INC_LPAC),
      FD7_COUNTABLE(PERF_SP_VGPR_ACTIVE_CONTEXTS),
      FD7_COUNTABLE(PERF_SP_PGPR_ALLOC_CONTEXTS),
      FD7_COUNTABLE(PERF_SP_VGPR_ALLOC_CONTEXTS),
      FD7_COUNTABLE(PERF_SP_RTU_RAY_BOX_INTERSECTIONS),
      FD7_COUNTABLE(PERF_SP_RTU_RAY_TRIANGLE_INTERSECTIONS),
      FD7_COUNTABLE(PERF_SP_SCH_STALL_CYCLES_RTU),
      FD7_COUNTABLE(PERF_SP_EFU_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_SP_BRANCH_TAKEN),
      FD7_COUNTABLE(PERF_SP_BRANCH_NOT_TAKEN),
      FD7_COUNTABLE(PERF_SP_BRANCH_INS_DIVERGENCY_COUNT),
      FD7_COUNTABLE(PERF_SP_BRANCH_INS_COUNT),
      FD7_COUNTABLE(PERF_SP_PREDICT_TAKEN),
      FD7_COUNTABLE(PERF_SP_PREDICT_NOT_TAKEN),
      FD7_COUNTABLE(PERF_SP_PREDICT_INS_DIVERGENCY_COUNT),
      FD7_COUNTABLE(PERF_SP_PREDICT_INS_COUNT),
      FD7_COUNTABLE(PERF_SP_CCHE_UAV_TOTAL_REQ),
      FD7_COUNTABLE(PERF_SP_CCHE_UAV_TOTAL_DUALQUAD),
      FD7_COUNTABLE(PERF_SP_CCHE_NONUAV_TOTAL_REQ),
      FD7_COUNTABLE(PERF_SP_CCHE_NONUAV_TOTAL_DUALQUAD),
      FD7_COUNTABLE(PERF_SP_LB_NONUAV_TOTAL_REQ),
      FD7_COUNTABLE(PERF_SP_LB_NONUAV_TOTAL_DUALQUAD),
      FD7_COUNTABLE(PERF_SP_LB_READ_XFER_ALU),
      FD7_COUNTABLE(PERF_SP_LB_ALU_READ_CONS),
      FD7_COUNTABLE(PERF_SP_LB_READ_ALU_BLOCK_OTHER),
      FD7_COUNTABLE(PERF_SP_LB_WRITE_XFER_VPC),
      FD7_COUNTABLE(PERF_SP_LB_WRITE_VPC_BLOCK_OTHER),
      FD7_COUNTABLE(PERF_SP_LB_LDST_RW_LM),
      FD7_COUNTABLE(PERF_SP_LB_LDST_RW_LM_BLOCKED),
      FD7_COUNTABLE(PERF_SP_LB_LDST_WRITE_CONS),
      FD7_COUNTABLE(PERF_SP_LB_LDST_WRITE_CONS_BLOCKED),
      FD7_COUNTABLE(PERF_SP_GPR_READ_BANK),
      FD7_COUNTABLE(PERF_SP_GPR_WRITE_BANK),
      FD7_COUNTABLE(PERF_SP_VS_WAVE_REQ_PENDING),
      FD7_COUNTABLE(PERF_SP_FS_WAVE_REQ_PENDING),
      FD7_COUNTABLE(PERF_SP_LPAC_WAVE_REQ_PENDING),
      FD7_COUNTABLE(PERF_SP_WAVE_SPLIT_CNT),
      FD7_COUNTABLE(PERF_SP_FS_OOO_WAVE_ACC),
};

static const struct fd_perfcntr_counter rb_counters[] = {
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(0)), REG7(RBBM_PERFCTR_RB(0))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(1)), REG7(RBBM_PERFCTR_RB(1))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(2)), REG7(RBBM_PERFCTR_RB(2))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(3)), REG7(RBBM_PERFCTR_RB(3))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(4)), REG7(RBBM_PERFCTR_RB(4))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(5)), REG7(RBBM_PERFCTR_RB(5))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(6)), REG7(RBBM_PERFCTR_RB(6))),
      FD7_COUNTER(REG6(RB_PERFCTR_RB_SEL(7)), REG7(RBBM_PERFCTR_RB(7))),
};

static const struct fd_perfcntr_countable rb_countables[] = {
      FD7_COUNTABLE(PERF_RB_NEVER_COUNT),
      FD7_COUNTABLE(PERF_RB_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_HLSQ),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_FIFO0_FULL),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_FIFO1_FULL),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_FIFO2_FULL),
      FD7_COUNTABLE(PERF_RB_STARVE_CYCLES_SP),
      FD7_COUNTABLE(PERF_RB_STARVE_CYCLES_LRZ_TILE),
      FD7_COUNTABLE(PERF_RB_STARVE_CYCLES_CCU),
      FD7_COUNTABLE(PERF_RB_STARVE_CYCLES_Z_PLANE),
      FD7_COUNTABLE(PERF_RB_STARVE_CYCLES_BARY_PLANE),
      FD7_COUNTABLE(PERF_RB_Z_WORKLOAD),
      FD7_COUNTABLE(PERF_RB_HLSQ_ACTIVE),
      FD7_COUNTABLE(PERF_RB_Z_READ),
      FD7_COUNTABLE(PERF_RB_Z_WRITE),
      FD7_COUNTABLE(PERF_RB_C_READ),
      FD7_COUNTABLE(PERF_RB_C_WRITE),
      FD7_COUNTABLE(PERF_RB_TOTAL_PASS),
      FD7_COUNTABLE(PERF_RB_Z_PASS),
      FD7_COUNTABLE(PERF_RB_Z_FAIL),
      FD7_COUNTABLE(PERF_RB_S_FAIL),
      FD7_COUNTABLE(PERF_RB_BLENDED_FXP_COMPONENTS),
      FD7_COUNTABLE(PERF_RB_BLENDED_FP16_COMPONENTS),
      FD7_COUNTABLE(PERF_RB_PS_INVOCATIONS),
      FD7_COUNTABLE(PERF_RB_2D_ALIVE_CYCLES),
      FD7_COUNTABLE(PERF_RB_2D_STARVE_CYCLES_SP),
      FD7_COUNTABLE(PERF_RB_2D_VALID_PIXELS),
      FD7_COUNTABLE(PERF_RB_3D_PIXELS),
      FD7_COUNTABLE(PERF_RB_BLENDER_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_RB_ZPROC_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_RB_CPROC_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_RB_SAMPLER_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_CCU_COLOR_READ),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_CCU_COLOR_WRITE),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_CCU_DEPTH_READ),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_CCU_DEPTH_WRITE),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_VPC_BE),
      FD7_COUNTABLE(PERF_RB_BLENDED_FP32_COMPONENTS),
      FD7_COUNTABLE(PERF_RB_COLOR_PIX_TILES),
      FD7_COUNTABLE(PERF_RB_STALL_CYCLES_CCU),
      FD7_COUNTABLE(PERF_RB_EARLY_Z_ARB3_GRANT),
      FD7_COUNTABLE(PERF_RB_LATE_Z_ARB3_GRANT),
      FD7_COUNTABLE(PERF_RB_EARLY_Z_SKIP_GRANT),
      FD7_COUNTABLE(PERF_RB_VRS_1X1_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_2X1_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_1X2_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_2X2_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_2X4_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_4X2_QUADS),
      FD7_COUNTABLE(PERF_RB_VRS_4X4_QUADS),
};

static const struct fd_perfcntr_counter vsc_counters[] = {
      FD7_COUNTER(REG6(VSC_PERFCTR_VSC_SEL(0)), REG7(RBBM_PERFCTR_VSC(0))),
      FD7_COUNTER(REG6(VSC_PERFCTR_VSC_SEL(1)), REG7(RBBM_PERFCTR_VSC(1))),
};

static const struct fd_perfcntr_countable vsc_countables[] = {
      FD7_COUNTABLE(PERF_VSC_NEVER_COUNT),
      FD7_COUNTABLE(PERF_VSC_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_VSC_WORKING_CYCLES),
      FD7_COUNTABLE(PERF_VSC_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_VSC_EOT_NUM),
      FD7_COUNTABLE(PERF_VSC_INPUT_TILES),
      FD7_COUNTABLE(PERF_VSC_TILE_COMP_TRAN),
      FD7_COUNTABLE(PERF_VSC_TILE_BYPASS_TRAN),
};

static const struct fd_perfcntr_counter ccu_counters[] = {
      FD7_COUNTER(REG6(RB_PERFCTR_CCU_SEL(0)), REG7(RBBM_PERFCTR_CCU(0))),
      FD7_COUNTER(REG6(RB_PERFCTR_CCU_SEL(1)), REG7(RBBM_PERFCTR_CCU(1))),
      FD7_COUNTER(REG6(RB_PERFCTR_CCU_SEL(2)), REG7(RBBM_PERFCTR_CCU(2))),
      FD7_COUNTER(REG6(RB_PERFCTR_CCU_SEL(3)), REG7(RBBM_PERFCTR_CCU(3))),
      FD7_COUNTER(REG6(RB_PERFCTR_CCU_SEL(4)), REG7(RBBM_PERFCTR_CCU(4))),
};

static const struct fd_perfcntr_countable ccu_countables[] = {
      FD7_COUNTABLE(PERF_CCU_NEVER_COUNT),
      FD7_COUNTABLE(PERF_CCU_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_CCU_STALL_CYCLES_RB_DEPTH_RETURN),
      FD7_COUNTABLE(PERF_CCU_STALL_CYCLES_RB_COLOR_RETURN),
      FD7_COUNTABLE(PERF_CCU_DEPTH_BLOCKS),
      FD7_COUNTABLE(PERF_CCU_COLOR_BLOCKS),
      FD7_COUNTABLE(PERF_CCU_DEPTH_BLOCK_HIT),
      FD7_COUNTABLE(PERF_CCU_COLOR_BLOCK_HIT),
      FD7_COUNTABLE(PERF_CCU_PARTIAL_BLOCK_READ),
      FD7_COUNTABLE(PERF_CCU_GMEM_READ),
      FD7_COUNTABLE(PERF_CCU_GMEM_WRITE),
      FD7_COUNTABLE(PERF_CCU_2D_RD_REQ),
      FD7_COUNTABLE(PERF_CCU_2D_WR_REQ),
      FD7_COUNTABLE(PERF_CCU_UBWC_COLOR_BLOCKS_CONCURRENT),
      FD7_COUNTABLE(PERF_CCU_UBWC_DEPTH_BLOCKS_CONCURRENT),
      FD7_COUNTABLE(PERF_CCU_COLOR_RESOLVE_DROPPED),
      FD7_COUNTABLE(PERF_CCU_DEPTH_RESOLVE_DROPPED),
      FD7_COUNTABLE(PERF_CCU_COLOR_RENDER_CONCURRENT),
      FD7_COUNTABLE(PERF_CCU_DEPTH_RENDER_CONCURRENT),
      FD7_COUNTABLE(PERF_CCU_COLOR_RESOLVE_AFTER_RENDER),
      FD7_COUNTABLE(PERF_CCU_DEPTH_RESOLVE_AFTER_RENDER),
      FD7_COUNTABLE(PERF_CCU_GMEM_EXTRA_DEPTH_READ),
      FD7_COUNTABLE(PERF_CCU_GMEM_COLOR_READ_4AA),
      FD7_COUNTABLE(PERF_CCU_GMEM_COLOR_READ_4AA_FULL),
      FD7_COUNTABLE(PERF_CCU_COLOR_EVB_STALL),
      FD7_COUNTABLE(PERF_CCU_RENDER_OVERLAP_CRE_C),
      FD7_COUNTABLE(PERF_CCU_RENDER_OVERLAP_CRE_Z),
      FD7_COUNTABLE(PERF_CCU_RENDER_STALL_BY_CRE_C),
      FD7_COUNTABLE(PERF_CCU_RENDER_STALL_BY_CRE_Z),
      FD7_COUNTABLE(PERF_CCU_FULL_SURFACE_RESOLVE_CYCLES),
      FD7_COUNTABLE(PERF_CCU_RENDER_OVERLAP_FULL_SURFACE_RESOLVE),
      FD7_COUNTABLE(PERF_CCU_STALL_BY_FULL_SURFACE_RESOLVE),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_REQUESTS),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_TAG_MISS),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_TAG_HIT_RDY),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_TAG_HIT_NRDY),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_IS_FCLEAR),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_IS_ALPHA0),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_IS_ALPHA1),
      FD7_COUNTABLE(PERF_UFC_L0_TP_HINT_IS_UNCOMP),
      FD7_COUNTABLE(PERF_UFC_L0_SP_REQUESTS),
      FD7_COUNTABLE(PERF_UFC_L0_SP_FILTER_HIT),
      FD7_COUNTABLE(PERF_UFC_L0_SP_FILTER_MISS),
      FD7_COUNTABLE(PERF_UFC_L0_SP_REQ_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_L0_TP_REQ_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_L0_TP_RTN_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_CRE_RESOLVE_EVENTS),
      FD7_COUNTABLE(PERF_CRE_CONCURRENT_RESOLVE_EVENTS),
      FD7_COUNTABLE(PERF_CRE_DROPPED_CLEAR_EVENTS),
      FD7_COUNTABLE(PERF_CRE_ST_BLOCKS_CONCURRENT),
      FD7_COUNTABLE(PERF_CRE_LRZ_ST_BLOCKS_CONCURRENT),
      FD7_COUNTABLE(PERF_CRE_SP_UFC_PREFETCH_REQUESTS),
      FD7_COUNTABLE(PERF_CRE_RESOLVE_CDP_PREFETCH_REQUESTS),
      FD7_COUNTABLE(PERF_CRE_RESOLVE_UFC_PREFETCH_REQUESTS),
      FD7_COUNTABLE(PERF_CRE_DR_UFC_PREFTCH_REQUESTS),
};

static const struct fd_perfcntr_counter lrz_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(0)), REG7(RBBM_PERFCTR_LRZ(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(1)), REG7(RBBM_PERFCTR_LRZ(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(2)), REG7(RBBM_PERFCTR_LRZ(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(3)), REG7(RBBM_PERFCTR_LRZ(3))),
};

static const struct fd_perfcntr_countable lrz_countables[] = {
      FD7_COUNTABLE(PERF_LRZ_NEVER_COUNT),
      FD7_COUNTABLE(PERF_LRZ_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_LRZ_STARVE_CYCLES_RAS),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_RB),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_VSC),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_VPC_BE),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_FLAG_ACR),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_UCHE),
      FD7_COUNTABLE(PERF_LRZ_LRZ_READ),
      FD7_COUNTABLE(PERF_LRZ_LRZ_WRITE),
      FD7_COUNTABLE(PERF_LRZ_READ_LATENCY),
      FD7_COUNTABLE(PERF_LRZ_MERGE_CACHE_UPDATING),
      FD7_COUNTABLE(PERF_LRZ_PRIM_KILLED_BY_MASKGEN),
      FD7_COUNTABLE(PERF_LRZ_PRIM_KILLED_BY_LRZ),
      FD7_COUNTABLE(PERF_LRZ_VISIBLE_PRIM_AFTER_LRZ),
      FD7_COUNTABLE(PERF_LRZ_FULL_8X8_TILES),
      FD7_COUNTABLE(PERF_LRZ_PARTIAL_8X8_TILES),
      FD7_COUNTABLE(PERF_LRZ_TILE_KILLED),
      FD7_COUNTABLE(PERF_LRZ_TOTAL_PIXEL),
      FD7_COUNTABLE(PERF_LRZ_VISIBLE_PIXEL_AFTER_LRZ),
      FD7_COUNTABLE(PERF_LRZ_FEEDBACK_ACCEPT),
      FD7_COUNTABLE(PERF_LRZ_FEEDBACK_DISCARD),
      FD7_COUNTABLE(PERF_LRZ_FEEDBACK_STALL),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_RB_ZPLANE),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_RB_BPLANE),
      FD7_COUNTABLE(PERF_LRZ_RAS_MASK_TRANS),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_MVC),
      FD7_COUNTABLE(PERF_LRZ_TILE_KILLED_BY_IMAGE_VRS),
      FD7_COUNTABLE(PERF_LRZ_TILE_KILLED_BY_Z),
      FD7_COUNTABLE(PERF_LRZ_STALL_CYCLES_HLSQ_BATCH),
      FD7_COUNTABLE(PERF_LRZ_NUM_FLOCK),
};

static const struct fd_perfcntr_counter cmp_counters[] = {
      FD7_COUNTER(REG6(RB_PERFCTR_CMP_SEL(0)), REG7(RBBM_PERFCTR_CMP(0))),
      FD7_COUNTER(REG6(RB_PERFCTR_CMP_SEL(1)), REG7(RBBM_PERFCTR_CMP(1))),
      FD7_COUNTER(REG6(RB_PERFCTR_CMP_SEL(2)), REG7(RBBM_PERFCTR_CMP(2))),
      FD7_COUNTER(REG6(RB_PERFCTR_CMP_SEL(3)), REG7(RBBM_PERFCTR_CMP(3))),
};

static const struct fd_perfcntr_countable cmp_countables[] = {
      FD7_COUNTABLE(PERF_CMPDECMP_NEVER_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_STALL_CYCLES_ARB),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_READ_DATA_CCU),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_WRITE_DATA_CCU),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_READ_REQUEST),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_WRITE_REQUEST),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_READ_DATA),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_WRITE_DATA),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG1_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG2_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG3_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG4_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG5_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG6_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG8_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG1_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG2_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG3_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG4_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG5_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG6_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG8_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_READ_DATA_UCHE_CH0),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_READ_DATA_UCHE_CH1),
      FD7_COUNTABLE(PERF_CMPDECMP_VBIF_WRITE_DATA_UCHE),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_WRITE_FLAG0_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAG0_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_WRITE_FLAGALPHA_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_CDP_FILTER_HIT),
      FD7_COUNTABLE(PERF_CMPDECMP_CDP_FILTER_MISS),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG0_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG1_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG2_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG3_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG4_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG5_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG6_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_DEPTH_READ_FLAG8_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG0_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG1_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG2_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG3_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG4_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG5_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG6_COUNT),
      FD7_COUNTABLE(PERF_CMPDECMP_COLOR_READ_FLAG8_COUNT),
};

static const struct fd_perfcntr_counter ufc_counters[] = {
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(0)), REG7(RBBM_PERFCTR_UFC(0))),
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(1)), REG7(RBBM_PERFCTR_UFC(1))),
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(2)), REG7(RBBM_PERFCTR_UFC(2))),
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(3)), REG7(RBBM_PERFCTR_UFC(3))),
};

static const struct fd_perfcntr_countable ufc_countables[] = {
      FD7_COUNTABLE(PERF_UFC_NEVER_COUNT),
      FD7_COUNTABLE(PERF_UFC_BUSY_CYCLES),
      FD7_COUNTABLE(PERF_UFC_READ_DATA_VBIF),
      FD7_COUNTABLE(PERF_UFC_WRITE_DATA_VBIF),
      FD7_COUNTABLE(PERF_UFC_READ_REQUEST_VBIF),
      FD7_COUNTABLE(PERF_UFC_WRITE_REQUEST_VBIF),
      FD7_COUNTABLE(PERF_UFC_MAIN_HIT_CRE_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_HIT_SP_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_HIT_TP_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_HIT_UBWC_READ),
      FD7_COUNTABLE(PERF_UFC_MAIN_HIT_UBWC_WRITE),
      FD7_COUNTABLE(PERF_UFC_MAIN_MISS_CRE_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_MISS_SP_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_MISS_TP_PREFETCH),
      FD7_COUNTABLE(PERF_UFC_MAIN_MISS_UBWC_READ),
      FD7_COUNTABLE(PERF_UFC_MAIN_MISS_UBWC_WRITE),
      FD7_COUNTABLE(PERF_UFC_MAIN_UBWC_RD_NRDY),
      FD7_COUNTABLE(PERF_UFC_MAIN_UBWC_RD_RDY),
      FD7_COUNTABLE(PERF_UFC_MAIN_TP_RD_NRDY),
      FD7_COUNTABLE(PERF_UFC_MAIN_TP_RD_RDY),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_GBIF_CMD),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_GBIF_RDATA),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_GBIF_WDATA),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_UBWC_WR_FLAG),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_UBWC_FLAG_RTN),
      FD7_COUNTABLE(PERF_UFC_STALL_CYCLES_UBWC_EVENT),
      FD7_COUNTABLE(PERF_UFC_UBWC_REQ_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_UBWC_RD_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_UBWC_WR_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_PREFETCH_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_EVICTION_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_LOCK_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_MISS_LATENCY_CYCLES),
      FD7_COUNTABLE(PERF_UFC_MISS_LATENCY_SAMPLES),
      FD7_COUNTABLE(PERF_UFC_L1_CRE_REQUESTS),
      FD7_COUNTABLE(PERF_UFC_L1_CRE_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_L1_CRE_FILTER_HIT),
      FD7_COUNTABLE(PERF_UFC_L1_CRE_FILTER_MISS),
      FD7_COUNTABLE(PERF_UFC_L1_SP_REQUESTS),
      FD7_COUNTABLE(PERF_UFC_L1_SP_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_L1_SP_FILTER_HIT),
      FD7_COUNTABLE(PERF_UFC_L1_SP_FILTER_MISS),
      FD7_COUNTABLE(PERF_UFC_L1_TP_HINT_REQUESTS),
      FD7_COUNTABLE(PERF_UFC_L1_TP_STALLED_CYCLES),
      FD7_COUNTABLE(PERF_UFC_L1_TP_HINT_TAG_MISS),
      FD7_COUNTABLE(PERF_UFC_L1_TP_HINT_TAG_HIT_RDY),
      FD7_COUNTABLE(PERF_UFC_L1_TP_HINT_TAG_HIT_NRDY),
};

static const struct fd_perfcntr_counter bv_cp_counters[] = {
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(0)), REG7(RBBM_PERFCTR2_CP(0))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(1)), REG7(RBBM_PERFCTR2_CP(1))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(2)), REG7(RBBM_PERFCTR2_CP(2))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(3)), REG7(RBBM_PERFCTR2_CP(3))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(4)), REG7(RBBM_PERFCTR2_CP(4))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(5)), REG7(RBBM_PERFCTR2_CP(5))),
      FD7_COUNTER(REG7(CP_BV_PERFCTR_CP_SEL(6)), REG7(RBBM_PERFCTR2_CP(6))),
};

static const struct fd_perfcntr_counter bv_pc_counters[] = {
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(8)), REG7(RBBM_PERFCTR_BV_PC(0))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(9)), REG7(RBBM_PERFCTR_BV_PC(1))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(10)), REG7(RBBM_PERFCTR_BV_PC(2))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(11)), REG7(RBBM_PERFCTR_BV_PC(3))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(12)), REG7(RBBM_PERFCTR_BV_PC(4))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(13)), REG7(RBBM_PERFCTR_BV_PC(5))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(14)), REG7(RBBM_PERFCTR_BV_PC(6))),
      FD7_COUNTER(REG7(PC_PERFCTR_PC_SEL(15)), REG7(RBBM_PERFCTR_BV_PC(7))),
};

static const struct fd_perfcntr_counter bv_vfd_counters[] = {
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(8)), REG7(RBBM_PERFCTR_BV_VFD(0))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(9)), REG7(RBBM_PERFCTR_BV_VFD(1))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(10)), REG7(RBBM_PERFCTR_BV_VFD(2))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(11)), REG7(RBBM_PERFCTR_BV_VFD(3))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(12)), REG7(RBBM_PERFCTR_BV_VFD(4))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(13)), REG7(RBBM_PERFCTR_BV_VFD(5))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(14)), REG7(RBBM_PERFCTR_BV_VFD(6))),
      FD7_COUNTER(REG7(VFD_PERFCTR_VFD_SEL(15)), REG7(RBBM_PERFCTR_BV_VFD(7))),
};

static const struct fd_perfcntr_counter bv_vpc_counters[] = {
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(6)), REG7(RBBM_PERFCTR_BV_VPC(0))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(7)), REG7(RBBM_PERFCTR_BV_VPC(1))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(8)), REG7(RBBM_PERFCTR_BV_VPC(2))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(9)), REG7(RBBM_PERFCTR_BV_VPC(3))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(10)), REG7(RBBM_PERFCTR_BV_VPC(4))),
      FD7_COUNTER(REG7(VPC_PERFCTR_VPC_SEL(11)), REG7(RBBM_PERFCTR_BV_VPC(5))),
};

static const struct fd_perfcntr_counter bv_tp_counters[] = {
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(12)), REG7(RBBM_PERFCTR2_TP(0))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(13)), REG7(RBBM_PERFCTR2_TP(1))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(14)), REG7(RBBM_PERFCTR2_TP(2))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(15)), REG7(RBBM_PERFCTR2_TP(3))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(16)), REG7(RBBM_PERFCTR2_TP(4))),
      FD7_COUNTER(REG7(TPL1_PERFCTR_TP_SEL(17)), REG7(RBBM_PERFCTR2_TP(5))),
};

static const struct fd_perfcntr_counter bv_sp_counters[] = {
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(24)), REG7(RBBM_PERFCTR2_SP(0))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(25)), REG7(RBBM_PERFCTR2_SP(1))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(26)), REG7(RBBM_PERFCTR2_SP(2))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(27)), REG7(RBBM_PERFCTR2_SP(3))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(28)), REG7(RBBM_PERFCTR2_SP(4))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(29)), REG7(RBBM_PERFCTR2_SP(5))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(30)), REG7(RBBM_PERFCTR2_SP(6))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(31)), REG7(RBBM_PERFCTR2_SP(7))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(32)), REG7(RBBM_PERFCTR2_SP(8))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(33)), REG7(RBBM_PERFCTR2_SP(9))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(34)), REG7(RBBM_PERFCTR2_SP(10))),
      FD7_COUNTER(REG7(SP_PERFCTR_SP_SEL(35)), REG7(RBBM_PERFCTR2_SP(11))),
};

static const struct fd_perfcntr_counter bv_ufc_counters[] = {
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(4)), REG7(RBBM_PERFCTR2_UFC(0))),
      FD7_COUNTER(REG7(RB_PERFCTR_UFC_SEL(5)), REG7(RBBM_PERFCTR2_UFC(1))),
};

static const struct fd_perfcntr_counter bv_tse_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(0)), REG7(RBBM_PERFCTR_BV_TSE(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(1)), REG7(RBBM_PERFCTR_BV_TSE(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(2)), REG7(RBBM_PERFCTR_BV_TSE(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_TSE_SEL(3)), REG7(RBBM_PERFCTR_BV_TSE(3))),
};

static const struct fd_perfcntr_counter bv_ras_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(0)), REG7(RBBM_PERFCTR_BV_RAS(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(1)), REG7(RBBM_PERFCTR_BV_RAS(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(2)), REG7(RBBM_PERFCTR_BV_RAS(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_RAS_SEL(3)), REG7(RBBM_PERFCTR_BV_RAS(3))),
};

static const struct fd_perfcntr_counter bv_lrz_counters[] = {
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(0)), REG7(RBBM_PERFCTR_BV_LRZ(0))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(1)), REG7(RBBM_PERFCTR_BV_LRZ(1))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(2)), REG7(RBBM_PERFCTR_BV_LRZ(2))),
      FD7_COUNTER(REG6(GRAS_PERFCTR_LRZ_SEL(3)), REG7(RBBM_PERFCTR_BV_LRZ(3))),
};

static const struct fd_perfcntr_counter bv_hlsq_counters[] = {
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(0)), REG7(RBBM_PERFCTR2_HLSQ(0))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(1)), REG7(RBBM_PERFCTR2_HLSQ(1))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(2)), REG7(RBBM_PERFCTR2_HLSQ(2))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(3)), REG7(RBBM_PERFCTR2_HLSQ(3))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(4)), REG7(RBBM_PERFCTR2_HLSQ(4))),
      FD7_COUNTER(REG7(SP_PERFCTR_HLSQ_SEL(5)), REG7(RBBM_PERFCTR2_HLSQ(5))),
};

const struct fd_perfcntr_group a7xx_perfcntr_groups[] = {
      GROUP("CP", cp_counters, cp_countables),
      GROUP("RBBM", rbbm_counters, rbbm_countables),
      GROUP("PC", pc_counters, pc_countables),
      GROUP("VFD", vfd_counters, vfd_countables),
      GROUP("HLSQ", hlsq_counters, hlsq_countables),
      GROUP("VPC", vpc_counters, vpc_countables),
      GROUP("TSE", tse_counters, tse_countables),
      GROUP("RAS", ras_counters, ras_countables),
      GROUP("UCHE", uche_counters, uche_countables),
      GROUP("TP", tp_counters, tp_countables),
      GROUP("SP", sp_counters, sp_countables),
      GROUP("RB", rb_counters, rb_countables),
      GROUP("VSC", vsc_counters, vsc_countables),
      GROUP("CCU", ccu_counters, ccu_countables),
      GROUP("LRZ", lrz_counters, lrz_countables),
      GROUP("CMP", cmp_counters, cmp_countables),
      GROUP("UFC", ufc_counters, ufc_countables),
      GROUP("BV_CP", bv_cp_counters, cp_countables),
      GROUP("BV_PC", bv_pc_counters, pc_countables),
      GROUP("BV_VFD", bv_vfd_counters, vfd_countables),
      GROUP("BV_VPC", bv_vpc_counters, vpc_countables),
      GROUP("BV_TP", bv_tp_counters, tp_countables),
      GROUP("BV_SP", bv_sp_counters, sp_countables),
      GROUP("BV_UFC", bv_ufc_counters, ufc_countables),
      GROUP("BV_TSE", bv_tse_counters, tse_countables),
      GROUP("BV_RAS", bv_ras_counters, ras_countables),
      GROUP("BV_LRZ", bv_lrz_counters, lrz_countables),
      GROUP("BV_HLSQ", bv_hlsq_counters, hlsq_countables),
};

const unsigned a7xx_num_perfcntr_groups = ARRAY_SIZE(a7xx_perfcntr_groups);

