/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

/**
 * This test involves a writer with buffer date and duration informed in each write, using
 *`copy_to_shm`. A follower reads the buffers and check for consitency in time related information.
 *
 * The test succed if all buffers has been checked.
 **/

#undef NDEBUG  // get assert in release mode

#include <array>
#include <cassert>
#include <future>
#include <iostream>
#include <thread>

#include "../sh4lt/follower.hpp"
#include "../sh4lt/logger/console.hpp"
#include "../sh4lt/writer.hpp"

using namespace sh4lt;

// a struct with contiguous data storage.
using Frame = struct frame_t { int64_t count{0}; };

const size_t num_buffer_to_write = 20;
size_t num_buffer_checked = 0;

void writer(logger::Logger::ptr logger) {
  auto shtype = ShType::deserialize(R"(
{
  "label": "check timecode",
  "group": "testing",
  "media": "application/x-check-timecode",
  "property": {}
}
)");

  Writer w(shtype, sizeof(Frame), logger);
  assert(w);
  int64_t frame_duration_ms = 1;
  int64_t frame_date_ms = 0;
  std::this_thread::sleep_for(std::chrono::milliseconds(50));
  Frame frame;
  for (size_t i = 0; i < num_buffer_to_write; ++i) {
    assert(
        w.copy_to_shm(&frame, sizeof(Frame), frame_date_ms * 1000000, frame_duration_ms * 1000000));
    std::this_thread::sleep_for(std::chrono::milliseconds(frame_duration_ms));
    // prepare next frame
    frame_date_ms += frame_duration_ms;
    ++frame_duration_ms;
    ++frame.count;
  }
}

auto main() -> int {
  using namespace sh4lt;
  auto logger = std::make_shared<logger::Console>();
  {
    Time::info_t previous{-1, -1, -1, -1, -1};
    Follower follower(
        ShType::get_path("check timecode", "testing"),
        // on data
        [&](void* data, size_t size, const Time::info_t* info) {
          auto frame = static_cast<Frame*>(data);
          std::cout << "(copy) new data for client " << frame->count << " (size " << size << ")"
                    << std::endl
                    << " system clock " << info->system_clock_date << std::endl
                    << " steady clock " << info->steady_clock_date << std::endl
                    << " buff num " << info->buffer_number << std::endl
                    << " buff date " << info->buffer_date << std::endl
                    << " buff dur " << info->buffer_duration << std::endl;
          // basic checks
          assert(info->system_clock_date > 0);
          assert(info->steady_clock_date > 0);
          assert(info->buffer_date != -1);
          assert(info->buffer_duration != -1);
          // check consistencies among time informations
          assert(info->buffer_number == frame->count);
          assert(info->buffer_number == previous.buffer_number + 1);
          if (previous.buffer_number != -1) {
            auto sys_dur = info->system_clock_date - previous.system_clock_date;
            auto steady_dur = info->steady_clock_date - previous.steady_clock_date;
            assert(0.99 * previous.buffer_duration < sys_dur ||
                   sys_dur < 1.01 * previous.buffer_duration);
            assert(0.99 * previous.buffer_duration < steady_dur ||
                   steady_dur < 1.01 * previous.buffer_duration);
            assert(previous.buffer_date + previous.buffer_duration == info->buffer_date);
          }
          // save date information for next message checks
          previous = *info;
          // buffer has been checked
          ++num_buffer_checked;
        },
        // on server connected
        [&](const ShType& shtype) {
          assert(shtype.media() == "application/x-check-timecode");
          assert(shtype.label() == "check timecode");
          assert(shtype.group() == "testing");
        },
        // on server disconnected
        [&]() {},
        logger);

    auto writer_handle = std::async(std::launch::async, writer, logger);
    writer_handle.get();
  }

  // check connected and disconnected handlers has been called:
  assert(num_buffer_to_write == num_buffer_checked);
  return 0;
}

