/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 */

#undef NDEBUG  // get assert in release mode

#include <array>
#include <cassert>
#include <future>
#include <iostream>
#include <thread>

#include "../sh4lt/follower.hpp"
#include "../sh4lt/logger/console.hpp"
#include "../sh4lt/writer.hpp"

using namespace sh4lt;

const int num_writes = 10;
const int ms_to_wait_before_writes = 50;

// a struct with contiguous data storage
using Frame = struct frame_t {
  size_t count{0};
  std::array<int, 3> data{{3, 1, 4}};
  // no vector ! vector.data is contiguous, but not vector
};

auto main() -> int {
  using namespace sh4lt;
  auto logger = std::make_shared<logger::Console>();
  auto on_data_called = false;
  {
    Writer w(ShType("application/x-check-sh4lt", "check-writer-follower", "testing"),
             sizeof(Frame),
             logger);
    assert(w);
    Frame frame;

    auto num_follow = 1;
    while (-1 != --num_follow) {
      Follower follower(
          ShType::get_path("check-writer-follower", "testing"),
          [&](void* data, size_t size, const sh4lt_time_info_t*) {
            auto frame = static_cast<Frame*>(data);
            std::cout << "(copy) new data for client " << frame->count << " (size " << size << ")"
                      << std::endl;
            on_data_called = true;
          },
          nullptr,
          nullptr,
          logger);
      auto i = num_writes;
      std::this_thread::sleep_for(std::chrono::milliseconds(ms_to_wait_before_writes));
      while (0 != i--) {
        assert(w.copy_to_shm(&frame, sizeof(Frame), -1, -1));
        frame.count++;
      }
    }
  }
  if (!on_data_called) return 1;
  return 0;
}

