import importlib

import pytest
from fastapi.testclient import TestClient
from inline_snapshot import snapshot

from ...utils import needs_py310


@pytest.fixture(
    name="client",
    params=[
        pytest.param("tutorial002_py39"),
        pytest.param("tutorial002_py310", marks=needs_py310),
    ],
)
def get_client(request: pytest.FixtureRequest):
    mod = importlib.import_module(f"docs_src.body_updates.{request.param}")

    client = TestClient(mod.app)
    return client


def test_get(client: TestClient):
    response = client.get("/items/baz")
    assert response.status_code == 200, response.text
    assert response.json() == {
        "name": "Baz",
        "description": None,
        "price": 50.2,
        "tax": 10.5,
        "tags": [],
    }


def test_patch_all(client: TestClient):
    response = client.patch(
        "/items/foo",
        json={
            "name": "Fooz",
            "description": "Item description",
            "price": 3,
            "tax": 10.5,
            "tags": ["tag1", "tag2"],
        },
    )
    assert response.json() == {
        "name": "Fooz",
        "description": "Item description",
        "price": 3,
        "tax": 10.5,
        "tags": ["tag1", "tag2"],
    }


def test_patch_name(client: TestClient):
    response = client.patch(
        "/items/bar",
        json={"name": "Barz"},
    )
    assert response.json() == {
        "name": "Barz",
        "description": "The bartenders",
        "price": 62,
        "tax": 20.2,
        "tags": [],
    }


def test_openapi_schema(client: TestClient):
    response = client.get("/openapi.json")
    assert response.status_code == 200, response.text
    assert response.json() == snapshot(
        {
            "openapi": "3.1.0",
            "info": {"title": "FastAPI", "version": "0.1.0"},
            "paths": {
                "/items/{item_id}": {
                    "get": {
                        "responses": {
                            "200": {
                                "description": "Successful Response",
                                "content": {
                                    "application/json": {
                                        "schema": {"$ref": "#/components/schemas/Item"}
                                    }
                                },
                            },
                            "422": {
                                "description": "Validation Error",
                                "content": {
                                    "application/json": {
                                        "schema": {
                                            "$ref": "#/components/schemas/HTTPValidationError"
                                        }
                                    }
                                },
                            },
                        },
                        "summary": "Read Item",
                        "operationId": "read_item_items__item_id__get",
                        "parameters": [
                            {
                                "required": True,
                                "schema": {"title": "Item Id", "type": "string"},
                                "name": "item_id",
                                "in": "path",
                            }
                        ],
                    },
                    "patch": {
                        "responses": {
                            "200": {
                                "description": "Successful Response",
                                "content": {
                                    "application/json": {
                                        "schema": {"$ref": "#/components/schemas/Item"}
                                    }
                                },
                            },
                            "422": {
                                "description": "Validation Error",
                                "content": {
                                    "application/json": {
                                        "schema": {
                                            "$ref": "#/components/schemas/HTTPValidationError"
                                        }
                                    }
                                },
                            },
                        },
                        "summary": "Update Item",
                        "operationId": "update_item_items__item_id__patch",
                        "parameters": [
                            {
                                "required": True,
                                "schema": {"title": "Item Id", "type": "string"},
                                "name": "item_id",
                                "in": "path",
                            }
                        ],
                        "requestBody": {
                            "content": {
                                "application/json": {
                                    "schema": {"$ref": "#/components/schemas/Item"}
                                }
                            },
                            "required": True,
                        },
                    },
                }
            },
            "components": {
                "schemas": {
                    "Item": {
                        "type": "object",
                        "title": "Item",
                        "properties": {
                            "name": {
                                "anyOf": [{"type": "string"}, {"type": "null"}],
                                "title": "Name",
                            },
                            "description": {
                                "anyOf": [{"type": "string"}, {"type": "null"}],
                                "title": "Description",
                            },
                            "price": {
                                "anyOf": [{"type": "number"}, {"type": "null"}],
                                "title": "Price",
                            },
                            "tax": {"title": "Tax", "type": "number", "default": 10.5},
                            "tags": {
                                "title": "Tags",
                                "type": "array",
                                "items": {"type": "string"},
                                "default": [],
                            },
                        },
                    },
                    "ValidationError": {
                        "title": "ValidationError",
                        "required": ["loc", "msg", "type"],
                        "type": "object",
                        "properties": {
                            "loc": {
                                "title": "Location",
                                "type": "array",
                                "items": {
                                    "anyOf": [{"type": "string"}, {"type": "integer"}]
                                },
                            },
                            "msg": {"title": "Message", "type": "string"},
                            "type": {"title": "Error Type", "type": "string"},
                            "input": {"title": "Input"},
                            "ctx": {"title": "Context", "type": "object"},
                        },
                    },
                    "HTTPValidationError": {
                        "title": "HTTPValidationError",
                        "type": "object",
                        "properties": {
                            "detail": {
                                "title": "Detail",
                                "type": "array",
                                "items": {
                                    "$ref": "#/components/schemas/ValidationError"
                                },
                            }
                        },
                    },
                }
            },
        }
    )
