#!/bin/sh
#
# srecord - Manipulate EPROM load files
# Copyright (C) 2011 Peter Miller
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#

TEST_SUBJECT="read stag prom prog bin"
. test_prelude

cat > test.in.1 << 'fubar'
S00600004844521B
S111000048656C6C6F2C20576F726C64210A7B
S9030000FC
fubar
if test $? -ne 0; then no_result; fi

srec_cat test.in.1 -mot -o test.in -ppb
if test $? -ne 0; then no_result; fi

cat > test.ok << 'fubar'
00000000: 48 65 6C 6C 6F 2C 20 57 6F 72 6C 64 21 0A        #Hello, World!.
fubar
if test $? -ne 0; then no_result; fi

srec_cat test.in -ppb -o test.out -hexdump
if test $? -ne 0; then fail; fi

diff test.ok test.out
if test $? -ne 0; then fail; fi

# Test stripping of any ASCII preamble
cat > test.preamble << 'fubar'
This text prepending the PPB binary should be stripped
away and ignored. While SRecord doesn't generated any
leading ASCII data, it appears that some tools do. So
thanks to dg1yfe's patch, SRecord now handles it gracefully
fubar

cat test.preamble test.in > test.in.p
if test $? -ne 0; then fail; fi

srec_cat test.in.p -ppb -o test.out -hexdump
if test $? -ne 0; then fail; fi

diff test.ok test.out
if test $? -ne 0; then fail; fi

# Test a large multi-block example ie with holes
cat > test.in.2 << 'fubar'
S00600004844521B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S9030000FC
fubar
if test $? -ne 0; then no_result; fi

srec_cat test.in.2 -mot -o test.in -ppb
if test $? -ne 0; then no_result; fi

cat > test.ok << 'fubar'
00000000: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00000010: 00 00 00 00 00 00 00 00 01 00 0A 00 00 00 5F 00  #.............._.
00000020: DC 00 BA 00 00 00 00 00 00 01 00 0A 00 04 00 00  #\.:.............
00000030: 02 E0 00 0F 00 00 00 00 00 00 01 00 0A 00 08 00  #.`..............
00000040: 02 02 E0 00 09 00 00 00 00 00 00 01 00 0A 00 0C  #..`.............
00000050: 00 04 02 E0 00 03 00 00 00 00 00 00 01 00 0A 00  #...`............
00000060: 10 00 06 02 E0 00 FD 00 00 00 00 00 00 01 00 0A  #....`.}.........
00000070: 00 14 00 0A 02 E0 00 F5 00 00 00 00 00 00 01 00  #.....`.u........
00000080: 0A 00 18 00 08 02 E0 00 F3 00 00 00 00 00 00 01  #......`.s.......
00000090: 00 0A 00 40 00 8A 13 C0 00 58 00 00 00 00 00 00  #...@...@.X......
000000A0: 01 00 0A 00 D0 00 DE 13 80 00 B4 00 00 00 00 00  #....P.^...4.....
000000B0: 00 01 00 0A 00 D4 00 4A 14 80 00 43 00 00 00 00  #.....T.J...C....
000000C0: 00 00 01 00 0A 00 D8 00 32 02 E0 00 09 00 00 00  #......X.2.`.....
000000D0: 00 00 00 01 00 74 00 44 00 00 00 00 00 00 00 00  #.....t.D........
000000E0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000000F0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00000300: 10 C2 E7 80 00 03 80 02 0B CE 55 00 20 C2 45 C0  #.Bg......NU. BE@
00000310: FF C2 00 02 0C 1F 10 80 00 C1 10 C3 E7 82 00 03  #.B.......A.Cg...
00000320: 80 03 0B CE 55 40 00 C3 45 C0 FF 5F 10 82 00 5F  #...NU@.CE@._..._
00000330: 20 86 00 02 04 5F 10 86 00 83 50 83 55 D5 10 CC  # ...._....P.UU.L
00000340: 01 15 15 CA 01 D5 15 00 F7 15 0A 87 00 00 95 C1  #...J.U..w......A
00000350: 17 4A 00 DF 09 5E 12 A6 10 00 93 C1 17 4C 00 DF  #.J._.^.&...A.L._
00000360: 09 5E 12 8E E0 00 9D FF FF C1 17 4C 00 DF 09 5E  #.^..`....A.L._.^
00000370: 12 A6 10 00 95 C1 17 4A 00 DF 09 5E 12 82 65 83  #.&...A.J._.^..e.
00000380: 15 03 0B 87 00 DF 0B 68 00 51 03 C0 17 64 00 C1  #....._.h.Q.@.d.A
00000390: 15 04 29 DF 09 4A 11 C0 15 64 00 DF 09 6C 12 DF  #..)_.J.@.d._.l._
000003A0: 10 66 00 C1 10 C0 17 6E 00 DF 09 4A 11 C0 15 DC  #.f.A.@.n._.J.@.\
000003B0: 05 DF 09 6C 12 C3 E7 68 00 05 81 03 0A DF 15 00  #._.l.Cgh....._..
000003C0: F7 C0 25 0C 01 03 0B DF 0B C0 25 08 03 D7 20 D0  #w@%...._.@%..W P
000003D0: 07 05 04 1F 0A C0 25 DF 55 00 00 02 F4 DF 10 68  #.....@%_U...t_.h
000003E0: 00 02 0A C0 15 0A 00 DF 09 6C 12 C3 65 DC 37 DF  #...@..._.l.Ce\7_
000003F0: 10 6A 00 C0 17 66 00 C1 15 D7 3E DF 09 4A 11 00  #.j.@.f.A.W>_.J..
00000400: 11 DF 09 6C 12 DF 10 4E 00 C0 10 C1 17 4A 00 DF  #._.l._.N.@.A.J._
00000410: 09 5E 12 9F 10 50 00 C0 17 4E 00 C1 17 76 00 00  #.^...P.@.N.A.v..
00000420: 00 00 00 00 01 00 64 01 98 03 4C 00 DF 09 5E 12  #......d...L._.^.
00000430: C2 E5 6E 0A 9F 10 52 00 87 00 DF 15 00 F7 C0 25  #Ben...R..._..w@%
00000440: 1F 0A 66 00 DF 15 DC 37 6A 00 DF 67 68 00 6A 00  #..f._.\7j._gh.j.
00000450: 1F 0A 4E 00 1F 0A 50 00 DF 15 92 F5 52 00 87 00  #..N...P._..uR...
00000460: 01 10 C4 17 64 00 84 0C 84 0C 84 0C 02 9D 66 27  #..D.d.........f'
00000470: 9F 0A 8A 00 C3 17 8A 00 C3 45 E0 FF C3 9C 73 27  #....C...CE`.C.s'
00000480: C2 60 9F 60 8C 00 C3 17 8C 00 C3 45 FC FF C4 15  #B`.`..C...CE|.D.
00000490: 0C 00 C5 15 B0 27 C3 65 93 27 D5 94 95 90 C5 65  #..E.0'Ce.'U...Ee
000004A0: 06 00 C4 0A FA 06 9F 0A 90 00 DF 45 FC FF 90 00  #..D.z....._E|...
000004B0: DF 65 80 01 8E 00 DF 45 7F FC 8E 00 DF 15 54 8C  #_e...._E.|.._.T.
000004C0: A8 27 DF 57 90 00 A8 27 DF 57 8E 00 A8 27 C0 15  #('_W..('_W..('@.
000004D0: A2 27 5F 00 42 02 C0 17 46 00 07 80 17 20 4C FF  #"'_.B.@.F.... L.
000004E0: 0A 06 C0 65 68 01 07 80 F9 01 17 20 B4 00 03 07  #..@eh...y.. 4...
000004F0: C0 E5 68 01 FA 81 1F 10 46 00 02 80 C0 65 68 01  #@eh.z...F...@eh.
00000500: C0 0C 1F 1C DC 31 4A 00 17 20 1C 02 02 05 C0 E5  #@...\1J.. ....@e
00000510: D0 02 1F 1C 90 32 4C 00 87 00 C1 17 50 00 01 80  #P....2L...A.P...
00000520: 01 0B C0 17 6E 00 DF 09 4A 11 C0 15 B8 0B DF 09  #..@.n._.J.@.8._.
00000530: 6C 12 DF 0B 50 00 01 80 03 0B C2 10 83 0C C3 67  #l._.P.....B...Cg
00000540: 54 00 9F 60 54 00 E6 10 C1 10 01 80 01 0B C0 17  #T..`T.f.A.....@.
00000550: 6E 00 DF 09 4A 11 C0 15 58 02 C3 67 56 00 42 0B  #n._.J.@.X.CgV.B.
00000560: DF 09 6C 12 D6 0B 01 80 03 0B DF 60 5C 00 9F 10  #_.l.V....._`\...
00000570: 56 00 C1 17 52 00 01 80 01 0B C0 17 6E 00 DF 09  #V.A.R.....@.n._.
00000580: 4A 11 C0 15 B8 0B DF 09 CB 00 00 00 00 00 00 01  #J.@.8._.K.......
00000590: 00 64 01 F6 04 6C 12 DF 0B 52 00 01 80 03 0B C2  #.d.v.l._.R.....B
000005A0: 10 83 0C C3 67 58 00 9F 60 58 00 E6 10 C1 10 01  #...CgX..`X.f.A..
000005B0: 80 01 0B C0 17 6E 00 DF 09 4A 11 C0 15 58 02 C3  #...@.n._.J.@.X.C
000005C0: 67 5A 00 42 0B DF 09 6C 12 D6 0B 01 80 03 0B DF  #gZ.B._.l.V....._
000005D0: 60 5E 00 9F 10 5A 00 87 00 DF 15 C0 FF 84 00 C5  #`^...Z..._.@...E
000005E0: 17 44 00 1F 0A 44 00 C0 17 6E 00 41 11 01 80 01  #.D...D.@.n.A....
000005F0: 0B DF 09 4A 11 C0 15 3C 00 C3 67 48 00 42 0B DF  #._.J.@.<.CgH.B._
00000600: 09 6C 12 C5 0B 01 80 03 0B DF 60 46 00 9F 10 48  #.l.E....._`F...H
00000610: 00 DF 09 44 04 DF 65 02 00 92 00 C1 17 92 00 C1  #._.D._e....A...A
00000620: 45 FD FF 41 1C AC 34 C0 15 D8 26 66 10 DF 09 42  #E}.A.,4@.X&f._.B
00000630: 02 9F 15 BA 34 DF 17 86 00 88 00 DF 17 60 00 64  #...:4_....._.`.d
00000640: 00 DF 09 00 03 DF 09 88 04 C4 17 5C 00 C4 65 80  #._..._...D.\.De.
00000650: 57 84 0C 84 0C 84 0C 84 0C 84 0C 1F 11 72 00 05  #W............r..
00000660: 11 C4 0C C4 65 F0 28 00 15 00 63 80 0C 1F 10 7C  #.D.Dep(...c....|
00000670: 00 C0 E7 5E 00 00 0B 1F 10 7E 00 DF 0B 66 00 08  #.@g^.....~._.f..
00000680: 07 C0 15 88 35 DF 09 CE 03 DF 15 88 35 C2 34 02  #.@..5_.N._..5B4.
00000690: 01 1F 0A C2 34 C4 17 72 00 36 81 17 21 0A 00 33  #...B4D.r.6..!..3
000006A0: 07 17 21 7A 03 35 04 C4 17 5E 00 53 81 17 21 A8  #..!z.5.D.^.S..!(
000006B0: 61 34 04 17 21 C2 01 4D 07 DF 17 72 00 B4 34 84  #a4..!B.M._.r.44.
000006C0: 0C 84 0C 84 0C 84 0C 84 0C C4 65 2B 00 1F 11 B6  #.........De+...6
000006D0: 34 DF 0B 9E 00 04 03 DF 09 EC 0C 1F 0A 9E 00 DF  #4_....._.l....._
000006E0: 09 78 15 C4 17 7E 00 C4 E5 10 00 0C 80 C0 15 80  #.x.D.~.De....@..
000006F0: 02 DF 09 15 00 00 00 00 00 00 01 00 64 01 54 06  #._..........d.T.
00000700: 06 0E 9F 11 9E 00 DF 09 EC 0C 1F 0A 9E 00 DF 09  #......_.l....._.
00000710: 32 0E 87 00 C0 15 0D 00 C1 15 FC 1F 08 01 C0 15  #2...@...A.|...@.
00000720: 77 03 C1 15 1C 20 03 01 00 11 C1 15 62 20 C0 0C  #w.A.. ....A.b @.
00000730: C0 0C C0 0C C0 0C C0 0C C0 E5 80 57 1F 10 5C 00  #@.@.@.@.@e.W..\.
00000740: 5F 10 CA 34 1F 0A 68 00 1F 0A 54 00 C3 17 5E 00  #_.J4..h...T.C.^.
00000750: 83 0C 83 0C 01 81 03 0B DF 10 58 00 5F 00 3A 05  #........_.X._.:.
00000760: CE 27 9E 00 6C 03 C0 17 72 00 C0 E5 09 00 1F 10  #N'..l.@.r.@e....
00000770: 74 00 C0 0C C0 0C C0 0C C0 0C C0 0C C0 E5 80 57  #t.@.@.@.@.@.@e.W
00000780: 1F 10 76 00 DF 09 04 0F 9F 13 9E 00 C0 17 5C 00  #..v._.......@.\.
00000790: C0 E7 76 00 03 10 C0 0C 03 60 83 0C DF 10 B4 34  #@gv...@..`.._.44
000007A0: 02 0A C0 15 30 00 DF 09 6C 12 C3 67 74 00 DF 10  #..@.0._.l.Cgt._.
000007B0: 78 00 C3 0C C4 10 C0 15 30 00 80 E0 85 10 01 1D  #x.C.D.@.0..`....
000007C0: F0 28 DF 09 3A 12 E6 10 40 11 01 1D F2 28 DF 09  #p(_.:.f.@...r(_.
000007D0: 3A 12 02 0A C0 15 30 00 83 65 05 80 03 0B DF 09  #:...@.0..e...._.
000007E0: 6C 12 03 0B 02 01 DF 09 6C 12 C4 10 83 0C 83 0C  #l....._.l.D.....
000007F0: DF 10 7C 00 DF 09 0C 10 1F 11 7A 00 C0 17 5E 00  #_.|._.....z.@.^.
00000900: 00 B2 03 26 0A 26 10 DF 09 90 0C DF 15 B8 23 CA  #.2.&.&._..._.8#J
00000910: 34 DF 09 04 0F A8 01 57 21 11 00 A5 06 17 21 A8  #4_...(.W!..%..!(
00000920: FD 5C 07 DF 15 FE 23 CA 34 DF 09 32 0E DF 09 78  #}\._.~#J4_.2._.x
00000930: 15 C5 17 7E 00 29 81 57 21 03 00 29 07 C4 17 58  #.E.~.).W!..).D.X
00000940: 00 17 21 A8 FD 15 05 17 21 D4 FE 16 05 17 21 6A  #..!(}...!T~...!j
00000950: FF 17 05 D7 27 CA 34 0A 21 02 02 1F 0A CA 34 C0  #...W'J4.!....J4@
00000960: 17 78 00 26 10 DF 09 CA 0C 81 15 C1 0C 79 00 EC  #.x.&._.J...A.y.l
00000970: 0F DF 15 AC 20 CA 34 F3 01 DF 15 D8 20 CA 34 EF  #._., J4s._.X J4o
00000980: 01 DF 15 0A 21 CA 34 EB 01 57 21 F6 FF 26 07 1F  #._..!J4k.W!v.&..
00000990: 0A 60 00 DF 15 C2 01 AA 25 1F 0A C2 34 1F 0A D2  #.`._.B.*%..B4..R
000009A0: 34 C5 0B 01 03 DC 80 C4 17 58 00 17 21 A8 FD 15  #4E...\.D.X..!(}.
000009B0: 07 17 21 D4 FE 0F 07 17 21 6A FF 09 07 17 21 B0  #..!T~...!j....!0
000009C0: FF 03 07 C0 15 4E 21 16 01 C0 15 76 21 13 01 C0  #...@.N!..@.v!..@
000009D0: 15 A4 21 10 01 C0 15 CE 21 0D 01 DF 15 12 22 CA  #.$!..@.N!.._.."J
000009E0: 34 C0 15 20 00 DF 09 06 0E DF 09 04 0F 1F 0A BA  #4@. ._..._.....:
000009F0: 34 DF 09 32 0E 1F 0A C2 34 1F 10 5F 00 00 00 00  #4_.2...B4.._....
00000A00: 00 00 01 00 64 01 24 09 CA 34 D7 27 54 00 64 00  #....d.$.J4W'T.d.
00000A10: 1B 06 D7 27 54 00 9C FF 17 05 D7 27 46 00 F1 FF  #..W'T.....W'F.q.
00000A20: 18 05 D7 27 46 00 0F 00 14 06 C1 17 78 00 C1 0C  #..W'F.....A.x.A.
00000A30: 40 1C F2 28 40 EC F0 28 02 10 01 80 02 0B 97 20  #@.r(@lp(.......
00000A40: 30 00 0C 04 DF 09 06 0B C0 17 54 00 C1 15 C6 22  #0..._...@.T.A.F"
00000A50: 07 01 C0 17 46 00 C1 15 58 22 02 01 C1 15 14 23  #..@.F.A.X"..A..#
00000A60: 5F 10 C2 34 C1 17 78 00 E6 15 03 00 C0 0B 01 81  #_.B4A.x.f...@...
00000A70: 8E 0A 66 10 DF 09 90 0C C1 0C 42 1C F2 28 42 EC  #..f._...A.B.r(Bl
00000A80: F0 28 83 10 C3 0C 83 60 83 0C 83 0C C2 E0 83 10  #p(..C..`....B`..
00000A90: 83 0C C2 60 06 80 97 20 D3 FF 08 04 C2 15 D3 FF  #..B`... S...B.S.
00000AA0: 05 01 97 20 2D 00 02 07 C2 15 2D 00 C3 15 5A 00  #... -...B.-.C.Z.
00000AB0: C0 0B 01 80 03 0B C2 60 9F 10 46 00 C1 17 92 00  #@.....B`..F.A...
00000AC0: C1 65 02 00 C1 45 FD FF 5F 10 92 00 41 1C AC 34  #Ae..AE}._...A.,4
00000AD0: 66 10 DF 09 44 04 C0 15 D8 26 DF 09 42 02 9F 15  #f._.D.@.X&_.B...
00000AE0: BA 34 DF E5 07 00 B6 34 DF 09 AA 13 0A 00 D7 27  #:4_e..64_.*...W'
00000AF0: 7E 00 96 00 77 04 C5 15 CA 35 C4 17 64 00 17 21  #~...w.E.J5D.d..!
00000B00: 3F 00 02 07 C4 15 3F 00 44 0C 44 0C 44 0C 44 0C  #?...D.?.D.D.D.D.
00000B10: C4 45 7F FC C4 55 50 9C 15 11 D7 27 46 00 2D 00  #DE.|DUP...W'F.-.
00000B20: 61 06 D7 27 46 00 D3 FF 5D 05 C1 17 4A 00 01 80  #a.W'F.S.].A.J...
00000B30: 01 0B C0 17 B6 34 C0 E7 7A 00 04 10 DF 09 4A 11  #..@.64@gz..._.J.
00000B40: C0 17 4C 00 DF 09 6C 12 C4 60 DF 0B 4A 00 01 81  #@.L._.l.D`_.J...
00000B50: 03 0B C3 67 B4 34 D5 10 D5 17 7A 00 D5 15 00 B0  #..Cg44U.U.z.U..0
00000B60: C4 E5 96 00 3F 80 F1 00 00 00 00 00 00 01 00 64  #De..?.q........d
00000B70: 01 82 0A 04 0B 00 11 C1 17 64 00 DF 09 4A 11 83  #.......A.d._.J..
00000B80: 0C 83 0C 83 0C 83 0C 34 03 E6 10 C2 15 60 09 82  #.......4.f.B.`..
00000B90: 0C 82 0C 8E 20 01 86 8E 10 C2 17 B0 00 C3 15 C0  #.... ....B.0.C.@
00000BA0: FF C2 67 70 00 82 0A C2 40 80 9C 66 27 C2 67 52  #.Bgp...B@..f'BgR
00000BB0: 00 C2 40 C0 40 C0 00 00 0C 44 0A C4 45 FF DF C4  #.B@@@@...D.DE._D
00000BC0: 55 00 40 00 51 81 9C 66 27 C1 40 40 50 15 10 C0  #U.@.Q..f'A@@P..@
00000BD0: 65 40 20 C0 45 00 C0 15 10 CE 0A E2 06 D5 15 00  #e@ @E.@..N.b.U..
00000BE0: F7 0D 0A 9F 10 B0 00 D6 0B DF 15 CA 35 D2 34 87  #w....0.V._.J5R4.
00000BF0: 00 1F 0A D2 34 87 00 DF 09 AA 13 FC FF DF 17 B4  #...R4.._.*.|._.4
00000C00: 34 7A 26 DF 17 B6 34 7C 26 DF 15 78 26 D2 34 DF  #4z&_.64|&_.x&R4_
00000C10: 35 7F 00 70 00 FC 02 DF 0B BA 07 52 03 C3 17 10  #5..p.|._.:.R.C..
00000C20: 26 C3 E7 B4 34 C2 17 12 26 C2 E7 7C 26 C2 65 03  #&Cg44B..&Bg|&Be.
00000C30: 00 A6 10 02 03 DF 09 36 0C C3 17 10 26 C3 E7 7A  #.&..._.6.C..&Cgz
00000C40: 26 C3 65 19 00 E6 10 02 0A DF 09 36 0C DF 15 A2  #&Ce..f..._.6._."
00000C50: 24 CA 34 DF 09 AA 13 F8 FF 1F 0A CA 34 DF 09 AA  #$J4_.*.x...J4_.*
00000C60: 13 FE FF 0E 0B 83 13 02 0A DF 09 36 0C 83 15 82  #.~......._.6....
00000C70: 15 02 0B 02 03 DF 09 36 0C DF 09 AA 13 FE FF DF  #....._.6._.*.~._
00000C80: 65 04 00 5E 00 DF 65 D0 07 68 00 1F 0A 58 00 1F  #e..^._eP.h...X..
00000C90: 0A 6C 00 C6 15 FE 3F 1F 0A CA 34 1F 0A 46 00 1F  #.l.F.~?..J4..F..
00000CA0: 0A 54 00 DF 15 1E 00 60 00 DF 09 28 01 DF 0B 9E  #.T._...`._.(._..
00000CB0: 00 F2 02 1F 0A BA 34 1F 0A C2 34 DF 09 AA 13 03  #.r...:4..B4_.*..
00000CC0: 00 E6 15 01 00 E6 17 78 00 DF 09 90 0C C2 15 E8  #.f...f.x._...B.h
00000CD0: FF AC 00 00 00 00 00 00 01 00 64 01 E0 0B C3 15  #.,........d.`.C.
00000CE0: 30 00 C5 17 70 00 05 0C 01 86 03 0B E6 10 DF 09  #0.E.p.......f._.
00000CF0: 36 0C 83 15 02 0A DF 09 36 0C DF 17 7A 26 AC 26  #6....._.6._.z&,&
00000D00: DF 17 7C 26 AE 26 DF 65 14 00 AC 26 DF 15 A8 26  #_.|&.&_e..,&_.(&
00000D10: C2 34 C0 17 78 00 DF 09 72 0C DF 09 72 0C E6 15  #B4@.x._.r._.r.f.
00000D20: 02 00 26 10 DF 09 90 0C DF 15 D6 24 CA 34 DF 09  #..&._..._.V$J4_.
00000D30: AA 13 0A 00 C5 15 80 0A C3 0B 03 80 C5 15 C0 0A  #*...E...C...E.@.
00000D40: 03 0B 5F 11 72 0C 05 0A C2 0B 05 03 85 0A 83 10  #.._.r...B.......
00000D50: 02 80 03 0B 05 0B C0 17 7A 26 DF 09 72 0C 1F 10  #......@.z&_.r...
00000D60: 7A 26 5F 61 7C 26 DF 09 76 0C C3 0A F6 06 87 00  #z&_a|&_.v.C.v...
00000D70: 00 00 87 00 DF 35 07 00 70 00 FC 03 DF 09 78 15  #...._5..p.|._.x.
00000D80: DF 35 07 00 70 00 FC 02 DF 09 78 15 87 00 26 11  #_5..p.|._.x...&.
00000D90: 84 1D 04 00 84 0C 0C 86 F6 0C 06 00 F6 0C 06 00  #........v...v...
00000DA0: F6 0C 06 00 F6 0C 06 00 F4 C5 F0 00 13 30 03 01  #v...v...tEp..0..
00000DB0: F4 C5 0F 00 13 30 B4 DD 06 00 13 30 84 15 B6 13  #tE...04]...0..6.
00000DC0: 04 00 C6 65 04 00 87 00 26 11 84 1D 04 00 84 0C  #..Fe....&.......
00000DD0: 04 9D 13 30 04 86 84 0C 84 0C 84 0C 84 0C C4 45  #...0..........DE
00000DE0: F0 FE 36 11 04 00 84 15 87 00 DF 09 3C 0D C4 17  #p~6......._.<.D.
00000DF0: F0 28 84 0C 84 0C 84 0C 84 0C 84 0C C4 65 17 00  #p(..........De..
00000E00: 15 11 1F 11 82 00 D5 15 50 8C C0 15 F0 28 C0 65  #......U.P.@.p(@e
00000E10: 08 00 04 12 84 0C 84 0C 84 0C 84 0C 84 0C C4 65  #..............De
00000E20: 17 00 DF 09 78 0D F3 01 D5 15 00 F7 0D 0A DF 15  #.._.x.s.U..w.._.
00000E30: 2A 18 DA 34 DF 09 66 0D 87 00 A6 10 04 00 00 00  #*.Z4_.f...&.....
00000E40: 00 00 00 01 00 64 01 3E 0D E6 10 26 10 66 10 26  #.....d.>.f.&.f.&
00000E50: 11 DF 15 E1 00 A0 00 1F 0A DA 34 1F 0A BA 07 C5  #._.a. ...Z4..:.E
00000E60: 15 2A 18 D5 15 54 98 D5 15 A0 F0 15 0A 7E 00 0A  #.*.U.T.U. p..~..
00000E70: 00 B6 13 0C 00 84 15 84 15 81 15 80 15 83 15 82  #.6..............
00000E80: 15 87 00 E6 15 00 02 17 21 00 04 05 87 C4 15 FF  #...f....!....D..
00000E90: 03 1F 21 82 00 08 03 C4 0B 04 80 04 0A DF 0B 82  #..!....D....._..
00000EA0: 00 02 03 CE 55 00 40 DF 0A A2 00 1B 80 9F 0A A4  #...NU.@_.".....$
00000EB0: 00 DF 45 FC FF A4 00 9F 0A A4 00 DF 17 A4 00 A2  #._E|.$...$._.$."
00000EC0: 00 DF 65 80 02 A6 00 DF 45 7F FC A6 00 9F 0A A8  #._e..&._E.|&...(
00000ED0: 00 DF 45 FC FF A8 00 CD 17 A6 00 CD 57 A8 00 D5  #._E|.(.M.&.MW(.U
00000EE0: 55 04 8C C4 E7 82 00 08 80 04 0B C4 45 C0 FF 1F  #U..Dg......DE@..
00000EF0: E1 82 00 C4 55 40 00 04 01 C4 45 C0 FF 1F 61 82  #a..DU@...DE@..a.
00000F00: 00 84 55 15 11 DF 0A A0 00 02 06 CE 65 02 00 87  #..U.._. ...Ne...
00000F10: 00 C1 17 72 00 44 10 C1 0C C1 65 F0 28 E6 15 03  #.A.r.D.A.Aep(f..
00000F20: 00 45 14 43 10 45 E2 01 81 8E 0A 26 11 DF 09 90  #.E.C.Eb....&._..
00000F30: 0C 11 E0 23 E0 80 0C 00 0B FB 02 87 00 1F 0A AE  #..`#`....{......
00000F40: 00 1F 0A C2 34 1F 0A D2 34 DF 55 00 00 02 F4 C5  #...B4..R4_U...tE
00000F50: 15 CA 35 C4 17 AC 00 84 0A C4 00 04 0C 44 63 C4  #.J5D.,...D...DcD
00000F60: 45 7F FC C4 55 50 9C 15 11 DF 09 98 0E DF 55 00  #E.|DUP..._..._U.
00000F70: 00 02 F4 DF E5 0A 00 AE 00 DF 09 98 0E D5 15 00  #..t_e...._...U..
00000F80: F7 0D 0A DF 15 CA 35 C2 34 DF 55 00 00 02 F4 DF  #w.._.J5B4_U...t_
00000F90: 65 21 00 AE 00 D7 27 AE 00 C0 00 D9 07 DF 09 AA  #e!...W'..@.Y._.*
00000FA0: 13 05 00 DF 15 E2 FF B9 00 00 00 00 00 00 01 00  #..._.b.9........
00000FB0: 64 01 9C 0E 46 00 E6 15 F1 00 DF 09 44 04 80 1D  #d...F.f.q._.D...
00000FC0: FA FF 80 0C 80 0A C0 67 70 00 C0 67 AC 00 1F 10  #z.....@gp.@g,...
00000FD0: AC 00 C0 45 E0 FF 04 9C 73 27 C4 67 AE 00 1B 81  #,.@E`...s'Dg....
00000FE0: 00 11 C1 17 4C 00 DF 09 5E 12 C2 67 B4 34 13 81  #..A.L._.^.Bg44..
00000FF0: C2 55 00 40 95 10 00 11 C1 17 4A 00 DF 09 5E 12  #BU.@....A.J._.^.
00001000: C2 67 B6 34 07 81 95 10 9F 0A 46 00 CE 0A D5 06  #Bg64......F.N.U.
00001010: D6 0B 87 00 25 0A 15 0A 15 0A F6 01 DF 09 3C 0D  #V...%.....v._.<.
00001020: 1F 0A AA 00 C0 17 74 00 C0 0C C0 65 F0 28 04 12  #..*.@.t.@.@ep(..
00001030: DF 09 0C 10 C4 0B 02 80 04 0A 05 01 17 21 00 04  #_...D........!..
00001040: 02 87 C4 15 FF 03 15 11 1F 11 82 00 D5 15 50 8C  #..D.........U.P.
00001050: 26 11 04 14 DF 09 0C 10 01 11 84 15 02 0A 26 10  #&..._.........&.
00001060: C0 15 0C 00 01 E1 08 80 C1 E5 06 00 01 0B 43 10  #@....a..Ae....C.
00001070: DF 09 6C 12 03 0B 05 01 C1 65 06 00 43 10 DF 09  #_.l.....Ae..C._.
00001080: 6C 12 C1 10 02 10 80 15 9F 0A AA 00 D7 27 AA 00  #l.A.......*.W'*.
00001090: 03 00 04 05 DF 15 DF 0A 70 0F 07 01 D7 27 AA 00  #...._._.p...W'*.
000010A0: FD FF 03 06 DF 15 9F 0A 70 0F C4 67 AA 00 44 60  #}..._...p.Dg*.D`
000010B0: 26 11 DF 09 78 0D 01 01 04 01 84 15 C2 0A E4 06  #&._.x.......B.d.
000010C0: C7 01 C2 17 74 00 CE 15 13 00 C0 15 18 00 83 10  #G.B.t.N...@.....
000010D0: C3 0C C3 65 F0 28 A6 10 82 0A DF 09 CA 0C 84 15  #C.Cep(&..._.J...
000010E0: C4 0C FC 09 FC 0F D3 0B C0 65 30 00 CE 0A F3 06  #D.|.|.S.@e0.N.s.
000010F0: D6 0B D5 15 00 F7 0D 0A DF 09 66 0D DF 15 2A 18  #V.U..w.._.f._.*.
00001100: DA 34 87 00 90 07 EE 07 20 08 40 08 40 08 D2 07  #Z4....n. .@.@.R.
00001110: B8 07 88 00 00 00 00 00 00 01 00 64 01 FA 0F B8  #8..........d.z.8
00001120: 07 04 0E 1E 10 B8 10 28 10 32 10 E0 10 90 07 14  #.....8.(.2.`....
00001130: 11 26 11 C4 0C 84 65 84 0C 84 0C 84 0C C4 65 17  #.&.D..e......De.
00001140: 00 87 00 DF 09 3C 10 00 00 17 00 E8 FF DF 09 3C  #..._.<.....h._.<
00001150: 10 A6 FF 10 00 EE FF DF 09 3C 10 5A 00 10 00 EE  #.&...n._.<.Z...n
00001160: FF 57 21 96 1F 3A 82 66 10 A6 10 E6 10 26 11 26  #.W!..:.f.&.f.&.&
00001170: 10 80 1D 0A 00 D5 15 00 98 95 13 E6 17 46 00 C4  #.....U.....f.F.D
00001180: 12 DF 09 0C 10 26 11 C4 1C 02 00 DF 09 0C 10 84  #._...&.D..._....
00001190: 65 84 0C 1F 14 46 00 04 64 1F 12 84 00 15 11 F3  #e....F..d......s
000011A0: 22 02 00 08 03 04 82 DF 65 EA FF 46 00 03 01 DF  #"......_ej.F..._
000011B0: 65 16 00 46 00 DF 09 44 04 41 11 C0 15 D8 26 DF  #e..F._.D.A.@.X&_
000011C0: 09 42 02 25 0A 25 0A 9F 15 46 00 DF 09 44 04 80  #.B.%.%...F._.D..
000011D0: 15 84 15 83 15 82 15 81 15 D6 0B 87 00 57 21 C6  #.........V...W!F
000011E0: 1F 2A 82 1F 10 AC 26 C4 12 DF 09 0C 10 01 11 C4  #.*...,&D._.....D
000011F0: 1C 02 00 DF 09 0C 10 44 60 84 0C 1F 11 AE 26 C1  #..._...D`.....&A
00001200: 15 A8 26 13 01 57 21 C2 1F 16 82 1F 10 DC 25 C4  #.(&..W!B.....\%D
00001210: 12 DF 09 0C 10 01 11 C4 1C 02 00 DF 09 0C 10 44  #._.....D..._...D
00001220: 60 84 0C 1F 11 DE 25 C1 15 D8 25 44 14 17 21 00  #`....^%A.X%D..!.
00001230: F7 02 03 15 11 FA 01 87 00 DF 0B C6 07 17 03 17  #w....z..._.F....
00001240: 20 19 00 14 07 17 20 70 03 11 04 C4 1C 02 00 C4  # ..... p...D...D
00001250: 22 01 04 C4 12 DF 09 0C 10 1F 11 12 26 1F 10 10  #"..D._......&...
00001260: 26 9F 11 BA 07 D5 15 00 F7 D5 15 0C 26 87 00 03  #&..:.U..wU..&...
00001270: 0A 40 20 05 86 42 10 25 03 01 10 A1 00 1A 00 00  #.@ ..B.%...!....
00001280: 00 00 00 00 01 00 64 01 58 11 02 01 02 10 20 03  #......d.X..... .
00001290: 42 0C 1F 87 42 0C 21 87 42 0C 24 87 42 0C 27 87  #B...B.!.B.$.B.'.
000012A0: 42 0C 2A 87 42 0C 2D 87 42 0C 30 87 42 0C 33 87  #B.*.B.-.B.0.B.3.
000012B0: 42 0C 36 87 42 0C 39 87 42 0C 3C 87 42 0C 3F 87  #B.6.B.9.B.<.B.?.
000012C0: 42 0C 42 87 42 0C 45 87 42 0C 48 87 02 0A 43 10  #B.B.B.E.B.H...C.
000012D0: 87 00 43 10 C3 0C 42 0C 02 86 43 60 42 0B C3 0C  #..C.C.B...C`B.C.
000012E0: 42 0C 02 86 43 60 42 0B C3 0C 42 0C 02 86 43 60  #B...C`B.C.B...C`
000012F0: 42 0B C3 0C 42 0C 02 86 43 60 42 0B C3 0C 42 0C  #B.C.B...C`B.C.B.
00001300: 02 86 43 60 42 0B C3 0C 42 0C 02 86 43 60 42 0B  #..C`B.C.B...C`B.
00001310: C3 0C 42 0C 02 86 43 60 42 0B C3 0C 42 0C 02 86  #C.B...C`B.C.B...
00001320: 43 60 42 0B C3 0C 42 0C 02 86 43 60 42 0B C3 0C  #C`B.C.B...C`B.C.
00001330: 42 0C 02 86 43 60 42 0B C3 0C 42 0C 02 86 43 60  #B...C`B.C.B...C`
00001340: 42 0B C3 0C 42 0C 02 86 43 60 42 0B C3 0C 42 0C  #B.C.B...C`B.C.B.
00001350: 02 86 43 60 42 0B C3 0C 42 0C 02 86 43 60 42 0B  #..C`B.C.B...C`B.
00001360: C3 0C 42 0C 02 86 43 60 42 0B 87 00 C0 0B 07 80  #C.B...C`B...@...
00001370: 00 0B C1 0B 07 80 01 0B DF 09 4A 11 87 00 C1 0B  #..A....._.J...A.
00001380: FB 80 01 0B DF 09 4A 11 03 0B 42 0B 02 0B 87 00  #{..._.J...B.....
00001390: DF 09 3A 12 C3 0C 42 0C C3 0C 42 0C 87 00 C3 0C  #_.:.C.B.C.B...C.
000013A0: 42 0C 02 E0 3E 80 C3 0C 42 0C 02 60 3F 80 C3 0C  #B..`>.C.B..`?.C.
000013B0: 42 0C 02 60 40 80 C3 0C 42 0C 02 60 41 80 C3 0C  #B..`@.C.B..`A.C.
000013C0: 42 0C 02 60 42 80 C3 0C 42 0C 02 60 43 80 C3 0C  #B..`B.C.B..`C.C.
000013D0: 42 0C 02 60 44 80 C3 0C 42 0C 02 60 45 80 C3 0C  #B..`D.C.B..`E.C.
000013E0: 42 0C 02 60 46 80 C3 0C 6D 00 00 00 00 00 00 01  #B..`F.C.m.......
000013F0: 00 64 01 B6 12 42 0C 02 60 47 80 C3 0C 42 0C 02  #.d.6.B..`G.C.B..
00001400: 60 48 80 C3 0C 42 0C 02 60 49 80 C3 0C 42 0C 02  #`H.C.B..`I.C.B..
00001410: 60 4A 80 C3 0C 42 0C 02 60 4B 80 C3 0C 42 0C 02  #`J.C.B..`K.C.B..
00001420: 60 4C 80 C3 0C 42 0C 02 60 4D 80 02 60 87 00 83  #`L.C.B..`M..`...
00001430: 0A C3 0C 42 0C 02 E0 C1 81 83 0A C3 0C 42 0C 02  #.C.B..`A...C.B..
00001440: E0 C0 81 83 0A C3 0C 42 0C 02 E0 BF 81 83 0A C3  #`@...C.B..`?...C
00001450: 0C 42 0C 02 E0 BE 81 83 0A C3 0C 42 0C 02 E0 BD  #.B..`>...C.B..`=
00001460: 81 83 0A C3 0C 42 0C 02 E0 BC 81 83 0A C3 0C 42  #...C.B..`<...C.B
00001470: 0C 02 E0 BB 81 83 0A C3 0C 42 0C 02 E0 BA 81 83  #..`;...C.B..`:..
00001480: 0A C3 0C 42 0C 02 E0 B9 81 83 0A C3 0C 42 0C 02  #.C.B..`9...C.B..
00001490: E0 B8 81 83 0A C3 0C 42 0C 02 E0 B7 81 83 0A C3  #`8...C.B..`7...C
000014A0: 0C 42 0C 02 E0 B6 81 83 0A C3 0C 42 0C 02 E0 B5  #.B..`6...C.B..`5
000014B0: 81 83 0A C3 0C 42 0C 02 E0 B4 81 83 0A C3 0C 42  #...C.B..`4...C.B
000014C0: 0C 02 E0 B3 81 83 0A 87 00 DF 55 40 00 66 FF 9F  #..`3....._U@.f..
000014D0: 0A 6C 00 9F 0A 70 00 DF 35 0F 00 70 00 04 02 1F  #.l...p._5..p....
000014E0: 0A 3A 15 1F 0A 3C 15 02 00 DF 15 00 F7 C0 25 80  #.:...<..._..w@%.
000014F0: 1F 00 00 C1 17 70 00 02 10 01 80 00 0B C1 65 3C  #...A.p.......Ae<
00001500: 00 C0 0A FC 06 01 00 DF 09 78 15 5F 20 70 00 FA  #.@.|..._.x._ p.z
00001510: 82 CE 65 02 00 C2 0B D7 81 5F 00 DC 00 26 10 66  #.Ne..B.W._.\.&.f
00001520: 10 C0 17 96 00 C1 1F 16 E0 1B 03 E0 17 00 F4 C8  #.@...A..`..`..tH
00001530: 65 02 00 E0 15 24 14 57 22 00 F7 0B 02 F1 0B 02  #e..`.$.W".w..q..
00001540: 00 0F 03 60 10 C8 65 04 00 E0 15 24 14 41 1C 02  #...`.He..`.$.A..
00001550: 00 F2 01 6E 00 00 00 00 00 00 01 00 64 01 14 14  #.r.n........d...
00001560: 1F 10 96 00 5F 10 00 F4 81 15 80 15 02 00 07 14  #...._..t........
00001570: 01 14 E8 01 9F 0A 94 00 C1 17 94 00 C1 0C 41 1C  #..h.....A...A.A.
00001580: 06 15 03 02 1F 0A 94 00 F7 01 DF 15 28 14 CE 00  #........w._.(.N.
00001590: C0 15 CE 00 D7 01 26 10 66 10 C1 17 94 00 5F 20  #@.N.W.&.f.A..._
000015A0: 9A 00 06 03 DF 15 0F 00 9C 00 5F 10 9A 00 28 01  #...._....._...(.
000015B0: DF 0A 9C 00 25 80 C1 0C 79 00 40 15 C1 17 06 F4  #_...%.A.y.@.A..t
000015C0: C1 45 00 FC C1 65 0D 00 C0 17 AA 25 C0 0C C0 0C  #AE.|Ae..@.*%@.@.
000015D0: C0 0C C0 E7 AA 25 01 60 81 0C 81 0C 81 0C 5F 10  #@.@g*%.`......_.
000015E0: AA 25 C1 E5 DB 01 81 0C 57 20 0A 00 02 80 C1 15  #*%Ae[...W ....A.
000015F0: 0A 00 C1 25 64 00 02 04 C1 15 64 00 5F 10 60 00  #..A%d...A.d._.`.
00001600: DF 55 01 00 00 F4 81 15 80 15 02 00 41 1C 06 15  #_U...t......A...
00001610: C0 17 98 00 03 03 F0 45 08 00 02 00 5F 10 98 00  #@.....pE...._...
00001620: F1 55 18 00 02 00 EC 01 C0 17 98 00 E9 03 1F 0A  #qU....l.@...i...
00001630: 98 00 31 10 8C 16 F0 45 08 00 02 00 E1 01 41 1C  #..1...pE....a.A.
00001640: 06 15 5F 1C FE FF 44 00 5F 10 3A 15 5F 10 3C 15  #.._.~.D._.:._.<.
00001650: D7 01 9E 16 B2 16 C6 16 DA 16 F2 16 0C 17 28 17  #W...2.F.Z.r...(.
00001660: 44 17 60 17 7A 17 94 17 AC 17 C6 17 E0 17 FA 17  #D.`.z...,.F.`.z.
00001670: 14 18 B0 34 C0 34 C8 34 D0 34 D8 34 22 25 56 25  #..04@4H4P4X4"%V%
00001680: 42 25 76 25 88 25 00 00 00 00 00 00 DC 14 DC 14  #B%v%.%......\.\.
00001690: DC 14 DC 14 C0 14 C0 14 C0 14 C0 14 C0 14 C0 14  #\.\.@.@.@.@.@.@.
000016A0: C0 14 C0 14 C0 14 C0 14 C0 14 C0 14 22 14 22 14  #@.@.@.@.@.@.".".
000016B0: 22 14 22 14 22 14 F2 14 F2 14 F2 14 F2 14 BC 00  #".".".r.r.r.r.<.
000016C0: 00 00 00 00 00 01 00 64 01 72 15 70 14 F2 14 F2  #.......d.r.p.r.r
000016D0: 14 26 10 66 10 A6 10 E6 10 26 11 66 11 04 0A 05  #.&.f.&.f.&.f....
000016E0: 1D 8C 16 20 03 01 1D 96 16 43 11 C3 65 0A 00 F1  #... .....C.Ce..q
000016F0: 10 12 00 43 1F FC FF 40 1D FE FF 0A 03 C3 0B 01  #...C.|.@.~...C..
00001700: 80 03 0B 02 0A DF 09 6C 12 FD 0B FC FF 01 80 03  #....._.l.}.|....
00001710: 0B 05 1D 96 16 C0 10 C1 15 0A 00 41 61 DF 09 74  #.....@.A...Aa_.t
00001720: 16 D4 0B DD 01 DF 27 60 00 62 00 08 03 C1 15 B8  #.T.]._'`.b...A.8
00001730: 25 C0 17 60 00 1F 10 62 00 DF 09 74 16 85 15 84  #%@.`...b._.t....
00001740: 15 83 15 82 15 81 15 80 15 87 00 D1 95 20 00 17  #...........Q. ..
00001750: 20 10 27 1F 86 D1 95 20 00 26 0A 17 20 64 00 24  # .'..Q. .&.. d.$
00001760: 05 03 10 02 0A C0 15 64 00 DF 09 6C 12 C3 0C C3  #.....@.d._.l.C.C
00001770: 65 14 28 DF 09 58 16 DF 09 58 16 83 10 C3 0C C3  #e.(_.X._.X...C.C
00001780: 65 14 28 DF 09 58 16 CE 11 DF 09 58 16 C6 65 02  #e.(_.X.N._.X.Fe.
00001790: 00 87 00 E6 11 D1 95 30 00 17 20 10 27 DE 87 B1  #...f.Q.0.. .'^.1
000017A0: 8A FF FF C0 E5 10 27 F8 01 02 10 C3 15 14 28 E1  #...@e.'x...C..(a
000017B0: 01 F6 0B 02 00 09 02 D7 A2 30 00 04 02 D1 95 20  #.v.....W"0...Q.
000017C0: 00 83 0A 87 00 B6 11 02 00 D1 94 87 00 C0 0B BD  #.....6...Q...@.=
000017D0: 80 00 0B DF 09 F2 15 40 10 E0 A5 20 00 FD 02 C8  #..._.r.@.`% .}.H
000017E0: 95 2D 00 87 00 F2 16 0C 17 28 17 44 17 00 00 9E  #.-...r...(.D....
000017F0: 16 B2 16 C6 16 DA 16 A0 F0 F0 9E 00 00 DA 02 00  #.2.F.Z. pp...Z..
00001800: 80 20 20 20 20 20 20 00 E0 FC 16 A0 F0 F0 9E FA  #.      .`|. pp.z
00001810: 00 DA 02 00 80 20 20 20 20 20 20 00 E0 16 17 A0  #.Z...      .`..
00001820: F0 F0 9E F4 01 DA 02 00 80 B6 00 00 00 00 00 00  #pp.t.Z...6......
00001830: 01 00 60 01 D0 16 20 20 20 20 20 20 00 E0 32 17  #..`.P.      .`2.
00001840: A0 F0 F0 9E EE 02 DA 02 00 80 20 20 20 20 20 20  # pp.n.Z...
00001850: 00 E0 4E 17 7E 00 00 00 B0 F0 F0 9D 84 03 F7 00  #.`N.~...0pp...w.
00001860: 00 80 20 48 45 49 47 48 54 20 00 F7 00 00 5E 00  #.. HEIGHT .w..^.
00001870: 00 00 B0 F0 F0 9D 84 03 E1 00 00 80 20 41 4C 54  #..0pp...a... ALT
00001880: 49 54 55 44 45 20 00 F7 00 00 5C 00 00 00 B0 F0  #ITUDE .w..\...0p
00001890: F0 9D 84 03 CB 00 00 80 20 44 49 53 54 41 4E 43  #p...K... DISTANC
000018A0: 45 20 00 F7 00 00 68 00 0A 00 B0 F0 F0 9D 84 03  #E .w..h...0pp...
000018B0: B5 00 00 80 20 46 55 45 4C 20 4C 45 46 54 00 F7  #5... FUEL LEFT.w
000018C0: 00 00 6A 00 00 00 B0 F0 F0 9D 84 03 9F 00 00 80  #..j...0pp.......
000018D0: 20 57 45 49 47 48 54 20 00 F7 00 00 66 00 00 00  # WEIGHT .w..f...
000018E0: B0 F0 F0 9D 84 03 89 00 00 80 20 54 48 52 55 53  #0pp....... THRUS
000018F0: 54 20 00 F7 00 00 46 00 00 00 B0 F0 F0 9D 84 03  #T .w..F...0pp...
00001900: 73 00 00 80 20 41 4E 47 4C 45 00 F7 00 00 58 00  #s... ANGLE.w..X.
00001910: 0A 00 B0 F0 F0 9D 84 03 5D 00 00 80 20 56 45 52  #..0pp...]... VER
00001920: 20 56 45 4C 00 F7 00 00 54 00 0A 00 B0 F0 F0 9D  # VEL.w..T...0pp.
00001930: 84 03 47 00 00 80 20 48 4F 52 20 56 45 4C 00 F7  #..G... HOR VEL.w
00001940: 00 00 52 00 F4 01 B0 F0 F0 9D 84 03 31 00 00 80  #..R.t.0pp...1...
00001950: 20 56 45 52 20 41 43 43 00 F7 00 00 50 00 F4 01  # VER ACC.w..P.t.
00001960: B0 F0 F0 9D 84 03 1B 00 00 80 20 48 4F 52 20 41  #0pp....... HOR A
00001970: 43 43 00 F7 00 00 70 00 3C 00 B0 F0 F0 9D 84 03  #CC.w..p.<.0pp...
00001980: 05 00 00 80 20 53 45 43 4F 4E 44 53 00 F7 00 00  #.... SECONDS.w..
00001990: 15 00 00 00 00 00 00 01 00 1B 00 FC 1F 50 9F 1E  #...........|.P..
000019A0: 00 58 02 00 80 42 4F 59 2C 20 41 52 45 20 59 4F  #.X...BOY, ARE YO
000019B0: 55 00 B7 00 00 00 00 00 00 01 00 0F 00 10 20 D8  #U.7........... X
000019C0: 87 20 49 4E 45 50 54 00 C1 00 00 00 00 00 00 01  #. INEPT.A.......
000019D0: 00 2A 00 18 20 00 F7 00 00 50 9F 0D 02 58 02 00  #.*.. .w..P...X..
000019E0: 80 20 20 59 4F 55 20 48 41 56 45 20 4A 55 53 54  #.  YOU HAVE JUST
000019F0: 20 43 52 41 53 48 45 44 00 CD 00 00 00 00 00 00  # CRASHED.M......
00001A00: 01 00 28 00 3C 20 50 9F 0D 02 3A 02 00 80 49 4E  #..(.< P...:...IN
00001A10: 54 4F 20 54 48 45 20 45 44 47 45 20 4F 46 20 54  #TO THE EDGE OF T
00001A20: 48 45 20 4D 4F 4F 4E 00 42 00 00 00 00 00 00 01  #HE MOON.B.......
00001A30: 00 50 00 5E 20 00 F7 00 00 50 9F 32 00 8A 02 00  #.P.^ .w..P.2....
00001A40: 80 53 4F 52 52 59 2C 20 42 55 54 20 57 48 45 4E  #.SORRY, BUT WHEN
00001A50: 20 59 4F 55 20 4C 4F 53 45 20 54 56 20 43 4F 56  # YOU LOSE TV COV
00001A60: 45 52 41 47 45 2C 20 59 4F 55 20 41 4C 53 4F 20  #ERAGE, YOU ALSO
00001A70: 4C 4F 53 45 20 59 4F 55 52 20 46 55 45 4C 00 99  #LOSE YOUR FUEL..
00001A80: 00 00 00 00 00 00 01 00 32 00 A8 20 00 F7 00 00  #........2.( .w..
00001A90: 58 9F 64 00 02 00 00 80 54 4F 4F 20 46 41 53 54  #X.d.....TOO FAST
00001AA0: 2E 20 59 4F 55 27 52 45 20 47 4F 49 4E 47 20 54  #. YOU'RE GOING T
00001AB0: 4F 20 43 52 41 53 48 00 00 00 00 00 00 00 00 01  #O CRASH.........
00001AC0: 00 39 00 D4 20 00 F7 00 00 50 9F 64 00 BC 02 00  #.9.T .w..P.d.<..
00001AD0: 80 42 45 54 54 45 52 20 53 54 41 52 54 20 53 4C  #.BETTER START SL
00001AE0: 4F 57 49 4E 47 20 49 54 20 55 50 20 50 52 45 54  #OWING IT UP PRET
00001AF0: 54 59 20 53 4F 4F 4E 00 AA 00 00 00 00 00 00 01  #TY SOON.*.......
00001B00: 00 4A 00 06 21 00 F7 00 00 50 9F 64 00 02 00 00  #.J..!.w..P.d....
00001B10: 80 54 41 4B 45 20 49 54 20 4E 49 43 45 20 41 4E  #.TAKE IT NICE AN
00001B20: 44 20 45 41 53 59 2E 20 41 20 50 45 52 46 45 43  #D EASY. A PERFEC
00001B30: 54 20 4C 41 4E 44 49 4E 47 20 49 53 20 55 4E 44  #T LANDING IS UND
00001B40: 45 52 20 38 20 46 50 53 00 CC 00 00 00 00 00 00  #ER 8 FPS.L......
00001B50: 01 00 2F 00 4A 21 00 F7 00 00 50 9F 64 00 58 02  #../.J!.w..P.d.X.
00001B60: 00 80 46 41 4E 54 41 53 54 49 43 2C 20 41 20 50  #..FANTASTIC, A P
00001B70: 45 52 46 45 43 54 20 4C 41 4E 44 49 4E 47 00 D1  #ERFECT LANDING.Q
00001B80: 00 00 00 00 00 00 01 00 34 00 72 21 00 F7 00 00  #........4.r!.w..
00001B90: 50 9F 64 00 58 02 00 80 43 4F 4E 47 52 41 54 55  #P.d.X...CONGRATU
00001BA0: 4C 41 54 49 4F 4E 53 20 4F 4E 20 41 20 47 4F 4F  #LATIONS ON A GOO
00001BB0: 44 20 4C 41 4E 44 49 4E 47 00 13 00 00 00 00 00  #D LANDING.......
00001BC0: 00 01 00 30 00 A0 21 00 F7 00 00 50 9F 64 00 58  #...0. !.w..P.d.X
00001BD0: 02 00 80 54 48 45 20 4C 41 4E 44 49 4E 47 20 57  #...THE LANDING W
00001BE0: 41 53 20 41 20 4C 49 54 54 4C 45 20 46 41 53 54  #AS A LITTLE FAST
00001BF0: 00 44 00 00 00 00 00 00 01 00 4B 00 CA 21 00 F7  #.D........K.J!.w
00001C00: 00 00 50 9F 64 00 58 02 00 80 54 48 45 20 4C 41  #..P.d.X...THE LA
00001C10: 4E 44 49 4E 47 20 57 41 53 20 54 4F 4F 20 46 41  #NDING WAS TOO FA
00001C20: 53 54 20 41 4E 44 20 44 41 4D 41 47 45 20 57 41  #ST AND DAMAGE WA
00001C30: 53 20 44 4F 4E 45 20 54 4F 20 54 48 45 20 53 48  #S DONE TO THE SH
00001C40: 49 50 00 21 00 00 00 00 00 00 01 00 4D 00 0E 22  #IP.!........M.."
00001C50: 00 F7 00 00 58 9F 64 00 26 02 00 80 57 45 4C 4C  #.w..X.d.&...WELL
00001C60: 2C 20 59 4F 55 20 43 45 52 54 41 49 4E 4C 59 20  #, YOU CERTAINLY
00001C70: 42 4C 45 57 20 54 48 41 54 20 4F 4E 45 2E 20 54  #BLEW THAT ONE. T
00001C80: 48 45 52 45 20 57 45 52 45 20 4E 4F 20 53 55 52  #HERE WERE NO SUR
00001C90: 56 49 52 4F 52 53 00 CE 00 00 00 00 00 00 01 00  #VIRORS.N........
00001CA0: 47 00 54 22 00 F7 00 00 50 9F 64 00 3A 02 00 80  #G.T".w..P.d.:...
00001CB0: 42 55 54 20 54 48 45 20 41 4E 47 4C 45 20 57 41  #BUT THE ANGLE WA
00001CC0: 53 20 54 4F 4F 20 47 52 45 41 54 20 41 4E 44 20  #S TOO GREAT AND
00001CD0: 54 48 45 20 53 48 49 50 20 54 49 50 50 45 44 20  #THE SHIP TIPPED
00001CE0: 4F 56 45 52 00 8F 00 00 00 00 00 00 01 00 25 00  #OVER..........%.
00001CF0: 94 22 50 9F 64 00 1C 02 00 80 53 4F 52 52 59 2C  #."P.d.....SORRY,
00001D00: 20 42 55 54 20 54 48 45 52 45 20 57 45 52 45 20  # BUT THERE WERE
00001D10: 00 52 00 00 00 00 00 00 01 00 17 00 B2 22 98 87  #.R..........2"..
00001D20: 4E 4F 50 87 20 53 55 52 56 49 56 4F 52 53 00 7E  #NOP. SURVIVORS.~
00001D30: 00 00 00 00 00 00 01 00 54 00 C2 22 00 F7 00 00  #........T.B".w..
00001D40: 50 9F 64 00 3A 02 00 80 42 55 54 20 54 48 45 20  #P.d.:...BUT THE
00001D50: 48 4F 52 49 5A 4F 4E 54 41 4C 20 56 45 4C 4F 43  #HORIZONTAL VELOC
00001D60: 49 54 59 20 57 41 53 20 54 4F 4F 20 47 52 45 41  #ITY WAS TOO GREA
00001D70: 54 2C 20 41 4E 44 20 59 4F 55 20 43 52 41 53 48  #T, AND YOU CRASH
00001D80: 45 44 20 41 4E 59 57 41 59 00 1D 00 00 00 00 00  #ED ANYWAY.......
00001D90: 00 01 00 44 00 10 23 00 E0 94 22 50 9F 64 00 3A  #...D..#.`."P.d.:
00001DA0: 02 00 80 42 55 54 20 54 48 45 20 54 45 52 52 41  #...BUT THE TERRA
00001DB0: 49 4E 20 49 53 20 54 4F 4F 20 52 4F 55 47 48 2C  #IN IS TOO ROUGH,
00001DC0: 20 41 4E 44 20 59 4F 55 20 54 49 50 50 45 44 20  # AND YOU TIPPED
00001DD0: 4F 56 45 52 00 D1 00 00 00 00 00 00 01 00 32 00  #OVER.Q........2.
00001DE0: 4E 23 00 E0 94 22 50 9F 64 00 3A 02 00 80 59 4F  #N#.`."P.d.:...YO
00001DF0: 55 20 4A 55 53 54 20 43 52 41 53 48 45 44 20 49  #U JUST CRASHED I
00001E00: 4E 54 4F 20 54 48 41 54 20 52 4F 43 4B 00 40 00  #NTO THAT ROCK.@.
00001E10: 00 00 00 00 00 01 00 41 00 7A 23 00 E0 94 22 50  #.......A.z#.`."P
00001E20: 9F 64 00 3A 02 00 80 59 4F 55 20 4A 55 53 54 20  #.d.:...YOU JUST
00001E30: 43 52 41 53 48 45 44 20 4F 4E 20 54 4F 50 20 4F  #CRASHED ON TOP O
00001E40: 46 20 41 4E 20 4F 4C 44 20 4C 55 4E 41 52 20 4D  #F AN OLD LUNAR M
00001E50: 4F 44 55 4C 45 00 44 00 00 00 00 00 00 01 00 4D  #ODULE.D........M
00001E60: 00 B4 23 00 E0 94 22 50 9F 32 00 03 00 00 80 59  #.4#.`."P.2.....Y
00001E70: 4F 55 20 48 41 56 45 20 4A 55 53 54 20 56 41 50  #OU HAVE JUST VAP
00001E80: 4F 52 49 5A 45 44 20 41 20 50 52 45 56 49 4F 55  #ORIZED A PREVIOU
00001E90: 53 4C 59 20 50 4C 41 4E 54 45 44 20 41 4D 45 52  #SLY PLANTED AMER
00001EA0: 49 43 41 4E 20 46 4C 41 47 00 C1 00 00 00 00 00  #ICAN FLAG.A.....
00001EB0: 00 01 00 4D 00 FA 23 00 F7 00 00 50 9F 64 00 3A  #...M.z#.w..P.d.:
00001EC0: 02 00 80 4E 49 43 45 20 57 4F 52 4B 2E 20 59 4F  #...NICE WORK. YO
00001ED0: 55 20 4A 55 53 54 20 43 52 41 53 48 45 44 20 49  #U JUST CRASHED I
00001EE0: 4E 54 4F 20 41 20 50 52 45 56 49 4F 55 53 4C 59  #NTO A PREVIOUSLY
00001EF0: 20 43 52 41 53 48 45 44 20 53 48 49 50 00 D7 00  # CRASHED SHIP.W.
00001F00: 00 00 00 00 00 01 00 43 00 40 24 00 E0 94 22 50  #.......C.@$.`."P
00001F10: 9F 0A 00 3A 02 00 80 57 65 6C 6C 2C 20 79 6F 75  #...:...Well, you
00001F20: 27 76 65 20 6A 75 73 74 20 64 65 73 74 72 6F 79  #'ve just destroy
00001F30: 65 64 20 74 68 65 20 6F 6E 6C 79 20 4D 61 63 44  #ed the only MacD
00001F40: 6F 6E 61 6C 64 27 73 00 95 00 00 00 00 00 00 01  #onald's.........
00001F50: 00 28 00 7C 24 50 9F 0A 00 1C 02 00 80 6F 6E 20  #.(.|$P.......on
00001F60: 74 68 65 20 6D 6F 6F 6E 2E 20 57 68 61 74 20 61  #the moon. What a
00001F70: 20 43 4C 4F 44 2E 00 B6 00 00 00 00 00 00 01 00  # CLOD..6........
00001F80: 3B 00 9E 24 00 F7 00 00 50 9F 96 00 02 00 00 80  #;..$.w..P.......
00001F90: B0 F0 54 57 4F 20 43 48 45 45 53 45 42 55 52 47  #0pTWO CHEESEBURG
00001FA0: 45 52 53 20 41 4E 44 20 41 20 42 49 47 20 4D 41  #ERS AND A BIG MA
00001FB0: 43 20 54 4F 20 47 4F 2E 00 A5 00 00 00 00 00 00  #C TO GO..%......
00001FC0: 01 00 51 00 D2 24 00 F7 00 00 50 9F 32 00 02 00  #..Q.R$.w..P.2...
00001FD0: 00 80 B0 F0 54 48 41 54 27 53 20 4F 4E 45 20 53  #..0pTHAT'S ONE S
00001FE0: 4D 41 4C 4C 20 53 54 45 50 20 46 4F 52 20 41 20  #MALL STEP FOR A
00001FF0: 4D 41 4E 2C 20 4F 4E 45 20 47 49 41 4E 54 20 4C  #MAN, ONE GIANT L
00002000: 45 41 50 20 46 4F 52 20 4D 41 4E 4B 49 4E 44 2E  #EAP FOR MANKIND.
00002010: 00 1E 00 00 00 00 00 00 01 00 64 01 1C 25 00 F7  #..........d..%.w
00002020: 00 00 F1 FF 10 98 A0 F0 B1 03 77 01 64 96 14 60  #..q... p1.w.d..`
00002030: 00 00 0C 40 08 00 0C 20 08 20 0C 40 08 20 00 F7  #...@... . .@. .w
00002040: 00 00 9C FF 10 98 A0 F0 B1 03 4A 01 64 96 28 60  #...... p1.J.d.(`
00002050: 00 00 00 E0 30 25 0F 00 10 98 A0 F0 BB 03 77 01  #...`0%.... p;.w.
00002060: 64 96 14 40 00 00 0C 60 08 00 0C 00 08 20 0C 60  #d..@...`..... .`
00002070: 08 20 00 F7 00 00 64 00 10 98 A0 F0 BB 03 4A 01  #. .w..d... p;.J.
00002080: 64 96 28 40 00 00 00 E0 64 25 F4 9D 80 F0 B3 03  #d.(@...`d%t..p3.
00002090: BC 02 60 90 00 40 FA 20 03 00 FA 00 00 40 FA 20  #<.`..@z ..z..@z
000020A0: 03 00 FA 00 00 40 FA 20 40 98 AC 03 00 00 00 90  #..z..@z @.,.....
000020B0: 2B 60 00 00 37 20 00 00 00 80 20 20 20 20 20 20  #+`..7 ....
000020C0: 25 20 00 F7 00 00 58 9E 5E 01 BC 02 00 80 46 55  #% .w..X.^.<...FU
000020D0: 45 4C 20 4C 4F 57 00 F7 00 00 D7 9D A0 F0 00 00  #EL LOW.w..W. p..
000020E0: 00 00 00 88 48 27 08 63 06 42 06 40 02 41 04 41  #....H'.c.B.@.A.A
000020F0: 06 43 04 40 02 42 00 41 42 41 42 43 42 42 46 41  #.C.@.B.ABABCBBFA
00002100: 46 43 44 40 44 41 44 61 48 61 00 F7 00 00 D4 9F  #FCD@DADaHa.w..T.
00002110: A0 F0 00 00 00 00 00 88 00 09 00 8E 1E 40 00 5B  # p...........@.[
00002120: 5E 40 00 7B D4 8A 00 06 08 40 00 43 48 40 80 29  #^@.{T....@.CH@.)
00002130: 00 90 49 20 00 00 00 F7 3C 26 00 88 00 0B D4 8F  #..I ...w<&....T.
00002140: C3 28 8A 40 89 40 86 40 85 40 84 40 82 40 84 40  #C(.@.@.@.@.@.@.@
00002150: 05 41 81 40 04 41 82 41 81 40 00 41 C1 40 C2 41  #.A.@.A.A.@.AA@BA
00002160: 44 41 C1 40 45 41 C4 40 C2 40 C4 40 C5 40 C6 40  #DAA@EAD@B@D@E@F@
00002170: C9 40 CA 40 83 28 00 F7 00 00 D0 9D 6D 00 00 00  #I@J@.(.w..P.m...
00002180: 00 00 00 01 00 64 01 7A 26 00 00 00 00 00 88 00  #.....d.z&.......
00002190: 01 04 61 44 62 04 02 05 40 80 40 81 40 01 40 81  #..aDb...@.@.@.@.
000021A0: 60 00 61 C1 60 41 40 C1 40 80 40 41 00 83 42 C5  #`.aA`A@A@.@A..BE
000021B0: 22 03 63 00 F7 00 00 D4 9A A0 F0 00 00 00 00 00  #".c.w..T. p.....
000021C0: 88 12 40 D7 95 10 40 00 00 00 40 08 20 10 60 00  #..@W..@...@. .`.
000021D0: 00 D6 94 00 00 03 00 10 40 00 00 00 00 02 00 10  #.V......@.......
000021E0: 60 00 00 00 F7 00 00 BC 02 80 F0 BC 02 54 8E 56  #`...w..<..p<.T.V
000021F0: 02 FA 00 5E 02 F2 08 5E 02 F2 14 5E 02 FA 1D 5E  #.z.^.r.^.r.^.z.^
00002200: 02 06 1D 5E 02 0E 14 5E 02 0E 08 5E 02 06 00 5E  #...^...^...^...^
00002210: 02 FA 00 56 02 EF 00 5E 02 EF F0 5E 02 11 F0 5E  #.z.V.o.^.op^..p^
00002220: 02 11 00 5E 02 EF 00 BC 02 54 8F 5E 02 E0 E8 56  #...^.o.<.T.^.`hV
00002230: 02 11 00 5E 02 20 E8 BC 02 D4 8C 56 02 EF F2 5E  #...^. h<.T.V.or^
00002240: 02 E4 EE 56 02 11 F2 5E 02 1C EE BC 02 54 8E 56  #.dnV..r^..n<.T.V
00002250: 02 24 E8 5E 02 1C E8 56 02 E4 E8 5E 02 DC E8 56  #.$h^..hV.dh^.\hV
00002260: 02 FD F0 5E 02 F9 EB 5E 02 07 EB 5E 02 03 F0 56  #.}p^.yk^..k^..pV
00002270: 02 00 00 C0 02 00 E2 E1 E0 DE DC DA D7 D4 D1 CE  #...@..ba`^\ZWTQN
00002280: CB C8 00 01 03 06 04 03 01 FE FA F9 FB FE 02 03  #KH.......~zy{~..
00002290: 05 06 02 01 FF FC FA FB FD 00 04 05 07 04 00 FF  #.....|z{}.......
000022A0: FD FF EC F0 F3 F6 F9 FC FE 00 02 04 07 0A 0D 10  #}.lpsvy|~.......
000022B0: 14 BC 02 80 F0 BC 02 00 00 56 02 FA EB 5E 02 00  #.<..p<...V.zk^..
000022C0: 00 5E 02 FB EB 5E 02 00 00 5E 02 FC EB 5E 02 00  #.^.{k^...^.|k^..
000022D0: 00 5E 02 FD EB 5E 02 00 00 5E 02 FE EB 5E 02 00  #.^.}k^...^.~k^..
000022E0: 00 5E 02 FF EB 5E 02 CF 00 00 00 00 00 00 01 00  #.^..k^.O........
000022F0: 64 01 D8 27 00 00 5E 02 00 EB 5E 02 00 00 5E 02  #d.X'..^..k^...^.
00002300: 01 EB 5E 02 00 00 5E 02 02 EB 5E 02 00 00 5E 02  #.k^...^..k^...^.
00002310: 03 EB 5E 02 00 00 5E 02 04 EB 5E 02 00 00 5E 02  #.k^...^..k^...^.
00002320: 05 EB 5E 02 00 00 5E 02 06 EB 56 02 00 00 C0 02  #.k^...^..kV...@.
00002330: 30 30 30 31 30 32 30 33 30 34 30 35 30 36 30 37  #0001020304050607
00002340: 30 38 30 39 31 30 31 31 31 32 31 33 31 34 31 35  #0809101112131415
00002350: 31 36 31 37 31 38 31 39 32 30 32 31 32 32 32 33  #1617181920212223
00002360: 32 34 32 35 32 36 32 37 32 38 32 39 33 30 33 31  #2425262728293031
00002370: 33 32 33 33 33 34 33 35 33 36 33 37 33 38 33 39  #3233343536373839
00002380: 34 30 34 31 34 32 34 33 34 34 34 35 34 36 34 37  #4041424344454647
00002390: 34 38 34 39 35 30 35 31 35 32 35 33 35 34 35 35  #4849505152535455
000023A0: 35 36 35 37 35 38 35 39 36 30 36 31 36 32 36 33  #5657585960616263
000023B0: 36 34 36 35 36 36 36 37 36 38 36 39 37 30 37 31  #6465666768697071
000023C0: 37 32 37 33 37 34 37 35 37 36 37 37 37 38 37 39  #7273747576777879
000023D0: 38 30 38 31 38 32 38 33 38 34 38 35 38 36 38 37  #8081828384858687
000023E0: 38 38 38 39 39 30 39 31 39 32 39 33 39 34 39 35  #8889909192939495
000023F0: 39 36 39 37 39 38 39 39 CE 02 CE 02 CE 02 EE 02  #96979899N.N.N.n.
00002400: EE 02 EE 02 0E 03 0E 03 0E 03 2E 03 0C 03 EE 02  #n.n...........n.
00002410: BF 02 8A 02 6C 02 3F 02 F4 01 A4 01 90 01 5E 01  #?...l.?.t.$...^.
00002420: 0E 01 F6 00 C8 00 B4 00 6B 00 18 00 36 00 35 00  #..v.H.4.k...6.5.
00002430: 33 00 52 00 50 00 4E 00 6D 00 6B 00 4A 00 48 00  #3.R.P.N.m.k.J.H.
00002440: 46 00 45 00 63 00 62 00 80 00 7E 00 7D 00 7B 00  #F.E.c.b...~.}.{.
00002450: 7A 00 36 00 00 00 00 00 00 01 00 64 01 36 29 98  #z.6........d.6).
00002460: 00 B6 00 AE 00 A6 00 9E 00 B5 00 AD 00 A5 00 9C  #.6...&...5.-.%..
00002470: 00 94 00 8C 00 83 00 5B 00 53 00 4B 00 B6 00 A1  #.......[.S.K.6.!
00002480: 00 EC 00 78 00 43 00 AE 00 99 00 C5 00 50 00 3B  #.l.x.C.....E.P.;
00002490: 00 62 03 7B 00 C9 00 FA 00 13 01 2C 01 77 01 9A  #.b.{.I.z...,.w..
000024A0: 01 43 02 9E 03 28 07 73 07 DD 07 47 08 D1 08 5B  #.C...(.s.].G.Q.[
000024B0: 09 E5 09 4F 0A B9 0A 43 0B CD 0B 38 0C 9A 0C 7D  #.e.O.9.C.M.8...}
000024C0: 0C 9F 0C 62 0C 44 0C 47 0C 49 0C 6C 0C AE 0C 51  #...b.D.G.I.l...Q
000024D0: 0C B4 0C D6 0C 19 0D 3B 0D 3E 0D 60 0D C3 0D 65  #.4.V...;.>.`.C.e
000024E0: 0D 70 0D 7A 0D 65 0D 4F 0D 3A 0D 44 0D 2F 0D F9  #.p.z.e.O.:.D./.y
000024F0: 0C E4 0C AE 0C B9 0C 83 0C 6E 0C 38 0C 23 0C 0D  #.d...9...n.8.#..
00002500: 0C F8 0B 02 0C 0D 0C 17 0C 7B 0C 5F 0C 43 0C A7  #.x.......{._.C.'
00002510: 0C 8A 0B 6E 0B 12 0B 96 0A BA 0A 9E 0A C1 0A A5  #...n.....:...A.%
00002520: 09 09 0A CD 09 D1 09 14 0A 18 0A 3C 0A A0 0A 04  #...M.Q.....<. ..
00002530: 0B 67 0B 2B 0B 8F 0A B3 0A 97 0B 5A 0B 3E 0C E2  #.g.+...3...Z.>.b
00002540: 0B 66 0B 2A 0B C7 0A A4 09 C2 09 5F 09 5C 09 3A  #.f.*.G.$.B._.\.:
00002550: 09 17 08 94 07 B2 07 0F 08 EC 07 8A 07 07 07 84  #.....2...l......
00002560: 06 E2 06 1F 07 7C 06 5A 06 F7 05 94 05 72 06 4F  #.b...|.Z.w...r.O
00002570: 06 8C 06 EA 06 A7 06 64 06 82 06 3F 06 9C 05 7A  #...j.'.d...?...z
00002580: 05 97 05 34 05 92 05 6F 06 EC 05 4A 05 27 05 64  #...4...o.l.J.'.d
00002590: 05 42 04 1F 04 7C 04 7A 04 B7 04 94 04 B2 04 2F  #.B...|.z.7...2./
000025A0: 04 0C 03 EA 02 C7 02 A5 03 44 04 44 05 03 06 23  #...j.G.%.D.D...#
000025B0: 07 42 07 42 08 21 09 C1 09 40 0A C0 0A 65 00 00  #.B.B.!.A.@.@.e..
000025C0: 00 00 00 00 01 00 64 01 94 2A 1F 0B BF 0B 5E 0C  #......d..*..?.^.
000025D0: 5E 0D FD 0D 7D 0E 7C 0F 1C 10 DB 10 5B 11 61 11  #^.}.}.|...[.[.a.
000025E0: 67 12 ED 12 13 13 79 13 BF 13 C5 13 8C 14 F2 14  #g.m...y.?.E...r.
000025F0: D8 15 5E 16 E4 16 6A 17 10 18 73 18 16 19 B8 19  #X.^.d.j...s...8.
00002600: 3B 1A DD 1A 60 1B E2 1B 85 1C 08 1D 8A 1D 0D 1E  #;.].`.b.........
00002610: 6F 1E 12 1F 94 1F 17 20 99 20 D0 20 E6 21 3D 22  #o...... . P f!="
00002620: 93 22 49 23 C0 23 96 24 AC 24 43 25 D9 26 CF 27  #."I#@#.$,$C%Y&O'
00002630: 89 28 A3 29 1D 2A 37 2B 30 2C 6A 2C 84 2C 7E 2D  #.(#).*7+0,j,.,~-
00002640: F8 2D D1 2E AB 2F 45 30 3F 31 39 32 A5 32 11 33  #x-Q.+/E0?192%2.3
00002650: 3D 33 89 33 B6 33 E2 33 2E 34 7A 34 A6 34 F3 34  #=3.363b3.4z4&4s4
00002660: 3F 35 8B 35 F7 35 63 36 8F 36 CD 36 AC 36 29 37  #?5.5w5c6.6M6,6)7
00002670: 87 37 E4 37 82 38 FF 38 9D 39 1A 3A 98 3A A2 39  #.7d7.8.8.9.:.:"9
00002680: 0C 39 F6 37 60 37 0A 37 74 36 5E 35 A8 34 52 34  #.9v7`7.7t6^5(4R4
00002690: 3C 33 04 31 CD 2C 56 29 1F 27 C8 23 00 23 F8 21  #<3.1M,V).'H#.#x!
000026A0: F0 20 08 20 00 1F 5A 1E 55 1D CF 1C 2A 1C A4 1B  #p . ..Z.U.O.*.$.
000026B0: FF 1A 59 1A F4 19 4E 19 29 18 83 17 7E 16 18 16  #..Y.t.N.)...~...
000026C0: 73 15 8D 14 E8 13 03 13 68 12 2E 11 B4 0F FA 0E  #s...h...h...4.z.
000026D0: E0 0C C6 0B 6C 0A 52 0A 78 09 5E 09 A4 08 8A 08  #`.F.l.R.x.^.$...
000026E0: 30 07 24 07 D8 07 6C 08 E0 08 34 09 08 09 DC 08  #0.$.X.l.`.4...\.
000026F0: B0 08 E4 08 18 09 6D 09 CF 09 71 0A 14 0B B6 0B  #0.d...m.O.q...6.
00002700: 39 0C 9B 0C 3E 0D A0 0D 02 0E 85 0E 27 0F AA 0F  #9...>. .....'.*.
00002710: 0C 10 8F 10 11 11 B3 11 56 12 D8 12 7B 13 FD 13  #......3.V.X.{.}.
00002720: 60 14 C2 14 45 15 A7 15 E5 00 00 00 00 00 00 01  #`.B.E.'.e.......
00002730: 00 64 01 F2 2B 49 16 AC 16 4E 17 D1 17 33 18 96  #.d.r+I.,.N.Q.3..
00002740: 18 F8 18 7A 19 1D 1A 9F 1A 02 1B A4 1B 47 1C A9  #.x.z.......$.G.)
00002750: 1C 0C 1D A2 1B 99 1A 30 19 E6 17 FD 15 94 14 8A  #..."...0.f.}....
00002760: 13 A1 12 B7 11 AE 10 45 0F BB 0D 24 0C EC 0A 55  #.!.7...E.;.$.l.U
00002770: 09 1D 08 E5 05 2E 05 36 04 7F 03 47 02 90 00 40  #...e...6...G...@
00002780: 00 F1 FF 81 FF 8E FF 9B FF C8 FF F6 FF 02 00 F1  #.q.......H.v...q
00002790: FF 1D 00 4A 00 38 00 45 00 52 00 5F 00 58 00 31  #...J.8.E.R._.X.1
000027A0: 00 49 00 22 00 3B 00 33 00 2C 00 44 00 1D 00 16  #.I.".;.3.,.D....
000027B0: 00 0E 00 07 00 20 00 55 00 AB 00 00 01 36 01 CC  #..... .U.+...6.L
000027C0: 01 2B 02 6A 02 A9 02 08 03 47 03 66 03 A5 03 E4  #.+.j.)...G.f.%.d
000027D0: 03 44 04 A3 04 E2 04 41 05 80 05 BF 05 1E 06 3D  #.D.#.b.A...?...=
000027E0: 06 5C 06 9C 06 DB 06 1A 07 59 07 98 07 D7 07 16  #.\...[...Y...W..
000027F0: 08 55 08 94 08 D4 08 13 09 52 09 B1 09 F0 09 2F  #.U...T...R.1.p./
00002800: 0A 6E 0A AD 0A CC 0A 0C 0B 2B 0B 6A 0B A9 0B 08  #.n.-.L...+.j.)..
00002810: 0C 27 0C 66 0C A5 0C E4 0C 44 0D 1C 0D B4 0C 8D  #.'.f.%.d.D...4..
00002820: 0B 65 0C 3E 0D 76 0D 4E 0D E7 0C 9F 0C 78 0D 46  #.e.>.v.N.g...x.F
00002830: 0C 54 0C 23 0C 11 0C 80 0B 13 0B A6 0A 39 0A AC  #.T.#.......&.9.,
00002840: 09 3F 09 45 09 6A 09 CF 09 14 0A 3A 0A 9F 0A A4  #.?.E.j.O...:...$
00002850: 0A E9 0A EE 0A D4 0A B4 0A 95 0A 55 0A 36 0A 16  #.i.n.T.4...U.6..
00002860: 0A F7 09 D7 09 B8 09 98 09 79 09 59 09 3A 09 3A  #.w.W.8...y.Y.:.:
00002870: 09 3B 09 3B 09 F8 08 B4 08 B0 08 2C 08 69 08 F0  #.;.;.x.4.0.,.i.p
00002880: 07 18 08 60 08 C8 08 50 08 B8 07 E0 07 48 07 90  #...`.H.P.8.`.H..
00002890: 07 78 07 7A 00 00 00 00 00 00 01 00 64 01 50 2D  #.x.z........d.P-
000028A0: 40 07 A8 06 50 06 D8 05 40 05 28 05 D0 04 B8 03  #@.(.P.X.@.(.P.8.
000028B0: E0 03 08 04 70 04 18 04 20 04 C8 03 B0 02 18 03  #`...p... .H.0...
000028C0: 00 03 E8 02 70 02 57 01 2A 01 BC 01 EE 01 21 01  #..h.p.W.*.<.n.!.
000028D0: D3 00 A5 00 98 00 4A 00 9C 00 AE 00 E1 00 93 00  #S.%...J.....a...
000028E0: C6 FF D9 FF AB FF 5D FF 8F 00 C1 01 F3 02 A6 03  #F.Y.+.]...A.s.&.
000028F0: 13 03 A1 02 8E 02 7C 02 6A 02 D7 01 45 01 93 01  #..!...|.j.W.E...
00002900: A0 01 0E 01 DC 00 69 00 57 00 65 00 F3 FF C1 FF  # ...\.i.W.e.s.A.
00002910: 2E 00 9B 00 89 00 B7 00 A4 00 32 00 1F 00 0D 00  #......7.$.2.....
00002920: 7B 00 68 00 D6 00 E4 00 11 01 1F 01 8D 00 FA 00  #{.h.V.d.......z.
00002930: 48 01 16 01 43 01 91 01 FF 01 6C 01 DA 00 C8 00  #H...C.....l.Z.H.
00002940: D4 00 A0 00 AC 00 98 00 84 00 70 00 5C 00 48 00  #T. .,.....p.\.H.
00002950: 34 00 3F 00 5E 00 5D 00 5C 00 5A 00 59 00 78 00  #4.?.^.].\.Z.Y.x.
00002960: 77 00 75 00 74 00 53 00 51 00 50 00 4F 00 6E 00  #w.u.t.S.Q.P.O.n.
00002970: 6C 00 8B 00 8A 00 88 00 67 00 86 00 A5 00 83 00  #l.......g...%...
00002980: A2 00 C1 00 9F 00 B9 00 B3 00 8C 00 66 00 80 00  #".A...9.3...f...
00002990: 99 00 93 00 AC 00 86 00 A0 00 99 00 B3 00 CC 00  #....,... ...3.L.
000029A0: A6 00 A0 00 99 00 B3 00 AC 00 A6 00 A0 00 99 00  #&. ...3.,.&. ...
000029B0: 7D 00 6E 00 5F 00 50 00 55 00 52 00 57 00 54 00  #}.n._.P.U.R.W.T.
000029C0: 51 00 56 00 67 00 7D 00 8C 00 98 00 87 00 82 00  #Q.V.g.}.........
000029D0: 7D 00 58 00 53 00 2E 00 28 00 23 00 1E 00 19 00  #}.X.S...(.#.....
000029E0: 34 00 2F 00 2A 00 25 00 3F 00 3D 00 5A 00 78 00  #4./.*.%.?.=.Z.x.
000029F0: 75 00 73 00 70 00 4E 00 4B 00 68 00 46 00 0F 00  #u.s.p.N.K.h.F...
00002A00: 00 00 00 00 00 01 00 64 01 AE 2E 43 00 21 00 3E  #.......d...C.!.>
00002A10: 00 3C 00 39 00 37 00 34 00 51 00 6F 00 8C 00 8A  #.<.9.7.4.Q.o....
00002A20: 00 A7 00 A5 00 A2 00 C0 00 89 00 B3 00 7C 00 E6  #.'.%.".@...3.|.f
00002A30: 00 50 01 79 01 83 01 2D 01 96 01 20 01 AA 01 D3  #.P.y...-... .*.S
00002A40: 01 1D 02 A7 01 10 02 FA 01 C4 01 6D 01 37 01 C1  #...'...z.D.m.7.A
00002A50: 00 2A 01 B4 01 FE 01 E7 01 D1 01 5B 01 04 01 4E  #.*.4.~.g.Q.[...N
00002A60: 01 B8 01 A1 01 AB 01 55 01 3E 01 E8 00 92 00 1B  #.8.!.+.U.>.h....
00002A70: 00 45 00 AF 00 18 01 02 01 CC 00 15 01 9F 01 A9  #.E./.....L.....)
00002A80: 01 D2 01 9C 01 46 01 CF 01 19 02 23 02 1E 03 99  #.R...F.O...#....
00002A90: 02 94 02 50 02 2B 02 86 02 02 02 FD 02 F8 02 F3  #...P.+.....}.x.s
00002AA0: 02 6B 03 E2 03 39 04 70 04 C7 04 1F 05 76 05 CD  #.k.b.9.p.G...v.M
00002AB0: 05 44 06 9B 06 D3 06 2A 07 81 07 B8 07 2F 08 87  #.D...S.*...8./..
00002AC0: 08 BE 08 35 09 6C 09 A4 09 76 09 88 09 DB 09 4D  #.>.5.l.$.v...[.M
00002AD0: 09 20 09 4C 09 58 09 64 09 91 09 BD 09 C9 09 D5  #. .L.X.d...=.I.U
00002AE0: 09 02 0A 0E 0A 1A 0A 26 0A 53 0A 7F 0A AB 0A D7  #.......&.S...+.W
00002AF0: 0A 21 0B 4A 0B 74 0B DD 0A C7 0A 10 0B FA 0A E3  #.!.J.t.].G...z.c
00002B00: 0A CD 09 B6 0A C0 0A 09 0B 53 0B 1C 0B 06 0B EF  #.M.6.@...S.....o
00002B10: 09 D9 0A 42 0B 4C 0B D5 0A 3F 0B E8 0A D2 0A BB  #.Y.B.L.U.?.h.R.;
00002B20: 0A 45 0A 6E 0A 57 09 C1 09 0A 0A F4 0A DD 0A 07  #.E.n.W.A...t.]..
00002B30: 0B F0 09 BA 09 83 09 AD 09 D6 09 40 09 09 09 73  #.p.:...-.V.@...s
00002B40: 09 DC 09 C6 09 AF 09 F9 09 02 0A EC 09 F5 09 1F  #.\.F./.y...l.u..
00002B50: 0A 68 0A 52 09 97 09 7C 09 61 08 47 08 6C 08 D1  #.h.R...|.a.G.l.Q
00002B60: 07 B6 07 9C 08 61 08 46 09 3E 00 00 00 00 00 00  #.6...a.F.>......
00002B70: 01 00 64 01 0C 30 AC 09 00 50 00 00 00 05 00 00  #..d..0,..P......
00002B80: 00 00 00 00 00 00 00 00 00 00 05 55 00 00 00 00  #...........U....
00002B90: 00 00 00 05 00 00 55 05 55 00 00 05 05 55 50 05  #......U.U....UP.
00002BA0: 55 55 50 05 05 00 05 50 05 00 05 00 00 55 00 05  #UUP....P.....U..
00002BB0: 55 05 55 05 05 50 55 55 55 05 55 55 55 05 55 05  #U.U..PUUU.UUU.U.
00002BC0: 55 55 55 05 05 55 55 50 55 55 55 50 00 55 05 05  #UUU..UUPUUUP.U..
00002BD0: 55 55 55 55 05 55 05 50 05 55 55 55 55 55 55 55  #UUUU.U.P.UUUUUUU
00002BE0: 05 00 50 00 00 00 05 00 00 00 00 50 55 00 00 00  #..P........PU...
00002BF0: 00 00 05 00 00 05 00 05 00 00 05 05 00 00 00 00  #................
00002C00: 00 00 05 00 00 00 00 00 00 50 00 00 00 00 00 00  #.........P......
00002C10: 00 00 00 55 55 50 55 05 55 05 55 55 55 55 55 55  #...UUPU.U.UUUUUU
00002C20: 55 55 55 55 55 55 55 55 55 55 55 55 00 00 00 50  #UUUUUUUUUUUU...P
00002C30: 55 55 00 00 00 00 00 00 00 00 05 00 05 00 05 00  #UU..............
00002C40: 00 00 00 00 00 55 55 55 55 55 55 55 55 50 55 55  #.....UUUUUUUUPUU
00002C50: 05 05 00 00 05 00 50 50 00 00 00 00 00 00 50 50  #......PP......PP
00002C60: 05 00 50 05 05 55 05 00 05 05 05 00 05 00 00 00  #..P..U..........
00002C70: 00 00 00 50 00 00 00 55 55 55 55 55 55 55 50 50  #...P...UUUUUUUPP
00002C80: 55 55 55 55 50 55 55 55 05 55 50 50 55 55 55 55  #UUUUPUUU.UPPUUUU
00002C90: 55 55 50 55 55 55 55 55 55 55 55 50 55 05 05 05  #UUPUUUUUUUUPU...
00002CA0: 50 50 55 55 55 55 55 55 55 00 00 00 00 00 00 00  #PPUUUUUUU.......
00002CB0: 00 50 50 00 00 00 00 00 00 50 00 00 00 00 00 05  #.PP......P......
00002CC0: 00 00 05 00 00 00 00 00 00 00 00 50 55 00 00 00  #...........PU...
00002CD0: 00 00 00 00 66 00 00 00 00 00 00 01 00 77 00 6A  #....f........w.j
00002CE0: 31 00 55 05 00 00 00 00 00 76 06 50 55 50 00 50  #1.U......v.PUP.P
00002CF0: 50 00 55 55 00 05 00 50 05 55 50 00 50 50 05 00  #P.UU...P.UP.PP..
00002D00: 00 50 05 00 50 50 00 00 50 00 55 50 00 50 00 50  #.P..PP..P.UP.P.P
00002D10: 00 00 05 00 00 00 00 00 00 00 00 00 00 05 00 00  #................
00002D20: 00 00 00 00 00 00 00 50 05 00 00 00 55 00 50 05  #.......P....U.P.
00002D30: 00 00 50 55 55 50 55 55 55 55 55 55 50 50 55 05  #..PUUPUUUUUUPPU.
00002D40: 55 55 55 55 00 55 55 55 55 55 55 55 55 55 55 55  #UUUU.UUUUUUUUUUU
00002D50: 55 05 0A 00 00 00 00 00 00 01 00 64 01 DC 31 00  #U..........d.\1.
00002D60: 00 1E 01 3C 02 59 03 77 04 94 05 B1 06 CD 07 E8  #...<.Y.w...1.M.h
00002D70: 08 03 0A 1D 0B 36 0C 4E 0D 66 0E 7C 0F 90 10 A4  #.....6.N.f.|...$
00002D80: 11 B6 12 C7 13 D6 14 E4 15 F0 16 FA 17 02 19 08  #.6.G.V.d.p.z....
00002D90: 1A 0C 1B 0E 1C 0E 1D 0C 1E 07 1F 00 20 F6 20 EA  #............ v j
00002DA0: 21 DB 22 CA 23 B5 24 9E 25 84 26 67 27 47 28 23  #!["J#5$.%.&g'G(#
00002DB0: 29 FD 29 D3 2A A6 2B 75 2C 41 2D 0A 2E CE 2E 90  #)})S*&+u,A-..N..
00002DC0: 2F 4D 30 07 31 BD 31 6F 32 1D 33 C7 33 6D 34 0F  #/M0.1=1o2.3G3m4.
00002DD0: 35 AD 35 46 36 DC 36 6D 37 FA 37 82 38 06 39 86  #5-5F6\6m7z7.8.9.
00002DE0: 39 01 3A 78 3A EA 3A 57 3B C0 3B 24 3C 83 3C DE  #9.:x:j:W;@;$<.<^
00002DF0: 3C 34 3D 85 3D D2 3D 19 3E 5C 3E 9A 3E D3 3E 07  #<4=.=R=.>\>.>S>.
00002E00: 3F 36 3F 61 3F 86 3F A6 3F C2 3F D8 3F EA 3F F6  #?6?a?.?&?B?X?j?v
00002E10: 3F FE 3F 00 40 FE 3F F6 3F EA 3F D8 3F C2 3F A6  #?~?.@~?v?j?X?B?&
00002E20: 3F 86 3F 61 3F 36 3F 07 3F D3 3E 9A 3E 5C 3E 19  #?.?a?6?.?S>.>\>.
00002E30: 3E D2 3D 85 3D 34 3D DE 3C 83 3C 24 3C C0 3B 57  #>R=.=4=^<.<$<@;W
00002E40: 3B EA 3A 78 3A 01 3A 86 39 06 39 82 38 FA 37 6D  #;j:x:.:.9.9.8z7m
00002E50: 37 DC 36 46 36 AD 35 0F 35 6D 34 C7 33 1D 33 6F  #7\6F6-5.5m4G3.3o
00002E60: 32 BD 31 07 31 4D 30 90 2F CE 2E 0A 2E 41 2D 75  #2=1.1M0./N...A-u
00002E70: 2C A6 2B D3 2A FD 29 23 29 47 28 67 27 84 26 9E  #,&+S*})#)G(g'.&.
00002E80: 25 B5 24 CA 23 DB 22 EA 21 F6 20 00 20 07 1F 0C  #%5$J#["j!v . ...
00002E90: 1E 0E 1D 0E 1C 0C 1B 08 1A 02 19 FA 17 F0 16 E4  #...........z.p.d
00002EA0: 15 D6 14 C7 13 B6 12 A4 11 90 10 7C 0F 66 0E 4E  #.V.G.6.$...|.f.N
00002EB0: 0D 36 0C 1D 0B 03 0A E8 08 CD 07 B1 06 5E 00 00  #.6.....h.M.1.^..
00002EC0: 00 00 00 00 01 00 64 01 3A 33 94 05 77 04 59 03  #......d.:3..w.Y.
00002ED0: 3C 02 1E 01 00 00 E4 FE C6 FD A7 FC 8B FB 6E FA  #<.....d~F}'|.{nz
00002EE0: 51 F9 35 F8 18 F7 FD F5 E3 F4 CA F3 B2 F2 9C F1  #Qy5x.w}uctJs2r.q
00002EF0: 86 F0 70 EF 5C EE 4A ED 3B EC 2A EB 1E EA 11 E9  #.ppo\nJm;l*k.j.i
00002F00: 08 E8 00 E7 FA E5 F4 E4 F2 E3 F2 E2 F6 E1 F9 E0  #.h.gzetdrcrbvay`
00002F10: 02 E0 0A DF 16 DE 25 DD 38 DC 4B DB 62 DA 7C D9  #.`._.^%]8\K[bZ|Y
00002F20: 9B D8 BB D7 DD D6 05 D6 2D D5 5C D4 8B D3 BF D2  #.X;W]V.V-U\T.S?R
00002F30: F8 D1 32 D1 72 D0 B3 CF FB CE 45 CE 93 CD E5 CC  #xQ2QrP3O{NEN.MeL
00002F40: 3B CC 95 CB F3 CA 55 CA BA C9 26 C9 95 C8 08 C8  #;L.KsJUJ:I&I.H.H
00002F50: 7E C7 FA C6 7C C6 01 C6 8A C5 18 C5 AB C4 42 C4  #~GzF|F.F.E.E+DBD
00002F60: DE C3 7D C3 22 C3 CC C2 7B C2 30 C2 E7 C1 A4 C1  #^C}C"CLB{B0BgA$A
00002F70: 68 C1 2F C1 F9 C0 CA C0 A1 C0 7C C0 5A C0 40 C0  #hA/Ay@J@!@|@Z@@@
00002F80: 28 C0 18 C0 0A C0 04 C0 00 C0 04 C0 0A C0 18 C0  #(@.@.@.@.@.@.@.@
00002F90: 28 C0 40 C0 5A C0 7C C0 A1 C0 CA C0 F9 C0 2F C1  #(@@@Z@|@!@J@y@/A
00002FA0: 68 C1 A4 C1 E7 C1 30 C2 7B C2 CC C2 22 C3 7D C3  #hA$AgA0B{BLB"C}C
00002FB0: DE C3 42 C4 AB C4 18 C5 8A C5 01 C6 7C C6 FA C6  #^CBD+D.E.E.F|FzF
00002FC0: 7E C7 08 C8 95 C8 26 C9 BA C9 55 CA F3 CA 95 CB  #~G.H.H&I:IUJsJ.K
00002FD0: 3B CC E5 CC 93 CD 45 CE FB CE B3 CF 72 D0 33 D1  #;LeL.MEN{N3OrP3Q
00002FE0: F8 D1 BF D2 8B D3 5C D4 2D D5 05 D6 DD D6 BB D7  #xQ?R.S\T-U.V]V;W
00002FF0: 9B D8 7C D9 62 DA 4B DB 38 DC 25 DD 16 DE 0A DF  #.X|YbZK[8\%].^._
00003000: 00 E0 F9 E0 F6 E1 F2 E2 F2 E3 F4 E4 FA E5 00 E7  #.`y`varbrctdze.g
00003010: 08 E8 12 E9 1E EA 2A EB 3B EC 4A ED 5C EE 70 EF  #.h.i.j*k;lJm\npo
00003020: 86 F0 9C F1 B2 F2 CA F3 CB 00 00 00 00 00 00 01  #.p.q2rJsK.......
00003030: 00 4E 00 98 34 E3 F4 FD F5 18 F7 35 F8 51 F9 6E  #.N..4ct}u.w5xQyn
00003040: FA 8B FB A7 FC C6 FD E4 FE E0 34 34 35 50 98 80  #z.{'|F}d~`445P..
00003050: F0 00 00 00 00 00 F7 00 00 00 F7 00 00 00 F7 00  #p.....w...w...w.
00003060: 00 00 F7 00 00 00 F7 00 00 00 F7 00 00 00 F7 00  #..w...w...w...w.
00003070: 00 00 F7 00 00 00 F7 00 00 00 F7 00 00 E5 00 00  #..w...w...w..e..
00003080: 00 00 00 00 01 00 64 01 E0 34 05 00 05 00 05 00  #......d.`4......
00003090: 05 00 05 00 C6 15 FE 3F 1F 0A FE FF DF 15 46 35  #....F.~?..~._.F5
000030A0: 04 00 1F 0A 06 00 DF 0B 8C FB 1F 0A 8E FB DF 09  #......_..{...{_.
000030B0: 5A 35 DF 15 83 00 8E FB DF 09 5A 35 DF 15 83 00  #Z5_....{_.Z5_...
000030C0: 8E FB DF 09 5A 35 DF 15 4B 00 8E FB DF 09 5A 35  #.{_.Z5_.K..{_.Z5
000030D0: DF 15 2F 00 8E FB DF 09 5A 35 DF 15 46 00 8E FB  #_./..{_.Z5_.F..{
000030E0: DF 09 5A 35 DF 15 0D 00 8E FB DF 09 5A 35 01 01  #_.Z5_....{_.Z5..
000030F0: 96 25 DF 15 40 00 66 FF DF 15 62 35 00 F4 DF 09  #.%_.@.f._.b5.t_.
00003100: AA 13 1E 00 DF 8B 8C FB FD 80 87 00 D0 9D 00 00  #*..._..{}...P...
00003110: 8A 02 A0 F0 00 80 20 20 20 20 20 20 20 20 20 20  #.. p..
00003120: 20 20 20 20 20 20 47 20 20 52 20 20 45 20 20 45  #      G  R  E  E
00003130: 20 20 54 20 20 49 20 20 4E 20 20 47 20 20 53 0D  #  T  I  N  G  S.
00003140: 0A 20 20 20 20 20 20 20 20 20 20 20 20 20 20 20  #.
00003150: 20 2D 20 20 2D 20 20 2D 20 20 2D 20 20 2D 20 20  # -  -  -  -  -
00003160: 2D 20 20 2D 20 20 2D 20 20 2D 0D 0A 0A 0A 20 20  #-  -  -  -....
00003170: 20 20 20 42 45 43 41 55 53 45 20 4F 46 20 43 48  #   BECAUSE OF CH
00003180: 41 4E 47 45 53 20 49 4E 20 54 48 45 20 4E 41 54  #ANGES IN THE NAT
00003190: 49 4F 4E 41 4C 20 50 4F 4C 49 43 59 2C 20 54 48  #IONAL POLICY, TH
000031A0: 45 20 4C 55 4E 41 52 0D 0A 45 58 50 4C 4F 52 41  #E LUNAR..EXPLORA
000031B0: 54 4F 52 59 20 50 52 4F 47 52 41 4D 20 49 53 20  #TORY PROGRAM IS
000031C0: 4F 4E 43 45 20 41 47 41 49 4E 20 46 55 4E 43 54  #ONCE AGAIN FUNCT
000031D0: 49 4F 4E 49 4E 47 2C 20 41 4E 44 20 59 4F 55 20  #IONING, AND YOU
000031E0: 48 41 56 45 0D 0A 42 45 45 00 00 00 00 00 00 01  #HAVE..BEE.......
000031F0: 00 5E 00 3E 36 45 4E 20 43 48 4F 53 45 4E 20 41  #.^.>6EN CHOSEN A
00003200: 53 20 54 48 45 20 50 49 4C 4F 54 20 49 4E 20 43  #S THE PILOT IN C
00003210: 4F 4D 4D 41 4E 44 20 4F 46 20 54 48 45 20 4C 55  #OMMAND OF THE LU
00003220: 4E 41 52 20 4D 4F 44 55 4C 45 2E 0D 0A 20 20 20  #NAR MODULE...
00003230: 20 20 59 4F 55 20 4D 49 53 53 49 4F 4E 20 49 53  #  YOU MISSION IS
00003240: 20 54 4F 20 4C 41 4E 44 20 4F 4E 20 00 8A 00 00  # TO LAND ON ....
00003250: 00 00 00 00 01 00 18 00 96 36 18 80 B0 F0 4D 61  #.........6..0pMa
00003260: 72 65 20 41 73 73 61 62 65 74 2C 00 4F 00 00 00  #re Assabet,.O...
00003270: 00 00 00 01 00 3A 01 A8 36 10 80 A0 F0 20 41 20  #.....:.(6.. p A
00003280: 42 41 52 52 45 4E 20 41 4E 44 0D 0A 48 41 52 53  #BARREN AND..HARS
00003290: 48 20 57 41 53 54 45 4C 41 4E 44 20 42 45 4C 49  #H WASTELAND BELI
000032A0: 45 56 45 44 20 49 4E 43 41 50 41 42 4C 45 20 4F  #EVED INCAPABLE O
000032B0: 46 20 53 55 50 50 4F 52 54 49 4E 47 20 41 4E 59  #F SUPPORTING ANY
000032C0: 20 4B 4E 4F 57 4E 20 46 4F 52 4D 20 4F 46 20 4C  # KNOWN FORM OF L
000032D0: 49 46 45 2E 0D 0A 0D 0A 20 20 20 20 20 55 4E 46  #IFE.....     UNF
000032E0: 4F 52 54 55 4E 41 54 45 4C 59 2C 20 54 48 45 20  #ORTUNATELY, THE
000032F0: 53 50 41 43 45 20 41 47 45 4E 43 59 20 43 48 4F  #SPACE AGENCY CHO
00003300: 53 45 20 54 4F 20 55 53 45 20 41 4E 20 49 4E 46  #SE TO USE AN INF
00003310: 45 52 49 4F 52 0D 0A 42 52 41 4E 44 20 4F 46 20  #ERIOR..BRAND OF
00003320: 4D 49 4E 49 43 4F 4D 50 55 54 45 52 20 49 4E 20  #MINICOMPUTER IN
00003330: 54 48 45 20 4C 55 4E 41 52 20 4C 41 4E 44 45 52  #THE LUNAR LANDER
00003340: 2C 20 41 4E 44 20 49 54 20 48 41 53 0D 0A 4A 55  #, AND IT HAS..JU
00003350: 53 54 20 44 49 45 44 20 4F 4E 20 59 4F 55 2E 20  #ST DIED ON YOU.
00003360: 59 4F 55 20 4D 55 53 54 20 4E 4F 57 20 4C 41 4E  #YOU MUST NOW LAN
00003370: 44 20 55 4E 44 45 52 20 4D 41 4E 55 41 4C 20 43  #D UNDER MANUAL C
00003380: 4F 4E 54 52 4F 4C 2E 0D 0A 0D 0A 20 20 20 20 20  #ONTROL.....
00003390: 20 20 20 20 20 2A 2A 2A 2A 2A 20 20 47 4F 4F 44  #     *****  GOOD
000033A0: 20 4C 55 43 4B 20 20 2A 2A 2A 2A 2A 00 E0 00 00  # LUCK  *****.`..
000033B0: 00 00 00 00 01 00 0A 00 DC 37 00 E0 62 35 6B 00  #........\7.`b5k.
000033C0: 00 00 00 00 00 01 00 06 00 E0 34 E5 00 00 00 00  #.........`4e....
000033D0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000033E0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000033F0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003400: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003410: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003420: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003430: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003440: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003450: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003460: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003470: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003480: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003490: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034A0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034B0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034C0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034D0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034E0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000034F0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003500: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003510: 00 00 00 00 00 00 00 00 FF 00 00 00 00 00 00 00  #................
00003520: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003530: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003540: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003550: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003560: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003570: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003580: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
00003590: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035A0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035B0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035C0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035D0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035E0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
000035F0: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  #................
fubar
if test $? -ne 0; then no_result; fi

srec_cat test.in -ppb -o test.out -hexdump
if test $? -ne 0; then fail; fi

diff test.ok test.out
if test $? -ne 0; then fail; fi



#
# The things tested here, worked.
# No other guarantees are made.
#
pass

# vim: set ts=8 sw=4 et :
