! This file is part of tblite.
! SPDX-Identifier: LGPL-3.0-or-later
!
! tblite is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! tblite is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with tblite.  If not, see <https://www.gnu.org/licenses/>.

!> @dir tblite/ceh/ceh.f90
!> Contains the specification of the Charge Extended Hückel (CEH) method.

module tblite_ceh_ceh
   use mctc_env, only : wp, error_type, fatal_error
   use mctc_io, only: structure_type
   use tblite_basis_slater, only : slater_to_gauss
   use tblite_coulomb_charge, only : new_effective_coulomb, effective_coulomb, &
      & arithmetic_average
   use tblite_coulomb_thirdorder, only : new_onsite_thirdorder
   use tblite_basis_type, only : cgto_type, new_basis, basis_type
   use tblite_ncoord, only : new_ncoord
   use tblite_output_format, only: format_string
   use tblite_integral_type, only : integral_type
   use tblite_xtb_spec, only : tb_h0spec
   use tblite_xtb_calculator, only : xtb_calculator
   use tblite_xtb_h0, only : new_hamiltonian

   implicit none
   private

   public :: ceh_h0spec, new_ceh_calculator, get_effective_qat

   integer, parameter, private :: max_elem = 103
   integer, parameter, private :: max_shell = 4

   !> Number of shells # parameters TF May, 5 2024 - expansion to actinoids
   integer, parameter :: nshell(max_elem) = [& 
   & 1,                                                       1,& !He =s only
   & 2, 2,                                     2, 2, 2, 2, 2, 2,& !Ne  sp 
   & 2, 3,                                     3, 3, 3, 3, 3, 3,& !Ar  spd
   & 2, 3, 3,       3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3,& !Kr  i.e. Ca-Ba are like TMs, Zn-Hg just sp
   & 2, 3, 3,       3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3,& !Xe
   & 2, 3,                                                      & !Cs-Ba
   &       3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,         & !La-Lu
   &                3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3,& !Rn
   & 2, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4]           !Fr,Ra,Ac-No

   !> Angular momentum of each shell 
   ! 0 = s, 1 = p, 2 = d, 3 = f
   ! CAUTION: Ordering from original CEH model is taken for consistency with the parameterization
   ! I.e., the ordering of the shells is always: "s", "p", "d", "f"
   integer, parameter :: ang_shell(max_shell, max_elem) = reshape([&
   & 0, 0, 0, 0,  0, 0, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0, & ! 1-5
   & 0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0, & ! 6-10
   & 0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 11-15 
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0, & ! 16-20
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 21-25
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0, & ! 26-30
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 31-35
   & 0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 36-40
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 41-45
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 46-50
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0, & ! 51-55
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 56-60
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 61-65
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 66-70
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 71-75
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0, & ! 76-80
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 81-85
   & 0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 3,  0, 1, 2, 3, & ! 86-90
   & 0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3, & ! 91-95
   & 0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3, & ! 96-100
   & 0, 1, 2, 3,  0, 1, 2, 3,  0, 1, 2, 3], shape(ang_shell))          ! 101-103

   !> Principal quantum number of each shell (see commeent above regarding angular momentum)
   integer, parameter :: principal_quantum_number(max_shell, max_elem) = reshape([&
   & 1, 0, 0, 0,  1, 0, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0, & ! 1-5
   & 2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0, & ! 6-10
   & 3, 3, 0, 0,  3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0, & ! 11-15
   & 3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0,  4, 4, 0, 0,  4, 4, 3, 0, & ! 16-20
   & 4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0, & ! 21-25
   & 4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 0, 0, & ! 26-30
   & 4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0, & ! 31-35
   & 4, 4, 4, 0,  5, 5, 0, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0, & ! 36-40
   & 5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0, & ! 41-45
   & 5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 0, 0,  5, 5, 5, 0,  5, 5, 5, 0, & ! 46-50
   & 5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0,  6, 6, 0, 0, & ! 51-55
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 56-60
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 61-65
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 66-70
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 71-75
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 0, 0, & ! 76-80
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 81-85
   & 6, 6, 5, 0,  6, 6, 0, 0,  6, 6, 5, 0,  6, 6, 5, 5,  6, 6, 5, 5, & ! 86-90
   & 6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5, & ! 91-95
   & 6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5, & ! 96-100
   & 6, 6, 5, 5,  6, 6, 5, 5,  6, 6, 5, 5], shape(principal_quantum_number)) ! 101-103

   !> Number of primitive gaussians per shell 
   integer, parameter :: number_of_primitives(max_shell, max_elem) = reshape([&
   & 6, 0, 0, 0,  6, 0, 0, 0,  6, 6, 0, 0,  6, 6, 0, 0,  6, 6, 0, 0, &  ! 1-5
   & 6, 6, 0, 0,  6, 6, 0, 0,  6, 6, 0, 0,  6, 6, 0, 0,  6, 6, 0, 0, &  ! 6-10
   & 6, 6, 0, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 11-15
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 0, 0,  6, 6, 6, 0, &  ! 16-20
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 21-25
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 0, 0, &  ! 26-30
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 31-35
   & 6, 6, 6, 0,  6, 6, 0, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 36-40
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 41-45
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 0, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 46-50
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 0, 0, &  ! 51-55
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 56-60
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 61-65
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 66-70
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 71-75
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 0, 0, &  ! 76-80
   & 6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0,  6, 6, 6, 0, &  ! 81-85
   & 6, 6, 6, 0,  6, 6, 0, 0,  6, 6, 6, 0,  6, 6, 6, 6,  6, 6, 6, 6, &  ! 86-90
   & 6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6, &  ! 91-95
   & 6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6, &  ! 96-100
   & 6, 6, 6, 6,  6, 6, 6, 6,  6, 6, 6, 6], shape(number_of_primitives))! 101-103

   !> Reference occupation of the atom
   real(wp), parameter :: reference_occ(max_shell, max_elem) = reshape([&
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 1-2
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 3-4
   & 2.0_wp, 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, & ! 5-6
   & 2.0_wp, 3.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, 0.0_wp, & ! 7-8
   & 2.0_wp, 5.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, 0.0_wp, & ! 9-10
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 11-12
   & 2.0_wp, 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, & ! 13-14
   & 2.0_wp, 3.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, 0.0_wp, & ! 15-16
   & 2.0_wp, 5.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, 0.0_wp, & ! 17-18
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 19-20
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 2.0_wp, 0.0_wp, & ! 21-22
   & 2.0_wp, 0.0_wp, 3.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 4.0_wp, 0.0_wp, & ! 23-24
   & 2.0_wp, 0.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 6.0_wp, 0.0_wp, & ! 25-26
   & 2.0_wp, 0.0_wp, 7.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 8.0_wp, 0.0_wp, & ! 27-28
   & 2.0_wp, 0.0_wp, 9.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 29-30
   & 2.0_wp, 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, & ! 31-32
   & 2.0_wp, 3.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, 0.0_wp, & ! 33-34
   & 2.0_wp, 5.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, 0.0_wp, & ! 35-36
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 37-38
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 2.0_wp, 0.0_wp, & ! 39-40
   & 2.0_wp, 0.0_wp, 3.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 4.0_wp, 0.0_wp, & ! 41-42
   & 2.0_wp, 0.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 6.0_wp, 0.0_wp, & ! 43-44
   & 2.0_wp, 0.0_wp, 7.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 8.0_wp, 0.0_wp, & ! 45-46
   & 2.0_wp, 0.0_wp, 9.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 47-48
   & 2.0_wp, 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, & ! 49-50
   & 2.0_wp, 3.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, 0.0_wp, & ! 51-52
   & 2.0_wp, 5.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, 0.0_wp, & ! 53-54
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 55-56
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 57-58
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 59-60
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 61-62
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 63-64
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 65-66
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 67-68
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp, & ! 69-70
   & 2.0_wp, 0.0_wp, 1.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 2.0_wp, 0.0_wp, & ! 71-72
   & 2.0_wp, 0.0_wp, 3.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 4.0_wp, 0.0_wp, & ! 73-74
   & 2.0_wp, 0.0_wp, 5.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 6.0_wp, 0.0_wp, & ! 75-76
   & 2.0_wp, 0.0_wp, 7.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 8.0_wp, 0.0_wp, & ! 77-78
   & 2.0_wp, 0.0_wp, 9.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 79-80
   & 2.0_wp, 1.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, & ! 81-82
   & 2.0_wp, 3.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 4.0_wp, 0.0_wp, 0.0_wp, & ! 83-84
   & 2.0_wp, 5.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 6.0_wp, 0.0_wp, 0.0_wp, & ! 85-86
   & 1.0_wp, 0.0_wp, 0.0_wp, 0.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & ! 87-88
   & 2.0_wp, 0.0_wp, 0.0_wp, 1.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 2.0_wp, & ! 89-90
   & 2.0_wp, 0.0_wp, 0.0_wp, 3.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 4.0_wp, & ! 91-92
   & 2.0_wp, 0.0_wp, 0.0_wp, 5.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 6.0_wp, & ! 93-94
   & 2.0_wp, 0.0_wp, 0.0_wp, 7.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 8.0_wp, & ! 95-96
   & 2.0_wp, 0.0_wp, 0.0_wp, 9.0_wp,  2.0_wp, 0.0_wp, 0.0_wp, 10.0_wp,& ! 97-98
   & 2.0_wp, 0.0_wp, 0.0_wp, 11.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, 12.0_wp,& ! 99-100
   & 2.0_wp, 0.0_wp, 0.0_wp, 13.0_wp, 2.0_wp, 0.0_wp, 0.0_wp, 14.0_wp,& ! 100-102
   & 2.0_wp, 0.0_wp, 0.0_wp, 15.0_wp], shape(reference_occ))            ! 103


   !> Exponent of the Slater function 
   real(wp), parameter :: slater_exponent(max_shell, max_elem) = reshape([&
   &  1.3492227515_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 1
   &  2.0297549295_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 2
   &  0.8115728674_wp,  1.3372980196_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 3
   &  1.8949180068_wp,  1.2453945983_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 4
   &  2.0235514893_wp,  1.3702489949_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 5
   &  2.0961707647_wp,  1.6690838605_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 6
   &  2.1129464376_wp,  1.7595385612_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 7
   &  2.9238454730_wp,  2.2373745606_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 8
   &  3.7816736284_wp,  2.3808382710_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 9
   &  3.8004570955_wp,  2.3927927290_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 10
   &  0.9780390712_wp,  1.6375686108_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 11
   &  2.0099680914_wp,  1.2343285705_wp,  2.3237411667_wp,  0.0000000000_wp, & ! 12
   &  2.1499724324_wp,  1.4874120230_wp,  2.1014413083_wp,  0.0000000000_wp, & ! 13
   &  2.2270276950_wp,  1.6759234646_wp,  1.8941501341_wp,  0.0000000000_wp, & ! 14
   &  2.5887065804_wp,  1.8007760190_wp,  1.5999563462_wp,  0.0000000000_wp, & ! 15
   &  2.7841325974_wp,  2.0109613546_wp,  1.6776972722_wp,  0.0000000000_wp, & ! 16
   &  3.2842923106_wp,  2.2740114681_wp,  1.9229605825_wp,  0.0000000000_wp, & ! 17
   &  3.0247541040_wp,  2.2384188553_wp,  1.6362995153_wp,  0.0000000000_wp, & ! 18
   &  1.0507900049_wp,  1.8141529710_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 19
   &  2.1811190138_wp,  1.3291817098_wp,  1.7528151180_wp,  0.0000000000_wp, & ! 20
   &  2.6500582337_wp,  1.6429381709_wp,  1.5972010751_wp,  0.0000000000_wp, & ! 21
   &  3.1210114487_wp,  2.2063750441_wp,  1.3450169469_wp,  0.0000000000_wp, & ! 22
   &  3.2304114238_wp,  2.8000000000_wp,  1.4355728481_wp,  0.0000000000_wp, & ! 23
   &  3.6833275964_wp,  2.5540241877_wp,  1.8100452071_wp,  0.0000000000_wp, & ! 24
   &  3.6624423504_wp,  2.1454756862_wp,  1.8084327802_wp,  0.0000000000_wp, & ! 25
   &  3.1428066843_wp,  2.4041020922_wp,  2.2713351955_wp,  0.0000000000_wp, & ! 26
   &  3.2109308678_wp,  1.7693545224_wp,  2.3986127161_wp,  0.0000000000_wp, & ! 27
   &  1.6800234525_wp,  1.8230059108_wp,  2.7136752943_wp,  0.0000000000_wp, & ! 28
   &  1.6921808257_wp,  1.9648586152_wp,  2.9743328935_wp,  0.0000000000_wp, & ! 29
   &  2.3096414432_wp,  1.8840219557_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 30
   &  2.5323515298_wp,  1.8712554407_wp,  2.2662521765_wp,  0.0000000000_wp, & ! 31
   &  2.6864342085_wp,  1.9644943210_wp,  1.8667657880_wp,  0.0000000000_wp, & ! 32
   &  2.9416076911_wp,  2.1001597782_wp,  1.6803727708_wp,  0.0000000000_wp, & ! 33
   &  2.9092984937_wp,  2.2701675106_wp,  1.7489108979_wp,  0.0000000000_wp, & ! 34
   &  3.6516375423_wp,  2.4699047002_wp,  1.4832134005_wp,  0.0000000000_wp, & ! 35
   &  4.6868325712_wp,  2.4380374306_wp,  1.8160827467_wp,  0.0000000000_wp, & ! 36
   &  1.1122591420_wp,  1.9530359548_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 37
   &  2.1700962328_wp,  1.6568420289_wp,  1.8354587881_wp,  0.0000000000_wp, & ! 38
   &  2.9065274114_wp,  1.7441125751_wp,  1.8765657176_wp,  0.0000000000_wp, & ! 39
   &  3.1483719836_wp,  2.5226675333_wp,  1.7412576712_wp,  0.0000000000_wp, & ! 40
   &  3.1120762338_wp,  3.3393108460_wp,  1.8515019279_wp,  0.0000000000_wp, & ! 41
   &  3.5757122514_wp,  2.8945430976_wp,  1.9286110184_wp,  0.0000000000_wp, & ! 42
   &  3.8039082297_wp,  2.7977943332_wp,  1.9108424098_wp,  0.0000000000_wp, & ! 43
   &  3.5911983522_wp,  2.5001885307_wp,  2.3881168152_wp,  0.0000000000_wp, & ! 44
   &  3.4341340755_wp,  1.9792447072_wp,  2.6337554251_wp,  0.0000000000_wp, & ! 45
   &  2.5673781587_wp,  2.5197108936_wp,  2.8058619890_wp,  0.0000000000_wp, & ! 46
   &  1.9730499701_wp,  2.4221932255_wp,  3.2722047449_wp,  0.0000000000_wp, & ! 47
   &  2.5310216009_wp,  2.0618879177_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 48
   &  2.7137627220_wp,  2.0589020500_wp,  2.4405484053_wp,  0.0000000000_wp, & ! 49
   &  2.7374562176_wp,  2.1456907046_wp,  1.9915924011_wp,  0.0000000000_wp, & ! 50
   &  3.2052138093_wp,  2.1745234903_wp,  1.6282018949_wp,  0.0000000000_wp, & ! 51
   &  2.9148428747_wp,  2.3788003259_wp,  1.9341312980_wp,  0.0000000000_wp, & ! 52
   &  3.2788193429_wp,  2.5028810955_wp,  1.6483182456_wp,  0.0000000000_wp, & ! 53
   &  3.4959615948_wp,  2.8681992514_wp,  2.1125878098_wp,  0.0000000000_wp, & ! 54
   &  1.1985088327_wp,  2.2469215362_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 55
   &  1.3353046103_wp,  1.0823412644_wp,  2.2476720524_wp,  0.0000000000_wp, & ! 56
   &  3.4792464313_wp,  2.0725957547_wp,  1.9509190518_wp,  0.0000000000_wp, & ! 57
   &  2.8413056910_wp,  2.0155958455_wp,  2.1791218519_wp,  0.0000000000_wp, & ! 58
   &  2.9410785177_wp,  1.9663079616_wp,  2.2143521638_wp,  0.0000000000_wp, & ! 59
   &  3.0240255453_wp,  1.9314493093_wp,  2.2439291805_wp,  0.0000000000_wp, & ! 60
   &  3.0901467738_wp,  1.9110198885_wp,  2.2678529019_wp,  0.0000000000_wp, & ! 61
   &  3.1394422033_wp,  1.9050196993_wp,  2.2861233280_wp,  0.0000000000_wp, & ! 62
   &  3.1719118336_wp,  1.9134487416_wp,  2.2987404589_wp,  0.0000000000_wp, & ! 63
   &  3.1875556649_wp,  1.9363070154_wp,  2.3057042945_wp,  0.0000000000_wp, & ! 64
   &  3.1863736971_wp,  1.9735945208_wp,  2.3070148348_wp,  0.0000000000_wp, & ! 65
   &  3.1683659302_wp,  2.0253112576_wp,  2.3026720799_wp,  0.0000000000_wp, & ! 66
   &  3.1335323642_wp,  2.0914572261_wp,  2.2926760297_wp,  0.0000000000_wp, & ! 67
   &  3.0818729991_wp,  2.1720324260_wp,  2.2770266843_wp,  0.0000000000_wp, & ! 68
   &  3.0133878350_wp,  2.2670368575_wp,  2.2557240435_wp,  0.0000000000_wp, & ! 69
   &  2.9280768717_wp,  2.3764705205_wp,  2.2287681076_wp,  0.0000000000_wp, & ! 70
   &  2.8259401094_wp,  2.5003334151_wp,  2.1961588763_wp,  0.0000000000_wp, & ! 71
   &  3.5245573534_wp,  2.4297871957_wp,  2.1591019214_wp,  0.0000000000_wp, & ! 72
   &  3.6204081892_wp,  3.4886162192_wp,  2.0348841671_wp,  0.0000000000_wp, & ! 73
   &  3.1874731453_wp,  3.5832948648_wp,  2.2114325234_wp,  0.0000000000_wp, & ! 74
   &  2.4889827666_wp,  3.2496786940_wp,  2.6078922006_wp,  0.0000000000_wp, & ! 75
   &  1.8752981823_wp,  2.5236235190_wp,  3.2170345714_wp,  0.0000000000_wp, & ! 76
   &  3.3518557037_wp,  2.1795375597_wp,  3.0704345355_wp,  0.0000000000_wp, & ! 77
   &  2.9755283461_wp,  2.4631605229_wp,  3.3072343576_wp,  0.0000000000_wp, & ! 78
   &  2.3254804374_wp,  2.8297362945_wp,  3.7921037119_wp,  0.0000000000_wp, & ! 79
   &  2.6611941708_wp,  2.3006727961_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 80
   &  2.6904584735_wp,  2.5189856427_wp,  2.5271623356_wp,  0.0000000000_wp, & ! 81
   &  3.1696908120_wp,  2.5317542591_wp,  2.5840902298_wp,  0.0000000000_wp, & ! 82
   &  3.8892708643_wp,  2.4529963935_wp,  1.9896835213_wp,  0.0000000000_wp, & ! 83
   &  3.2526076311_wp,  2.5790181159_wp,  2.3230448536_wp,  0.0000000000_wp, & ! 84
   &  3.9949484574_wp,  2.8552456418_wp,  1.9701615730_wp,  0.0000000000_wp, & ! 85
   &  3.6264048754_wp,  2.7395056301_wp,  2.1603932350_wp,  0.0000000000_wp, & ! 86
   &  1.8191322012_wp,  3.2306440829_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 87
   &  2.5148686140_wp,  2.4608040579_wp,  2.1239900632_wp,  0.0000000000_wp, & ! 88
   &  3.7600847232_wp,  3.4422103009_wp,  1.9765721411_wp,  2.0126790384_wp, & ! 89
   &  3.4457768232_wp,  2.8108963773_wp,  2.0591592480_wp,  2.0505286357_wp, & ! 90
   &  3.1504721425_wp,  2.2562397801_wp,  2.1333241969_wp,  2.0912170981_wp, & ! 91
   &  2.8741706811_wp,  1.7782405092_wp,  2.1990669879_wp,  2.1347444257_wp, & ! 92
   &  2.6168724391_wp,  1.3768985647_wp,  2.2563876209_wp,  2.1811106185_wp, & ! 93
   &  2.3785774165_wp,  1.0522139464_wp,  2.3052860960_wp,  2.2303156765_wp, & ! 94
   &  2.1592856131_wp,  0.8041866545_wp,  2.3457624132_wp,  2.2823595997_wp, & ! 95
   &  1.9589970291_wp,  0.6328166889_wp,  2.3778165724_wp,  2.3372423880_wp, & ! 96
   &  1.7777116644_wp,  0.5381040496_wp,  2.4014485737_wp,  2.3949640415_wp, & ! 97
   &  1.6154295191_wp,  0.5200487367_wp,  2.4166584170_wp,  2.4555245602_wp, & ! 98
   &  1.4721505931_wp,  0.5786507501_wp,  2.4234461024_wp,  2.5189239441_wp, & ! 99
   &  1.3478748864_wp,  0.7139100898_wp,  2.4218116298_wp,  2.5851621932_wp, & ! 100
   &  1.2426023991_wp,  0.9258267558_wp,  2.4117549993_wp,  2.6542393074_wp, & ! 101
   &  1.1563331311_wp,  1.2144007482_wp,  2.3932762109_wp,  2.7261552868_wp, & ! 102
   &  1.0890670824_wp,  1.5796320669_wp,  2.3663752645_wp,  2.8009101314_wp],& ! 103
   & shape(slater_exponent))

   !> Atomic shell energy level 
   real(wp), parameter :: p_ceh_selfenergy(max_shell, max_elem) = reshape([&
   & -0.5000000000_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 1
   & -0.6105701118_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 2
   & -0.3980717125_wp, -0.1712410421_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 3
   & -0.5493798195_wp, -0.2313407389_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 4
   & -0.5986214615_wp, -0.3739274044_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 5
   & -0.6211977182_wp, -0.3722531247_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 6
   & -0.5914793762_wp, -0.4283089200_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 7
   & -0.5879099448_wp, -0.4682745513_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 8
   & -0.6848208290_wp, -0.5440900872_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 9
   & -0.8507792601_wp, -0.6419314780_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 10
   & -0.4082881538_wp, -0.1764848179_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 11
   & -0.5357768614_wp, -0.1530659439_wp, -0.0782114458_wp,  0.0000000000_wp, & ! 12
   & -0.5993559020_wp, -0.3643716697_wp, -0.0960944289_wp,  0.0000000000_wp, & ! 13
   & -0.6282175504_wp, -0.4210134270_wp, -0.1359234692_wp,  0.0000000000_wp, & ! 14
   & -0.7859765547_wp, -0.4506042491_wp, -0.0224801543_wp,  0.0000000000_wp, & ! 15
   & -0.6424030955_wp, -0.4974377817_wp, -0.0478440353_wp,  0.0000000000_wp, & ! 16
   & -0.7519476112_wp, -0.5459064862_wp,  0.0085248988_wp,  0.0000000000_wp, & ! 17
   & -1.1346640054_wp, -0.6094886492_wp, -0.1531871109_wp,  0.0000000000_wp, & ! 18
   & -0.3784622555_wp, -0.1809363315_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 19
   & -0.4976772120_wp, -0.1100899630_wp, -0.1723178857_wp,  0.0000000000_wp, & ! 20
   & -0.4698199591_wp, -0.1476330080_wp, -0.3744164929_wp,  0.0000000000_wp, & ! 21
   & -0.6032842175_wp,  0.0435384168_wp, -0.3071125792_wp,  0.0000000000_wp, & ! 22
   & -0.5416770905_wp,  0.0632161139_wp, -0.3675241615_wp,  0.0000000000_wp, & ! 23
   & -0.4028182579_wp, -0.1874033025_wp, -0.4141888852_wp,  0.0000000000_wp, & ! 24
   & -0.5027540878_wp, -0.2338729034_wp, -0.4139156380_wp,  0.0000000000_wp, & ! 25
   & -0.4443179593_wp, -0.1995113070_wp, -0.4390663883_wp,  0.0000000000_wp, & ! 26
   & -0.4466317436_wp, -0.0856088250_wp, -0.4915053205_wp,  0.0000000000_wp, & ! 27
   & -0.3710648050_wp, -0.0292837480_wp, -0.5510762431_wp,  0.0000000000_wp, & ! 28
   & -0.4431825724_wp, -0.1044875989_wp, -0.6144703794_wp,  0.0000000000_wp, & ! 29
   & -0.5526004308_wp, -0.2255431036_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 30
   & -0.6107600537_wp, -0.3687596736_wp, -0.1154596368_wp,  0.0000000000_wp, & ! 31
   & -0.6882149886_wp, -0.4013056377_wp, -0.1147737534_wp,  0.0000000000_wp, & ! 32
   & -0.7620445756_wp, -0.4286700977_wp, -0.0520823107_wp,  0.0000000000_wp, & ! 33
   & -0.6763541094_wp, -0.4976426001_wp, -0.1393615016_wp,  0.0000000000_wp, & ! 34
   & -0.8343227021_wp, -0.5427979460_wp,  0.0159985712_wp,  0.0000000000_wp, & ! 35
   & -0.6348672274_wp, -0.5923924152_wp, -0.2714352446_wp,  0.0000000000_wp, & ! 36
   & -0.3693790748_wp, -0.1892524914_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 37
   & -0.5052776667_wp, -0.1291876799_wp, -0.1726957789_wp,  0.0000000000_wp, & ! 38
   & -0.5036347592_wp, -0.0858402498_wp, -0.3722867061_wp,  0.0000000000_wp, & ! 39
   & -0.4242069681_wp, -0.2080486677_wp, -0.3960459576_wp,  0.0000000000_wp, & ! 40
   & -0.4187213574_wp, -0.1183400848_wp, -0.4011165625_wp,  0.0000000000_wp, & ! 41
   & -0.4027437853_wp, -0.1603440759_wp, -0.4147156229_wp,  0.0000000000_wp, & ! 42
   & -0.4842348588_wp, -0.2288337106_wp, -0.4431201813_wp,  0.0000000000_wp, & ! 43
   & -0.4233369874_wp, -0.1801095129_wp, -0.4288923629_wp,  0.0000000000_wp, & ! 44
   & -0.4150225384_wp, -0.1327083880_wp, -0.4933212999_wp,  0.0000000000_wp, & ! 45
   & -0.3223546220_wp, -0.1098673573_wp, -0.5168515731_wp,  0.0000000000_wp, & ! 46
   & -0.4431358984_wp, -0.1924311146_wp, -0.6349344975_wp,  0.0000000000_wp, & ! 47
   & -0.5938620850_wp, -0.1816448409_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 48
   & -0.6219725121_wp, -0.3821819350_wp,  0.0336167627_wp,  0.0000000000_wp, & ! 49
   & -0.6580476011_wp, -0.4053474511_wp, -0.1392316470_wp,  0.0000000000_wp, & ! 50
   & -0.7712752463_wp, -0.4370255437_wp, -0.0731349297_wp,  0.0000000000_wp, & ! 51
   & -0.6190627403_wp, -0.4923030968_wp, -0.1394817831_wp,  0.0000000000_wp, & ! 52
   & -1.0675221725_wp, -0.5202000960_wp, -0.0585257643_wp,  0.0000000000_wp, & ! 53
   & -0.7640495768_wp, -0.5687349737_wp, -0.2450415648_wp,  0.0000000000_wp, & ! 54
   & -0.3646022319_wp, -0.2567662304_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 55
   & -0.4958573980_wp, -0.0579313200_wp, -0.2754944964_wp,  0.0000000000_wp, & ! 56
   & -0.4806029798_wp, -0.0760916017_wp, -0.3716134681_wp,  0.0000000000_wp, & ! 57
   & -0.5980037903_wp, -0.1054158913_wp, -0.3605357884_wp,  0.0000000000_wp, & ! 58
   & -0.5872365756_wp, -0.0897207717_wp, -0.3617131051_wp,  0.0000000000_wp, & ! 59
   & -0.5772549666_wp, -0.0767114956_wp, -0.3621843355_wp,  0.0000000000_wp, & ! 60
   & -0.5680589634_wp, -0.0663880630_wp, -0.3619494797_wp,  0.0000000000_wp, & ! 61
   & -0.5596485658_wp, -0.0587504740_wp, -0.3610085375_wp,  0.0000000000_wp, & ! 62
   & -0.5520237740_wp, -0.0537987284_wp, -0.3593615091_wp,  0.0000000000_wp, & ! 63
   & -0.5451845879_wp, -0.0515328263_wp, -0.3570083944_wp,  0.0000000000_wp, & ! 64
   & -0.5391310075_wp, -0.0519527677_wp, -0.3539491934_wp,  0.0000000000_wp, & ! 65
   & -0.5338630328_wp, -0.0550585526_wp, -0.3501839061_wp,  0.0000000000_wp, & ! 66
   & -0.5293806639_wp, -0.0608501811_wp, -0.3457125326_wp,  0.0000000000_wp, & ! 67
   & -0.5256839006_wp, -0.0693276530_wp, -0.3405350728_wp,  0.0000000000_wp, & ! 68
   & -0.5227727431_wp, -0.0804909684_wp, -0.3346515267_wp,  0.0000000000_wp, & ! 69
   & -0.5206471913_wp, -0.0943401274_wp, -0.3280618943_wp,  0.0000000000_wp, & ! 70
   & -0.5193072452_wp, -0.1108751298_wp, -0.3207661756_wp,  0.0000000000_wp, & ! 71
   & -0.4623147732_wp, -0.0827972193_wp, -0.3951227578_wp,  0.0000000000_wp, & ! 72
   & -0.5676076872_wp, -0.1297231213_wp, -0.3841538494_wp,  0.0000000000_wp, & ! 73
   & -0.4440276965_wp, -0.1517198525_wp, -0.4111516483_wp,  0.0000000000_wp, & ! 74
   & -0.5204612030_wp, -0.1605302366_wp, -0.4043815933_wp,  0.0000000000_wp, & ! 75
   & -0.4428739353_wp, -0.1685230522_wp, -0.4513851187_wp,  0.0000000000_wp, & ! 76
   & -0.4437336187_wp, -0.1626165253_wp, -0.5109398647_wp,  0.0000000000_wp, & ! 77
   & -0.4451972897_wp, -0.1197326575_wp, -0.5017573255_wp,  0.0000000000_wp, & ! 78
   & -0.4662987636_wp, -0.1804766995_wp, -0.6309289193_wp,  0.0000000000_wp, & ! 79
   & -0.5906284737_wp, -0.1775507756_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 80
   & -0.6128157508_wp, -0.3611736039_wp, -0.0260021017_wp,  0.0000000000_wp, & ! 81
   & -0.7906349302_wp, -0.4008308461_wp,  0.0442013091_wp,  0.0000000000_wp, & ! 82
   & -0.6866755608_wp, -0.4357542354_wp, -0.1050226008_wp,  0.0000000000_wp, & ! 83
   & -0.6866322660_wp, -0.4817422956_wp, -0.1554260867_wp,  0.0000000000_wp, & ! 84
   & -1.3541073935_wp, -0.5105541129_wp, -0.0006561065_wp,  0.0000000000_wp, & ! 85
   & -1.0161301492_wp, -0.5914148774_wp, -0.2482937153_wp,  0.0000000000_wp, & ! 86
   & -0.4263498888_wp, -0.1759887493_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 87
   & -0.5235095933_wp, -0.2385352306_wp, -0.1981714405_wp,  0.0000000000_wp, & ! 88
   & -0.7181667125_wp, -0.2012905526_wp, -0.3070120533_wp, -0.3280755854_wp, & ! 89
   & -0.6532587639_wp, -0.2034245339_wp, -0.3137814395_wp, -0.3461010689_wp, & ! 90
   & -0.5963200744_wp, -0.2053171647_wp, -0.3188341432_wp, -0.3632237589_wp, & ! 91
   & -0.5473506438_wp, -0.2069684450_wp, -0.3221701645_wp, -0.3794436554_wp, & ! 92
   & -0.5063504723_wp, -0.2083783748_wp, -0.3237895033_wp, -0.3947607586_wp, & ! 93
   & -0.4733195599_wp, -0.2095469540_wp, -0.3236921596_wp, -0.4091750682_wp, & ! 94
   & -0.4482579064_wp, -0.2104741827_wp, -0.3218781335_wp, -0.4226865845_wp, & ! 95
   & -0.4311655120_wp, -0.2111600608_wp, -0.3183474249_wp, -0.4352953073_wp, & ! 96
   & -0.4220423766_wp, -0.2116045884_wp, -0.3131000339_wp, -0.4470012367_wp, & ! 97
   & -0.4208885003_wp, -0.2118077655_wp, -0.3061359604_wp, -0.4578043726_wp, & ! 98
   & -0.4277038829_wp, -0.2117695920_wp, -0.2974552044_wp, -0.4677047151_wp, & ! 99
   & -0.4424885246_wp, -0.2114900681_wp, -0.2870577660_wp, -0.4767022641_wp, & ! 100
   & -0.4652424254_wp, -0.2109691935_wp, -0.2749436451_wp, -0.4847970197_wp, & ! 101
   & -0.4959655851_wp, -0.2102069685_wp, -0.2611128418_wp, -0.4919889819_wp, & ! 102
   & -0.5346580039_wp, -0.2092033929_wp, -0.2455653560_wp, -0.4982781506_wp],& ! 103
   & shape(p_ceh_selfenergy))

   !> Dependence of orbital shell energy level on standard CN (shell-resolved) 
   real(wp), parameter :: p_ceh_kcn(max_shell, max_elem) = reshape([&
   & -0.0287030802_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 1
   & -0.7428586490_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 2
   &  0.1470744624_wp, -0.0310672140_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 3
   &  0.0087398075_wp, -0.0111848697_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 4
   & -0.0078637881_wp, -0.0096266909_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 5
   & -0.0157862882_wp, -0.0807336431_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 6
   & -0.1395435014_wp, -0.1811072143_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 7
   & -0.4455963150_wp, -0.3472472055_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 8
   & -0.1311669918_wp, -0.3533789894_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 9
   & -0.1910189749_wp, -0.5287019681_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 10
   &  0.1672462439_wp, -0.0378718118_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 11
   &  0.0337257603_wp,  0.0163507574_wp, -0.0376385054_wp,  0.0000000000_wp, & ! 12
   &  0.0120262702_wp,  0.0141483265_wp, -0.0371011858_wp,  0.0000000000_wp, & ! 13
   & -0.0014838071_wp, -0.0070970502_wp, -0.0101045581_wp,  0.0000000000_wp, & ! 14
   &  0.0164531822_wp, -0.0285720390_wp, -0.0187198058_wp,  0.0000000000_wp, & ! 15
   & -0.0524983990_wp, -0.0719038371_wp, -0.0104596777_wp,  0.0000000000_wp, & ! 16
   &  0.0034199504_wp, -0.2001699623_wp, -0.0494733665_wp,  0.0000000000_wp, & ! 17
   & -0.7170813593_wp, -0.2135862574_wp,  0.0382448609_wp,  0.0000000000_wp, & ! 18
   &  0.1443871554_wp, -0.0214758300_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 19
   &  0.0226017610_wp,  0.0132354502_wp,  0.0177998281_wp,  0.0000000000_wp, & ! 20
   &  0.0361668478_wp, -0.0220258222_wp,  0.0326482498_wp,  0.0000000000_wp, & ! 21
   & -0.0120495413_wp, -0.0289237879_wp,  0.0182515548_wp,  0.0000000000_wp, & ! 22
   & -0.0210242738_wp,  0.0377821442_wp, -0.0131326524_wp,  0.0000000000_wp, & ! 23
   &  0.0290821935_wp,  0.0052524662_wp, -0.0343666975_wp,  0.0000000000_wp, & ! 24
   &  0.0566198540_wp, -0.0149216373_wp, -0.0558722457_wp,  0.0000000000_wp, & ! 25
   &  0.0436537975_wp, -0.0191743333_wp, -0.0843585820_wp,  0.0000000000_wp, & ! 26
   &  0.0087300373_wp, -0.0556290752_wp, -0.0459200974_wp,  0.0000000000_wp, & ! 27
   &  0.0264807806_wp, -0.0360595125_wp, -0.1029686420_wp,  0.0000000000_wp, & ! 28
   &  0.0281356716_wp, -0.0042000643_wp, -0.1357665574_wp,  0.0000000000_wp, & ! 29
   &  0.0036868729_wp,  0.0156292714_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 30
   & -0.0013747720_wp,  0.0165273203_wp, -0.0274063500_wp,  0.0000000000_wp, & ! 31
   & -0.0031318347_wp, -0.0021775381_wp,  0.0004220989_wp,  0.0000000000_wp, & ! 32
   &  0.0195660321_wp, -0.0383787165_wp, -0.0079495985_wp,  0.0000000000_wp, & ! 33
   & -0.0592374915_wp, -0.0589382100_wp,  0.0097875292_wp,  0.0000000000_wp, & ! 34
   &  0.0204839123_wp, -0.1289853201_wp, -0.0457438550_wp,  0.0000000000_wp, & ! 35
   & -0.0434385124_wp, -0.1368181660_wp,  0.0349141290_wp,  0.0000000000_wp, & ! 36
   &  0.2076215151_wp, -0.0183268941_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 37
   &  0.0260614775_wp,  0.0166416700_wp,  0.0155177225_wp,  0.0000000000_wp, & ! 38
   &  0.0768902098_wp, -0.0277621605_wp,  0.0265531040_wp,  0.0000000000_wp, & ! 39
   &  0.0366091610_wp, -0.0033394840_wp,  0.0000310477_wp,  0.0000000000_wp, & ! 40
   & -0.0115045674_wp,  0.0499341536_wp, -0.0158565748_wp,  0.0000000000_wp, & ! 41
   &  0.0155487434_wp,  0.0074705341_wp, -0.0385941409_wp,  0.0000000000_wp, & ! 42
   &  0.0690066746_wp, -0.0162048876_wp, -0.0668611478_wp,  0.0000000000_wp, & ! 43
   &  0.0239092664_wp, -0.0172210982_wp, -0.0791379467_wp,  0.0000000000_wp, & ! 44
   &  0.0498197785_wp, -0.0280157703_wp, -0.0823250401_wp,  0.0000000000_wp, & ! 45
   & -0.0161251218_wp, -0.0030432486_wp, -0.1701069549_wp,  0.0000000000_wp, & ! 46
   &  0.0250790564_wp,  0.0443824182_wp, -0.2101706179_wp,  0.0000000000_wp, & ! 47
   &  0.0376387616_wp, -0.0226473318_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 48
   &  0.0040538946_wp,  0.0238424667_wp, -0.0673514981_wp,  0.0000000000_wp, & ! 49
   & -0.0052818978_wp,  0.0000014419_wp,  0.0044455966_wp,  0.0000000000_wp, & ! 50
   &  0.0336085302_wp, -0.0245616519_wp,  0.0030791331_wp,  0.0000000000_wp, & ! 51
   & -0.0881786884_wp, -0.0526656006_wp,  0.0127696417_wp,  0.0000000000_wp, & ! 52
   &  0.0627632061_wp, -0.0950004224_wp, -0.0140403107_wp,  0.0000000000_wp, & ! 53
   & -0.2054131373_wp, -0.1247887668_wp,  0.0164615308_wp,  0.0000000000_wp, & ! 54
   &  0.3083124766_wp, -0.0007495884_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 55
   &  0.0986045967_wp,  0.0070073511_wp,  0.0177178064_wp,  0.0000000000_wp, & ! 56
   &  0.1084931570_wp, -0.0196167880_wp,  0.0279526025_wp,  0.0000000000_wp, & ! 57
   &  0.1689219255_wp, -0.0243010451_wp,  0.0057683202_wp,  0.0000000000_wp, & ! 58
   &  0.1423328386_wp, -0.0201125572_wp,  0.0104905669_wp,  0.0000000000_wp, & ! 59
   &  0.1180526324_wp, -0.0166381375_wp,  0.0149102147_wp,  0.0000000000_wp, & ! 60
   &  0.0960813069_wp, -0.0138777858_wp,  0.0190272635_wp,  0.0000000000_wp, & ! 61
   &  0.0764188620_wp, -0.0118315023_wp,  0.0228417133_wp,  0.0000000000_wp, & ! 62
   &  0.0590652978_wp, -0.0104992869_wp,  0.0263535642_wp,  0.0000000000_wp, & ! 63
   &  0.0440206143_wp, -0.0098811396_wp,  0.0295628162_wp,  0.0000000000_wp, & ! 64
   &  0.0312848114_wp, -0.0099770604_wp,  0.0324694692_wp,  0.0000000000_wp, & ! 65
   &  0.0208578893_wp, -0.0107870494_wp,  0.0350735233_wp,  0.0000000000_wp, & ! 66
   &  0.0127398477_wp, -0.0123111064_wp,  0.0373749784_wp,  0.0000000000_wp, & ! 67
   &  0.0069306869_wp, -0.0145492316_wp,  0.0393738345_wp,  0.0000000000_wp, & ! 68
   &  0.0034304067_wp, -0.0175014249_wp,  0.0410700917_wp,  0.0000000000_wp, & ! 69
   &  0.0022390072_wp, -0.0211676863_wp,  0.0424637500_wp,  0.0000000000_wp, & ! 70
   &  0.0033564884_wp, -0.0255480158_wp,  0.0435548093_wp,  0.0000000000_wp, & ! 71
   & -0.0410682161_wp, -0.0349462671_wp,  0.0472843961_wp,  0.0000000000_wp, & ! 72
   &  0.0206997708_wp,  0.0097996817_wp, -0.0144950816_wp,  0.0000000000_wp, & ! 73
   & -0.0388066806_wp, -0.0262920894_wp, -0.0154375007_wp,  0.0000000000_wp, & ! 74
   & -0.0240737455_wp, -0.0218584977_wp, -0.0598926997_wp,  0.0000000000_wp, & ! 75
   & -0.0259319101_wp,  0.0068785541_wp, -0.0456887308_wp,  0.0000000000_wp, & ! 76
   &  0.0060194895_wp, -0.0288400805_wp, -0.0400707881_wp,  0.0000000000_wp, & ! 77
   &  0.0422933104_wp, -0.0072395368_wp, -0.1539779625_wp,  0.0000000000_wp, & ! 78
   & -0.0328386264_wp,  0.0191566653_wp, -0.0955303298_wp,  0.0000000000_wp, & ! 79
   & -0.0283909236_wp, -0.0047644400_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 80
   & -0.0849292560_wp,  0.0593488691_wp,  0.0211691879_wp,  0.0000000000_wp, & ! 81
   &  0.0093144078_wp,  0.0148931798_wp, -0.0394199195_wp,  0.0000000000_wp, & ! 82
   & -0.0476994679_wp, -0.0242564574_wp,  0.0522000134_wp,  0.0000000000_wp, & ! 83
   & -0.0463978409_wp, -0.0680801540_wp,  0.0190057635_wp,  0.0000000000_wp, & ! 84
   &  0.0499477549_wp, -0.1762688412_wp,  0.0320243300_wp,  0.0000000000_wp, & ! 85
   & -0.3897982439_wp, -0.1144257641_wp,  0.0513899590_wp,  0.0000000000_wp, & ! 86
   &  0.2162172638_wp, -0.1025595391_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 87
   &  0.0524900864_wp,  0.0705231253_wp,  0.0080451606_wp,  0.0000000000_wp, & ! 88
   &  0.0820444908_wp, -0.0374291454_wp, -0.0061772789_wp,  0.0061588752_wp, & ! 89
   &  0.0701065656_wp, -0.0073268289_wp, -0.0099205545_wp, -0.0019655425_wp, & ! 90
   &  0.0589721361_wp,  0.0184228616_wp, -0.0124248143_wp, -0.0096747847_wp, & ! 91
   &  0.0486412023_wp,  0.0398199263_wp, -0.0136900581_wp, -0.0169688515_wp, & ! 92
   &  0.0391137642_wp,  0.0568643652_wp, -0.0137162861_wp, -0.0238477429_wp, & ! 93
   &  0.0303898218_wp,  0.0695561782_wp, -0.0125034982_wp, -0.0303114589_wp, & ! 94
   &  0.0224693752_wp,  0.0778953653_wp, -0.0100516944_wp, -0.0363599995_wp, & ! 95
   &  0.0153524242_wp,  0.0818819265_wp, -0.0063608747_wp, -0.0419933647_wp, & ! 96
   &  0.0090389689_wp,  0.0815158619_wp, -0.0014310391_wp, -0.0472115545_wp, & ! 97
   &  0.0035290094_wp,  0.0767971714_wp,  0.0047378123_wp, -0.0520145688_wp, & ! 98
   & -0.0011774544_wp,  0.0677258550_wp,  0.0121456797_wp, -0.0564024078_wp, & ! 99
   & -0.0050804226_wp,  0.0543019127_wp,  0.0207925629_wp, -0.0603750713_wp, & ! 100
   & -0.0081798950_wp,  0.0365253446_wp,  0.0306784620_wp, -0.0639325594_wp, & ! 101
   & -0.0104758717_wp,  0.0143961507_wp,  0.0418033770_wp, -0.0670748721_wp, & ! 102
   & -0.0119683527_wp, -0.0120856692_wp,  0.0541673079_wp, -0.0698020094_wp],& ! 103
   & shape(p_ceh_kcn))

   !> Dependence of orbital shell energy level on EN-weighted CN (atom-resolved) 
   real(wp), parameter :: p_ceh_kcn_en(max_shell, max_elem) = reshape([&
   & -0.1602781645_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 1
   & -1.8552984851_wp,  0.0000000000_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 2
   &  0.2277088252_wp,  0.0691372807_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 3
   &  0.0750184353_wp, -0.0837823101_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 4
   & -0.0606225174_wp, -0.0940329531_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 5
   & -0.1178974363_wp, -0.1899725596_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 6
   & -0.3505888579_wp, -0.3597322261_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 7
   & -0.6204154084_wp, -0.4797029321_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 8
   & -0.4533098239_wp, -0.6899160143_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 9
   & -0.4256898040_wp, -0.7511337166_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 10
   &  0.1780497249_wp,  0.0416203930_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 11
   &  0.1063321033_wp,  0.0363711107_wp, -0.0591362752_wp,  0.0000000000_wp, & ! 12
   & -0.0319923529_wp,  0.0202807712_wp,  0.0278269173_wp,  0.0000000000_wp, & ! 13
   & -0.0836644946_wp, -0.0360531395_wp, -0.0193672936_wp,  0.0000000000_wp, & ! 14
   & -0.0790604278_wp, -0.0964098170_wp, -0.0541924416_wp,  0.0000000000_wp, & ! 15
   & -0.2622369137_wp, -0.1845104293_wp, -0.0432625519_wp,  0.0000000000_wp, & ! 16
   & -0.2318862452_wp, -0.3292364936_wp, -0.0804051573_wp,  0.0000000000_wp, & ! 17
   & -2.7091935503_wp, -0.5409490584_wp, -0.2614548698_wp,  0.0000000000_wp, & ! 18
   &  0.1315333954_wp,  0.0378246100_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 19
   &  0.1446048804_wp,  0.1012005250_wp, -0.0960919612_wp,  0.0000000000_wp, & ! 20
   &  0.1862095261_wp,  0.1229007013_wp, -0.0117579593_wp,  0.0000000000_wp, & ! 21
   &  0.0976041704_wp,  0.0466535538_wp, -0.0420636751_wp,  0.0000000000_wp, & ! 22
   & -0.0522799743_wp, -0.1483926995_wp, -0.0228208319_wp,  0.0000000000_wp, & ! 23
   &  0.1924431859_wp,  0.0332767477_wp, -0.0187096623_wp,  0.0000000000_wp, & ! 24
   &  0.3891253652_wp,  0.0696163784_wp, -0.1052168118_wp,  0.0000000000_wp, & ! 25
   &  0.2492971015_wp,  0.0317529716_wp, -0.0732584430_wp,  0.0000000000_wp, & ! 26
   &  0.2201775045_wp, -0.0083715969_wp, -0.1435587818_wp,  0.0000000000_wp, & ! 27
   &  0.0563809932_wp, -0.0016068546_wp, -0.1989103943_wp,  0.0000000000_wp, & ! 28
   &  0.1028690859_wp,  0.1091171081_wp, -0.3200256963_wp,  0.0000000000_wp, & ! 29
   &  0.0312383069_wp,  0.0592419051_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 30
   & -0.0545945731_wp,  0.0169997299_wp,  0.0105619082_wp,  0.0000000000_wp, & ! 31
   & -0.0967339374_wp, -0.0470953937_wp,  0.0168461952_wp,  0.0000000000_wp, & ! 32
   & -0.1315473946_wp, -0.1486788805_wp, -0.0281930173_wp,  0.0000000000_wp, & ! 33
   & -0.3240917368_wp, -0.1904468033_wp, -0.0306677467_wp,  0.0000000000_wp, & ! 34
   & -0.4358045709_wp, -0.2975053240_wp, -0.0660744311_wp,  0.0000000000_wp, & ! 35
   & -0.2464079943_wp, -0.3878934004_wp, -0.1479683952_wp,  0.0000000000_wp, & ! 36
   &  0.2320123068_wp,  0.0339446999_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 37
   &  0.1381282754_wp,  0.1435176679_wp, -0.1424064758_wp,  0.0000000000_wp, & ! 38
   &  0.5155615416_wp,  0.1079256779_wp,  0.0012804591_wp,  0.0000000000_wp, & ! 39
   &  0.1286574072_wp,  0.0733181139_wp,  0.0277413717_wp,  0.0000000000_wp, & ! 40
   &  0.3358982273_wp, -0.2504495240_wp, -0.1222013015_wp,  0.0000000000_wp, & ! 41
   &  0.1068004069_wp,  0.0405849144_wp, -0.0182648418_wp,  0.0000000000_wp, & ! 42
   &  0.2601653573_wp,  0.0262579951_wp, -0.1004549186_wp,  0.0000000000_wp, & ! 43
   &  0.1849727354_wp, -0.0073430542_wp, -0.1233297935_wp,  0.0000000000_wp, & ! 44
   &  0.1786963968_wp,  0.0287449309_wp, -0.1075764917_wp,  0.0000000000_wp, & ! 45
   &  0.0263654682_wp,  0.1057347555_wp, -0.1696052714_wp,  0.0000000000_wp, & ! 46
   &  0.1425869040_wp,  0.0977308622_wp, -0.3573683734_wp,  0.0000000000_wp, & ! 47
   &  0.0723880788_wp,  0.0071717938_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 48
   & -0.0469095498_wp,  0.0138235064_wp, -0.0376002167_wp,  0.0000000000_wp, & ! 49
   & -0.0895346718_wp, -0.0462564130_wp,  0.0096937153_wp,  0.0000000000_wp, & ! 50
   & -0.2263088378_wp, -0.1114816528_wp,  0.0031197144_wp,  0.0000000000_wp, & ! 51
   & -0.3757926475_wp, -0.1893004402_wp, -0.0085271161_wp,  0.0000000000_wp, & ! 52
   & -0.6410301810_wp, -0.2391088385_wp, -0.0382371542_wp,  0.0000000000_wp, & ! 53
   &  0.0264784970_wp,  0.0671733335_wp, -0.0084692508_wp,  0.0000000000_wp, & ! 54
   &  0.2917227117_wp,  0.0000216839_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 55
   &  0.4812034707_wp,  0.0784010992_wp, -0.0976444962_wp,  0.0000000000_wp, & ! 56
   &  0.6105034666_wp,  0.1042348119_wp,  0.0348299314_wp,  0.0000000000_wp, & ! 57
   & -0.0478758796_wp,  0.0650664880_wp,  0.0751791497_wp,  0.0000000000_wp, & ! 58
   & -0.0571049775_wp,  0.0706068675_wp,  0.0622178407_wp,  0.0000000000_wp, & ! 59
   & -0.0626072842_wp,  0.0751209602_wp,  0.0501215759_wp,  0.0000000000_wp, & ! 60
   & -0.0643827998_wp,  0.0786087663_wp,  0.0388903551_wp,  0.0000000000_wp, & ! 61
   & -0.0624315243_wp,  0.0810702856_wp,  0.0285241784_wp,  0.0000000000_wp, & ! 62
   & -0.0567534576_wp,  0.0825055183_wp,  0.0190230458_wp,  0.0000000000_wp, & ! 63
   & -0.0473485998_wp,  0.0829144643_wp,  0.0103869573_wp,  0.0000000000_wp, & ! 64
   & -0.0342169509_wp,  0.0822971236_wp,  0.0026159129_wp,  0.0000000000_wp, & ! 65
   & -0.0173585108_wp,  0.0806534962_wp, -0.0042900874_wp,  0.0000000000_wp, & ! 66
   &  0.0032267204_wp,  0.0779835821_wp, -0.0103310436_wp,  0.0000000000_wp, & ! 67
   &  0.0275387427_wp,  0.0742873813_wp, -0.0155069558_wp,  0.0000000000_wp, & ! 68
   &  0.0555775561_wp,  0.0695648938_wp, -0.0198178238_wp,  0.0000000000_wp, & ! 69
   &  0.0873431607_wp,  0.0638161196_wp, -0.0232636478_wp,  0.0000000000_wp, & ! 70
   &  0.1228355564_wp,  0.0570410587_wp, -0.0258444276_wp,  0.0000000000_wp, & ! 71
   &  0.1549577606_wp,  0.0541566346_wp, -0.0348875545_wp,  0.0000000000_wp, & ! 72
   & -0.1247231471_wp, -0.1057823857_wp, -0.0614243524_wp,  0.0000000000_wp, & ! 73
   & -0.1284387534_wp,  0.0809659363_wp, -0.1094561851_wp,  0.0000000000_wp, & ! 74
   & -0.0972529119_wp,  0.0405718906_wp, -0.0204635155_wp,  0.0000000000_wp, & ! 75
   & -0.0480709149_wp,  0.0491730303_wp, -0.0552971754_wp,  0.0000000000_wp, & ! 76
   &  0.0880408587_wp, -0.0393893555_wp, -0.1453293814_wp,  0.0000000000_wp, & ! 77
   &  0.0462750537_wp,  0.0326255347_wp, -0.1586517966_wp,  0.0000000000_wp, & ! 78
   &  0.0214275307_wp,  0.0502188143_wp, -0.2636075796_wp,  0.0000000000_wp, & ! 79
   & -0.0175624887_wp, -0.0537704039_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 80
   & -0.1678655238_wp,  0.1646421344_wp,  0.0577282620_wp,  0.0000000000_wp, & ! 81
   &  0.0120276945_wp,  0.0364386772_wp,  0.0705430579_wp,  0.0000000000_wp, & ! 82
   & -0.1117634187_wp,  0.0029826540_wp,  0.0181906383_wp,  0.0000000000_wp, & ! 83
   & -0.2089879796_wp, -0.0883716870_wp,  0.0870605050_wp,  0.0000000000_wp, & ! 84
   & -0.7511439644_wp, -0.1546043648_wp,  0.1214907832_wp,  0.0000000000_wp, & ! 85
   & -1.6011085694_wp, -0.1832406276_wp, -0.0142351095_wp,  0.0000000000_wp, & ! 86
   & -0.1069575719_wp, -0.2275650733_wp,  0.0000000000_wp,  0.0000000000_wp, & ! 87
   &  0.1212131032_wp,  0.3964607554_wp, -0.0439467695_wp,  0.0000000000_wp, & ! 88
   &  0.3782472113_wp, -0.0044168747_wp, -0.0252237821_wp,  0.0348447458_wp, & ! 89
   &  0.3210281049_wp,  0.0430933879_wp, -0.0452587414_wp, -0.0227433756_wp, & ! 90
   &  0.2692781314_wp,  0.0832698499_wp, -0.0635800782_wp, -0.0742021172_wp, & ! 91
   &  0.2229972906_wp,  0.1161125113_wp, -0.0801877924_wp, -0.1195314788_wp, & ! 92
   &  0.1821855827_wp,  0.1416213722_wp, -0.0950818840_wp, -0.1587314606_wp, & ! 93
   &  0.1468430076_wp,  0.1597964325_wp, -0.1082623531_wp, -0.1918020625_wp, & ! 94
   &  0.1169695653_wp,  0.1706376922_wp, -0.1197291996_wp, -0.2187432846_wp, & ! 95
   &  0.0925652559_wp,  0.1741451513_wp, -0.1294824235_wp, -0.2395551267_wp, & ! 96
   &  0.0736300793_wp,  0.1703188098_wp, -0.1375220249_wp, -0.2542375890_wp, & ! 97
   &  0.0601640355_wp,  0.1591586678_wp, -0.1438480037_wp, -0.2627906713_wp, & ! 98
   &  0.0521671245_wp,  0.1406647252_wp, -0.1484603599_wp, -0.2652143738_wp, & ! 99
   &  0.0496393463_wp,  0.1148369820_wp, -0.1513590936_wp, -0.2615086964_wp, & ! 100
   &  0.0525807010_wp,  0.0816754382_wp, -0.1525442047_wp, -0.2516736392_wp, & ! 101
   &  0.0609911885_wp,  0.0411800938_wp, -0.1520156932_wp, -0.2357092020_wp, & ! 102
   &  0.0748708088_wp, -0.0066490511_wp, -0.1497735592_wp, -0.2136153850_wp],& ! 103
   & shape(p_ceh_kcn_en))

   !> Interaction type- and atom-wise resolved scal. fact. 
   !> for overlap mat. elements of sigma, pi, and delta type
   real(wp), parameter :: p_ceh_h0k(3, max_elem) = reshape([&
   &  2.3389142067_wp,  0.0000000000_wp,  3.0000000000_wp, & ! 1
   &  2.6975918527_wp,  0.0000000000_wp,  3.0000000000_wp, & ! 2
   &  3.8887277659_wp,  4.5575413026_wp,  3.0000000000_wp, & ! 3
   &  3.2747423901_wp,  2.5771200965_wp,  3.0000000000_wp, & ! 4
   &  2.4851671693_wp,  2.1372361448_wp,  3.0000000000_wp, & ! 5
   &  2.0268127106_wp,  2.4063222414_wp,  3.0000000000_wp, & ! 6
   &  1.8320642611_wp,  2.1423741348_wp,  3.0000000000_wp, & ! 7
   &  2.7125671056_wp,  2.8961020176_wp,  3.0000000000_wp, & ! 8
   &  2.8153076393_wp,  2.3315043826_wp,  3.0000000000_wp, & ! 9
   &  2.7474399881_wp,  1.8980976242_wp,  3.0000000000_wp, & ! 10
   &  4.2494086260_wp,  5.6922166791_wp,  3.0000000000_wp, & ! 11
   &  4.8126807597_wp,  3.8331119997_wp,  3.0000000000_wp, & ! 12
   &  3.1143029310_wp,  2.5620894150_wp,  3.0000000000_wp, & ! 13
   &  2.4954358063_wp,  2.3182616364_wp,  3.0000000000_wp, & ! 14
   &  2.5976255455_wp,  2.6941772273_wp,  3.0000000000_wp, & ! 15
   &  2.6093723871_wp,  2.8146010238_wp,  3.0000000000_wp, & ! 16
   &  2.9835995083_wp,  2.9827143007_wp,  3.0000000000_wp, & ! 17
   &  2.3678782294_wp,  2.1729346735_wp,  3.0000000000_wp, & ! 18
   &  4.5202875437_wp,  7.2649219185_wp,  3.0000000000_wp, & ! 19
   &  5.2800682038_wp,  3.2548236597_wp,  3.0000000000_wp, & ! 20
   &  5.3577592942_wp,  1.8849683766_wp,  3.0000000000_wp, & ! 21
   &  4.1552644850_wp,  2.3948483540_wp,  3.0000000000_wp, & ! 22
   &  3.4101455952_wp,  2.1373008856_wp,  3.0000000000_wp, & ! 23
   &  2.2475055752_wp,  3.0778360089_wp,  3.0000000000_wp, & ! 24
   &  1.7997250246_wp,  1.9422198793_wp,  3.0000000000_wp, & ! 25
   &  2.3972266594_wp,  2.8837938775_wp,  3.0000000000_wp, & ! 26
   &  2.2398464916_wp,  2.6677676693_wp,  3.0000000000_wp, & ! 27
   &  3.4521227798_wp,  4.4413743908_wp,  3.0000000000_wp, & ! 28
   &  3.2441515878_wp,  4.6756618178_wp,  3.0000000000_wp, & ! 29
   &  3.3377932289_wp,  4.2461011050_wp,  3.0000000000_wp, & ! 30
   &  2.9729952454_wp,  2.8566424437_wp,  3.0000000000_wp, & ! 31
   &  2.7770317484_wp,  2.8472480139_wp,  3.0000000000_wp, & ! 32
   &  2.5259029448_wp,  3.0569179271_wp,  3.0000000000_wp, & ! 33
   &  2.4760075727_wp,  2.9477705449_wp,  3.0000000000_wp, & ! 34
   &  2.8044684907_wp,  3.0836543266_wp,  3.0000000000_wp, & ! 35
   &  1.8775967434_wp,  2.4249535478_wp,  3.0000000000_wp, & ! 36
   &  4.3457987323_wp, 10.8313533220_wp,  3.0000000000_wp, & ! 37
   &  4.9448105347_wp,  2.6403897663_wp,  3.0000000000_wp, & ! 38
   &  4.7318273993_wp,  2.4517542884_wp,  3.0000000000_wp, & ! 39
   &  3.6261177423_wp,  2.1702194229_wp,  3.0000000000_wp, & ! 40
   &  2.6856482045_wp,  2.1971921861_wp,  3.0000000000_wp, & ! 41
   &  2.3896559903_wp,  2.3402279356_wp,  3.0000000000_wp, & ! 42
   &  1.6143917555_wp,  1.6500357343_wp,  3.0000000000_wp, & ! 43
   &  2.1889506128_wp,  2.4037485251_wp,  3.0000000000_wp, & ! 44
   &  2.4315153044_wp,  2.9206851807_wp,  3.0000000000_wp, & ! 45
   &  2.6229781104_wp,  3.4941956638_wp,  3.0000000000_wp, & ! 46
   &  2.9373777311_wp,  4.3265354012_wp,  3.0000000000_wp, & ! 47
   &  3.4816766031_wp,  4.7645118509_wp,  3.0000000000_wp, & ! 48
   &  3.2895912971_wp,  3.3764871256_wp,  3.0000000000_wp, & ! 49
   &  2.6968148126_wp,  2.8805889817_wp,  3.0000000000_wp, & ! 50
   &  2.6180768813_wp,  2.6392519555_wp,  3.0000000000_wp, & ! 51
   &  2.2783789173_wp,  3.0980358429_wp,  3.0000000000_wp, & ! 52
   &  2.4180958199_wp,  2.9952630283_wp,  3.0000000000_wp, & ! 53
   &  2.2538059706_wp,  2.7421983831_wp,  3.0000000000_wp, & ! 54
   &  4.4617284446_wp,  8.9961592805_wp,  3.0000000000_wp, & ! 55
   &  4.0664557839_wp,  3.8019596202_wp,  3.0000000000_wp, & ! 56
   &  4.4405519370_wp,  2.1838840855_wp,  3.0000000000_wp, & ! 57
   &  3.4833140718_wp,  2.1387148845_wp,  3.0000000000_wp, & ! 58
   &  4.1248525492_wp,  2.1023927023_wp,  3.0000000000_wp, & ! 59
   &  4.6622461946_wp,  2.0795202498_wp,  3.0000000000_wp, & ! 60
   &  5.0954950079_wp,  2.0700975268_wp,  3.0000000000_wp, & ! 61
   &  5.4245989890_wp,  2.0741245334_wp,  3.0000000000_wp, & ! 62
   &  5.6495581381_wp,  2.0916012697_wp,  3.0000000000_wp, & ! 63
   &  5.7703724551_wp,  2.1225277355_wp,  3.0000000000_wp, & ! 64
   &  5.7870419400_wp,  2.1669039309_wp,  3.0000000000_wp, & ! 65
   &  5.6995665928_wp,  2.2247298559_wp,  3.0000000000_wp, & ! 66
   &  5.5079464135_wp,  2.2960055106_wp,  3.0000000000_wp, & ! 67
   &  5.2121814021_wp,  2.3807308948_wp,  3.0000000000_wp, & ! 68
   &  4.8122715586_wp,  2.4789060086_wp,  3.0000000000_wp, & ! 69
   &  4.3082168831_wp,  2.5905308520_wp,  3.0000000000_wp, & ! 70
   &  3.7000173754_wp,  2.7156054250_wp,  3.0000000000_wp, & ! 71
   &  3.7126866072_wp,  2.2746161525_wp,  3.0000000000_wp, & ! 72
   &  2.6386761966_wp,  1.9344337246_wp,  3.0000000000_wp, & ! 73
   &  1.8688653188_wp,  2.4511101852_wp,  3.0000000000_wp, & ! 74
   &  1.7572256028_wp,  2.8334308963_wp,  3.0000000000_wp, & ! 75
   &  2.5340163249_wp,  4.1767892642_wp,  3.0000000000_wp, & ! 76
   &  2.2143197689_wp,  2.5627224289_wp,  3.0000000000_wp, & ! 77
   &  2.7787977634_wp,  3.9648388130_wp,  3.0000000000_wp, & ! 78
   &  2.8058458314_wp,  5.2182463270_wp,  3.0000000000_wp, & ! 79
   &  3.1660750781_wp,  5.7964397126_wp,  3.0000000000_wp, & ! 80
   &  3.2155973091_wp,  4.5053410884_wp,  3.0000000000_wp, & ! 81
   &  3.2018924820_wp,  4.4509420097_wp,  3.0000000000_wp, & ! 82
   &  2.8928576460_wp,  2.9350163416_wp,  3.0000000000_wp, & ! 83
   &  2.2044387551_wp,  3.0400695175_wp,  3.0000000000_wp, & ! 84
   &  2.7409639842_wp,  3.9799405892_wp,  3.0000000000_wp, & ! 85
   &  1.9058778779_wp,  2.9340475712_wp,  3.0000000000_wp, & ! 86
   &  6.6080935677_wp,  3.2501733955_wp,  3.0000000000_wp, & ! 87
   &  4.6593663334_wp,  2.5195885846_wp,  3.0000000000_wp, & ! 88
   &  4.3409778645_wp,  2.2052183716_wp,  3.0000000000_wp, & ! 89
   &  3.8889462796_wp,  1.8583394763_wp,  3.0000000000_wp, & ! 90
   &  3.4932413298_wp,  1.5832447919_wp,  3.0000000000_wp, & ! 91
   &  3.1538630152_wp,  1.3799343185_wp,  3.0000000000_wp, & ! 92
   &  2.8708113357_wp,  1.2484080561_wp,  3.0000000000_wp, & ! 93
   &  2.6440862913_wp,  1.1886660046_wp,  3.0000000000_wp, & ! 94
   &  2.4736878821_wp,  1.2007081640_wp,  3.0000000000_wp, & ! 95
   &  2.3596161081_wp,  1.2845345344_wp,  3.0000000000_wp, & ! 96
   &  2.3018709692_wp,  1.4401451157_wp,  3.0000000000_wp, & ! 97
   &  2.3004524655_wp,  1.6675399080_wp,  3.0000000000_wp, & ! 98
   &  2.3553605968_wp,  1.9667189113_wp,  3.0000000000_wp, & ! 99
   &  2.4665953634_wp,  2.3376821255_wp,  3.0000000000_wp, & ! 100
   &  2.6341567651_wp,  2.7804295506_wp,  3.0000000000_wp, & ! 101
   &  2.8580448019_wp,  3.2949611867_wp,  3.0000000000_wp, & ! 102
   &  3.1382594739_wp,  3.8812770338_wp,  3.0000000000_wp],& ! 103
   & shape(p_ceh_h0k))

   !> Scaling factor for total charge in the estimate of atomic charges
   real(wp), parameter :: p_ceh_total_to_q = 0.2_wp

   !> Scaling factor for the electronegativity-weighted CN in the estimate of atomic charges 
   real(wp), parameter :: p_ceh_en_to_q(max_elem) = [&
   & -0.3821678086_wp, -1.5963362591_wp, -0.4777408938_wp, -0.3524644693_wp, & ! 1-4
   & -0.4953354727_wp, -0.3190181251_wp, -0.3983889327_wp, -0.3835108436_wp, & ! 5-8
   & -0.7389560636_wp, -0.2149332110_wp, -0.5685022535_wp, -0.4770245629_wp, & ! 9-12
   & -0.4184546059_wp, -0.4533174643_wp, -0.5416802220_wp, -0.6565502477_wp, & ! 13-16
   & -0.7486642133_wp, -1.4866862644_wp, -0.7487045571_wp, -0.4389360460_wp, & ! 17-20
   & -0.4435794463_wp, -0.3492097512_wp, -0.2559240098_wp,  0.0587270735_wp, & ! 21-24
   &  0.0847762066_wp,  0.3777914872_wp,  0.0126778206_wp, -0.4037708741_wp, & ! 25-28
   & -0.0430518102_wp, -0.4333136605_wp, -0.3662244390_wp, -0.4692572104_wp, & ! 29-32
   & -0.6038995589_wp, -0.6553758078_wp, -0.7806287786_wp, -0.9395417764_wp, & ! 33-36
   & -0.6444073809_wp, -0.4305706057_wp, -0.3291196886_wp, -0.2715887054_wp, & ! 37-40
   & -0.2948608196_wp,  0.0885864677_wp,  0.0564408774_wp,  0.1263580424_wp, & ! 41-44
   &  0.0210181971_wp,  0.0760835299_wp,  0.0240586477_wp, -0.4813289933_wp, & ! 45-48
   & -0.3516913323_wp, -0.5093780930_wp, -0.7474566958_wp, -0.6839190137_wp, & ! 49-52
   & -0.6667804803_wp, -0.1515550540_wp, -0.3791910542_wp, -0.1748587545_wp, & ! 53-56
   & -0.2879748107_wp, -0.2793073041_wp, -0.3681729955_wp, -0.4428978307_wp, & ! 57-60
   & -0.5034818095_wp, -0.5499249321_wp, -0.5822271984_wp, -0.6003886085_wp, & ! 61-64
   & -0.6044091623_wp, -0.5942888598_wp, -0.5700277010_wp, -0.5316256860_wp, & ! 65-68
   & -0.4790828147_wp, -0.4123990871_wp, -0.3315745032_wp, -0.2714861258_wp, & ! 69-72
   & -0.4473568861_wp, -0.3710050436_wp, -0.1260547444_wp, -0.0856067227_wp, & ! 73-76
   & -0.2254599020_wp, -0.2592342254_wp, -0.2052677849_wp, -0.4310509793_wp, & ! 77-80
   & -0.0873656217_wp, -0.3243816707_wp, -0.6832381802_wp, -0.8365324836_wp, & ! 81-84
   & -0.9061539639_wp, -1.8061239458_wp,  0.7480980761_wp, -0.1418163077_wp, & ! 85-88
   & -0.3881315281_wp, -0.4484745291_wp, -0.4972162562_wp, -0.5343567093_wp, & ! 89-92
   & -0.5598958886_wp, -0.5738337939_wp, -0.5761704254_wp, -0.5669057829_wp, & ! 93-96
   & -0.5460398665_wp, -0.5135726762_wp, -0.4695042120_wp, -0.4138344739_wp, & ! 97-100
   & -0.3465634619_wp, -0.2676911759_wp, -0.1772176161_wp]                     ! 101-103

   !> Exponent in the Klopman-Ohno/Mataga-Nishimoto electrostatics kernel
   !> CEH uses the pure Mataga-Nishimoto kernel (gexp=1)
   real(wp), parameter :: gexp = 1.0_wp

   !> Hubbard U values for the second order ES correction in CEH. 
   !> Values for Cs and Ba are used for Fr and Ra, respectively.
   !> Lanthanoids and actinoids are based on 3-point quadratic interpolation 1, 7, and 14 f-el.
   !> Actinoid values are scaled from the lanthanoid series by a constant factor of 2.55
   real(wp), parameter :: p_ceh_hubbard(max_elem) = [&
   &  0.47259288_wp,  0.92203391_wp,  0.17452888_wp,  0.25700733_wp, & ! 1-4
   &  0.33949086_wp,  0.42195412_wp,  0.50438193_wp,  0.58691863_wp, & ! 5-8
   &  0.66931351_wp,  0.75191607_wp,  0.17964105_wp,  0.22157276_wp, & ! 9-12
   &  0.26348578_wp,  0.30539645_wp,  0.34734014_wp,  0.38924725_wp, & ! 13-16
   &  0.43115670_wp,  0.47308269_wp,  0.17105469_wp,  0.20276244_wp, & ! 17-20
   &  0.21007322_wp,  0.21739647_wp,  0.22471039_wp,  0.23201501_wp, & ! 21-24
   &  0.23933969_wp,  0.24665638_wp,  0.25398255_wp,  0.26128863_wp, & ! 25-28
   &  0.26859476_wp,  0.27592565_wp,  0.30762999_wp,  0.33931580_wp, & ! 29-32
   &  0.37235985_wp,  0.40273549_wp,  0.43445776_wp,  0.46611708_wp, & ! 33-36
   &  0.15585079_wp,  0.18649324_wp,  0.19356210_wp,  0.20063311_wp, & ! 37-40
   &  0.20770522_wp,  0.21477254_wp,  0.22184614_wp,  0.22891872_wp, & ! 41-44
   &  0.23598621_wp,  0.24305612_wp,  0.25013018_wp,  0.25719937_wp, & ! 45-48
   &  0.28784780_wp,  0.31848673_wp,  0.34912431_wp,  0.37976593_wp, & ! 49-52
   &  0.41040808_wp,  0.44105777_wp,  0.05019332_wp,  0.06762570_wp, & ! 53-56
   &  0.08504445_wp,  0.10247736_wp,  0.11991105_wp,  0.13732772_wp, & ! 57-60
   &  0.15476297_wp,  0.17218265_wp,  0.18961288_wp,  0.20704760_wp, & ! 61-64
   &  0.22446752_wp,  0.24189645_wp,  0.25932503_wp,  0.27676094_wp, & ! 65-68
   &  0.29418231_wp,  0.31159587_wp,  0.32902274_wp,  0.34592298_wp, & ! 69-72
   &  0.36388048_wp,  0.38130586_wp,  0.39877476_wp,  0.41614298_wp, & ! 73-76
   &  0.43364510_wp,  0.45104014_wp,  0.46848986_wp,  0.48584550_wp, & ! 77-80
   &  0.12526730_wp,  0.14268677_wp,  0.16011615_wp,  0.17755889_wp, & ! 81-84
   &  0.19497557_wp,  0.21240778_wp,  0.05019332_wp,  0.06762570_wp, & ! 85-88 Cs,Ba values for Fr,Ra 
   &  0.21686335_wp,  0.26131727_wp,  0.30577318_wp,  0.35018569_wp, & ! 89-92
   &  0.39464557_wp,  0.43906576_wp,  0.48351284_wp,  0.52797138_wp, & ! 93-96
   &  0.57239218_wp,  0.61683595_wp,  0.66127883_wp,  0.70574040_wp, & ! 97-100
   &  0.75016489_wp,  0.79456947_wp,  0.83900799_wp]                   ! 101-103 

   !> Derivatives of the Hubbard U values for the 3rd order onsite ES correction in CEH
   real(wp), parameter :: p_ceh_hubbard_derivs(max_elem) = [&
   &  0.8936213810_wp, -0.3936567743_wp, -0.7726174171_wp, -0.2849896764_wp, & ! 1-4
   &  0.0126634714_wp, -0.0082561791_wp,  0.0992949802_wp, -0.0267387652_wp, & ! 5-8
   & -0.0632999086_wp, -1.0106414497_wp, -0.3492075197_wp, -0.3191627473_wp, & ! 9-12
   &  0.0467483747_wp, -0.0920002125_wp, -0.0728788864_wp, -0.0213909690_wp, & ! 13-16
   & -0.0206065548_wp, -0.0432378706_wp, -0.0686554093_wp, -0.1671301006_wp, & ! 17-20
   &  0.0852629923_wp, -0.0800504596_wp, -0.2529244991_wp, -0.2606578074_wp, & ! 21-24
   & -0.1734379247_wp, -0.2077661412_wp, -0.2424646149_wp, -0.3276441297_wp, & ! 25-28
   & -0.2606642504_wp,  0.1382131675_wp,  0.0338408971_wp, -0.0145059965_wp, & ! 29-32
   & -0.0246097401_wp, -0.0225681829_wp,  0.1263933845_wp, -0.1272473418_wp, & ! 33-36
   &  0.0026465431_wp, -0.3372527639_wp,  0.1040774701_wp,  0.2083790384_wp, & ! 37-40
   & -0.7505598533_wp,  0.5285653781_wp,  0.2472213017_wp, -0.6999052793_wp, & ! 41-44
   & -0.6527087042_wp, -0.3872393306_wp, -0.7380122563_wp, -0.0818070299_wp, & ! 45-48
   & -0.0667160243_wp, -0.0341990432_wp,  0.0485366599_wp, -0.0325375796_wp, & ! 49-52
   &  0.0475767797_wp, -0.1675805377_wp,  0.8623109992_wp, -0.5512198287_wp, & ! 53-56
   &  0.1328669432_wp, -0.3483361179_wp, -0.2453274608_wp, -0.1591099170_wp, & ! 57-60
   & -0.0896834866_wp, -0.0370481696_wp, -0.0012039660_wp,  0.0178491243_wp, & ! 61-64
   &  0.0201111012_wp,  0.0055819647_wp, -0.0257382851_wp, -0.0738496484_wp, & ! 65-68
   & -0.1387521250_wp, -0.2204457150_wp, -0.3189304183_wp,  0.2049216732_wp, & ! 69-72
   & -0.0367179966_wp, -0.0523922395_wp, -0.1269668699_wp, -0.2393009746_wp, & ! 73-76
   & -0.1291882239_wp, -0.8333463793_wp, -0.1890458247_wp, -0.0338279364_wp, & ! 77-80
   &  1.0558886510_wp,  0.0309888899_wp, -0.0301855544_wp, -0.0233409549_wp, & ! 81-84
   & -0.0044103287_wp, -0.0097427070_wp,  1.2088288616_wp, -0.2663782186_wp, & ! 85-88
   & -0.0105348872_wp, -0.0164469354_wp, -0.0191418341_wp, -0.0186195834_wp, & ! 89-92
   & -0.0148801832_wp, -0.0079236335_wp,  0.0022500657_wp,  0.0156409143_wp, & ! 93-96
   &  0.0322489124_wp,  0.0520740600_wp,  0.0751163571_wp,  0.1013758036_wp, & ! 97-100
   &  0.1308523996_wp,  0.1635461451_wp,  0.1994570401_wp]                     ! 101-103

   !> Empirical atomic radii for calculation of the coordination number
   real(wp), parameter :: ceh_cov_radii(max_elem) = 0.5_wp * [&
   &  2.4040551903_wp,  1.8947380542_wp,  3.4227634078_wp,  3.5225408137_wp, & ! 1-4
   &  3.6150631704_wp,  2.8649682108_wp,  2.4695867541_wp,  2.3533691180_wp, & ! 5-8
   &  2.4992147462_wp,  3.3390521781_wp,  4.4665909451_wp,  4.3877250907_wp, & ! 9-12
   &  4.6647077385_wp,  4.2086223530_wp,  4.4750280107_wp,  4.2847281423_wp, & ! 13-16
   &  3.8560304959_wp,  3.9017061017_wp,  5.2392192639_wp,  5.1872031383_wp, & ! 17-20
   &  5.6698338669_wp,  4.7977447316_wp,  4.5861091144_wp,  4.6297520548_wp, & ! 21-24
   &  4.3156951613_wp,  4.0198277420_wp,  4.2368476457_wp,  4.2804501894_wp, & ! 25-28
   &  3.7106325558_wp,  3.9338789689_wp,  4.6890656513_wp,  4.3914474133_wp, & ! 29-32
   &  4.8305138771_wp,  4.5234214298_wp,  4.2953425631_wp,  4.1463855171_wp, & ! 33-36
   &  5.4646907443_wp,  5.9245671473_wp,  5.6482501066_wp,  5.4206701211_wp, & ! 37-40
   &  5.0914446806_wp,  4.9547118756_wp,  4.7560109429_wp,  4.4628560808_wp, & ! 41-44
   &  4.5224914695_wp,  4.2251916534_wp,  3.9110698488_wp,  4.5677637406_wp, & ! 45-48
   &  5.5315121057_wp,  4.9785372098_wp,  5.2144462735_wp,  5.0623069704_wp, & ! 49-52
   &  4.9304049014_wp,  4.1155402329_wp,  5.9110572848_wp,  5.5478044574_wp, & ! 53-56
   &  5.7948986136_wp,  5.2692712122_wp,  5.5270787388_wp,  5.7379529771_wp, & ! 57-60
   &  5.9018939273_wp,  6.0189015892_wp,  6.0889759630_wp,  6.1121170485_wp, & ! 61-64
   &  6.0883248458_wp,  6.0175993550_wp,  5.8999405759_wp,  5.7353485087_wp, & ! 65-68
   &  5.5238231532_wp,  5.2653645096_wp,  4.9599725777_wp,  4.8784195370_wp, & ! 69-72
   &  5.6188905210_wp,  5.0560398438_wp,  4.4541146688_wp,  4.4499347179_wp, & ! 73-76
   &  4.7852969991_wp,  4.4148540875_wp,  4.0416460175_wp,  5.0533987752_wp, & ! 77-80
   &  4.6572211182_wp,  4.8001161773_wp,  5.0989704974_wp,  5.1523321147_wp, & ! 81-84
   &  4.8320368306_wp,  4.9114720970_wp,  5.6081911873_wp,  5.4940127402_wp, & ! 85-88
   &  7.0543206078_wp,  6.5470179170_wp,  6.0976511257_wp,  5.7062202339_wp, & ! 89-92
   &  5.3727252415_wp,  5.0971661486_wp,  4.8795429552_wp,  4.7198556612_wp, & ! 93-96
   &  4.6181042667_wp,  4.5742887717_wp,  4.5884091762_wp,  4.6604654801_wp, & ! 97-100
   &  4.7904576835_wp,  4.9783857864_wp,  5.2242497887_wp]                     ! 101-103

   !> Empirical Pauling EN normalized to EN(F)=1 as start values
   !> Used for EN-scaled Coordination number in CEH
   real(wp), parameter :: pauling_en_ceh(max_elem) = (1e0_wp/3.98e0_wp) * [&
   &  1.9435211923_wp,  3.6116085622_wp,  2.4630915335_wp,  2.0658837656_wp, & ! 1-4
   &  2.3619778807_wp,  2.9484294262_wp,  3.8753937411_wp,  4.6235054741_wp, & ! 5-8
   &  3.9800000000_wp,  3.6615073276_wp,  2.3578254072_wp,  2.4225832022_wp, & ! 9-12
   &  2.1120078826_wp,  2.4607564741_wp,  2.7410779326_wp,  3.3517034720_wp, & ! 13-16
   &  4.1093492601_wp,  3.7979559518_wp,  2.4147937668_wp,  2.1974781961_wp, & ! 17-20
   &  2.1128960348_wp,  2.3993988439_wp,  2.6122720364_wp,  2.0915846800_wp, & ! 21-24
   &  2.5054632371_wp,  3.3721970712_wp,  2.7766991529_wp,  3.0232154392_wp, & ! 25-28
   &  3.2015022468_wp,  2.7135609794_wp,  2.0771853881_wp,  2.5529198911_wp, & ! 29-32
   &  2.7315021736_wp,  3.3166077566_wp,  4.0540213414_wp,  3.3533770560_wp, & ! 33-36
   &  2.7281582909_wp,  2.2609009562_wp,  2.2540471516_wp,  2.2653753348_wp, & ! 37-40
   &  2.5842800981_wp,  2.1508819371_wp,  2.7693219880_wp,  3.0179026228_wp, & ! 41-44
   &  3.2279256063_wp,  3.6606062056_wp,  3.6748385722_wp,  2.6031649837_wp, & ! 45-48
   &  1.9301205946_wp,  2.4861062492_wp,  2.6447840966_wp,  3.2456229156_wp, & ! 49-52
   &  3.7831065186_wp,  4.3242426632_wp,  2.8372704556_wp,  2.7274387249_wp, & ! 53-56
   &  2.2647941586_wp,  2.4969001529_wp,  2.4447666833_wp,  2.4012383066_wp, & ! 57-60
   &  2.3663150226_wp,  2.3399968315_wp,  2.3222837332_wp,  2.3131757278_wp, & ! 61-64
   &  2.3126728152_wp,  2.3207749954_wp,  2.3374822684_wp,  2.3627946343_wp, & ! 65-68
   &  2.3967120930_wp,  2.4392346445_wp,  2.4903622888_wp,  2.4502060958_wp, & ! 69-72
   &  2.3364159315_wp,  2.4167832435_wp,  2.8091245086_wp,  2.9876741666_wp, & ! 73-76
   &  2.6173344873_wp,  3.5705156606_wp,  3.5497522101_wp,  1.8146455761_wp, & ! 77-80
   &  2.9211076716_wp,  3.1182156253_wp,  2.7394661670_wp,  3.2420409515_wp, & ! 81-84
   &  4.4238424833_wp,  3.7545504524_wp,  3.5942173046_wp,  2.5050207941_wp, & ! 85-88
   &  1.7674300866_wp,  2.2358342835_wp,  2.6360616126_wp,  2.9681120739_wp, & ! 89-92
   &  3.2319856674_wp,  3.4276823931_wp,  3.5552022510_wp,  3.6145452411_wp, & ! 93-96
   &  3.6057113634_wp,  3.5287006179_wp,  3.3835130046_wp,  3.1701485235_wp, & ! 97-100
   &  2.8886071747_wp,  2.5388889580_wp,  2.1209938735_wp]                     ! 101-103

   !> Angular momentum-specific scaling factors for H0 
   real(wp), parameter   :: kll(1:4) = [0.6379_wp, 0.9517_wp, 1.18_wp, 2.84_wp]

   !> Conversion constant
   real(wp), parameter   :: kt = 3.166808578545117e-06_wp

   !> Specification of the CEH hamiltonian
   type, extends(tb_h0spec) :: ceh_h0spec
   contains
      !> Generator for the self energy / atomic levels of the Hamiltonian
      procedure :: get_selfenergy
      !> Generator for the coordination number dependent shift of the self energy
      procedure :: get_cnshift
      !> Generator for the coordination number dependent shift of the self energy
      procedure :: get_cnenshift
      !> Generator for the enhancement factor to for scaling Hamiltonian elements
      procedure :: get_hscale
      !> Generator for the polynomial parameters for the distant dependent scaling
      procedure :: get_shpoly
      !> Generator for the reference occupation numbers of the atoms
      procedure :: get_reference_occ
      !> Generator for the diatomic frame scaling factors
      procedure :: get_diat_scale
   end type ceh_h0spec

   interface ceh_h0spec
      module procedure :: new_ceh_h0spec
   end interface ceh_h0spec

contains


   subroutine new_ceh_calculator(calc, mol, error)
      !> Instance of the CEH evaluator
      type(xtb_calculator), intent(out) :: calc
      !> Molecular structure data
      type(structure_type), intent(in)  :: mol
      !> Error handling
      type(error_type), allocatable, intent(out) :: error

      !> Check if all atoms of mol%nat are supported (Z <= 103)
      if (any(mol%num > max_elem)) then
         call fatal_error(error, "No support for elements with Z >" // format_string(max_elem, '(i0)') // ".")
         return
      end if

      call add_ceh_basis(calc, mol)
      call add_ncoord(calc, mol)
      call add_ncoord_en(calc, mol)
      call add_hamiltonian(calc, mol)
      call add_coulomb(calc, mol)

   end subroutine new_ceh_calculator


   subroutine add_ceh_basis(calc, mol)
      !> Instance of the CEH evaluator
      type(xtb_calculator), intent(inout) :: calc
      !> Molecular structure data
      type(structure_type), intent(in) :: mol

      integer :: isp, izp, ish, stat, ng, il
      integer, allocatable :: nsh_id(:)
      integer :: ang_idx(0:3), ortho(max_shell)
      type(cgto_type), allocatable :: cgto(:, :)

      nsh_id = nshell(mol%num)
      allocate(cgto(maxval(nsh_id), mol%nid))
      do isp = 1, mol%nid
         ang_idx = 0
         ortho = 0
         izp = mol%num(isp)
         do ish = 1, nsh_id(isp)
            il = ang_shell(ish, izp)
            ng = number_of_primitives(ish, izp)
            if (ang_idx(il) > 0) then
               ortho(ish) = ang_idx(il)
            else
               ang_idx(il) = ish
            end if
            call slater_to_gauss(ng, principal_quantum_number(ish, izp), il, &
            & slater_exponent(ish, izp), cgto(ish, isp), .true., stat)
         end do
      end do

      call new_basis(calc%bas, mol, nsh_id, cgto, 1.0_wp)

   end subroutine add_ceh_basis


   subroutine add_ncoord(calc, mol)
      !> Instance of the CEH evaluator
      type(xtb_calculator), intent(inout) :: calc
      !> Molecular structure data
      type(structure_type), intent(in) :: mol

      call new_ncoord(calc%ncoord, mol, cn_type="erf", &
      & rcov=ceh_cov_radii(mol%num))
   end subroutine add_ncoord


   subroutine add_ncoord_en(calc, mol)
      !> Instance of the CEH evaluator
      type(xtb_calculator), intent(inout) :: calc
      !> Molecular structure data
      type(structure_type), intent(in) :: mol

      call new_ncoord(calc%ncoord_en, mol, cn_type="erf_en", &
      & rcov=ceh_cov_radii(mol%num), en=pauling_en_ceh(mol%num))
   end subroutine add_ncoord_en


   subroutine add_hamiltonian(calc, mol)
      !> Instance of the CEH evaluator
      type(xtb_calculator), intent(inout) :: calc
      !> Molecular structure data
      type(structure_type), intent(in) :: mol

      call new_hamiltonian(calc%h0, mol, calc%bas, new_ceh_h0spec(mol))

   end subroutine add_hamiltonian

   subroutine add_coulomb(calc, mol)
      !> Instance of the xTB evaluator
      type(xtb_calculator), intent(inout) :: calc
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
   
      real(wp), allocatable :: hardness(:, :), hubbard_derivs(:, :)
      type(effective_coulomb), allocatable :: es2

      allocate(calc%coulomb)
      allocate(es2)
      call get_atomic_hardness(mol, calc%bas, hardness)
      call new_effective_coulomb(es2, mol, gexp, hardness, arithmetic_average) 
      call move_alloc(es2, calc%coulomb%es2)
   
      allocate(calc%coulomb%es3)
      call get_hubbard_derivs(mol, calc%bas, hubbard_derivs)
      call new_onsite_thirdorder(calc%coulomb%es3, mol, hubbard_derivs)
   
   end subroutine add_coulomb

   pure function new_ceh_h0spec(mol) result(self)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Instance of the Hamiltonian specification
      type(ceh_h0spec) :: self

   end function new_ceh_h0spec


   !> Generator for the enhancement factor to for scaling Hamiltonian elements
   subroutine get_hscale(self, mol, bas, hscale)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Scaling parameters for the Hamiltonian elements
      real(wp), intent(out) :: hscale(:, :, :, :)

      integer :: isp, jsp, izp, jzp, ish, jsh, il, jl
      real(wp) :: km

      hscale(:, :, :, :) = 0.0_wp

      do isp = 1, mol%nid
         izp = mol%num(isp)
         do jsp = 1, mol%nid
            jzp = mol%num(jsp)
            do ish = 1, bas%nsh_id(isp)
               il = bas%cgto(ish, isp)%ang
               do jsh = 1, bas%nsh_id(jsp)
                  jl = bas%cgto(jsh, jsp)%ang
                  km = ( kll(il + 1) + kll(jl + 1) )
                  hscale(jsh, ish, jsp, isp) = 0.5_wp * km
               end do
            end do
         end do
      end do

   end subroutine get_hscale


   !> Generator for the self energy / atomic levels of the Hamiltonian
   subroutine get_selfenergy(self, mol, bas, selfenergy)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Self energy / atomic levels
      real(wp), intent(out) :: selfenergy(:, :)

      integer :: isp, izp, ish, iat

      selfenergy(:, :) = 0.0_wp

      do isp = 1, mol%nid
         izp = mol%num(isp)
         do ish = 1, bas%nsh_id(isp)
            iat = bas%ao2at(ish)
            selfenergy(ish, isp) = p_ceh_selfenergy(ish, izp)
         end do
      end do

   end subroutine get_selfenergy


   !> Generator of the coordination number dependent shift of the self energy
   subroutine get_cnshift(self, mol, bas, kcn)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Coordination number dependent shift
      real(wp), intent(out) :: kcn(:, :)

      integer :: isp, izp, ish

      kcn(:, :) = 0.0_wp

      do isp = 1, mol%nid
         izp = mol%num(isp)
         do ish = 1, bas%nsh_id(isp)
            kcn(ish, isp) = p_ceh_kcn(ish, izp)
         end do
      end do

   end subroutine get_cnshift


   !> Generator of the coordination number dependent shift of the self energy
   subroutine get_cnenshift(self, mol, bas, kcn_en)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Coordination number dependent shift
      real(wp), intent(out) :: kcn_en(:, :)

      integer :: isp, izp, ish

      kcn_en(:, :) = 0.0_wp

      do isp = 1, mol%nid
         izp = mol%num(isp)
         do ish = 1, bas%nsh_id(isp)
            kcn_en(ish, isp) = p_ceh_kcn_en(ish, izp)
         end do
      end do

   end subroutine get_cnenshift


   subroutine get_reference_occ(self, mol, bas, refocc)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Reference occupation numbers
      real(wp), intent(out) :: refocc(:, :)
      logical, allocatable  :: valence(:,:)

      integer :: isp, izp, ish, il, mshell
      integer :: ang_idx(0:3)

      allocate(valence(4, mol%nid))
      do isp = 1, mol%nid
         ang_idx = 0
         izp = mol%num(isp)
         do ish = 1, nshell(izp)
            il = ang_shell(ish, izp)
            valence(ish, isp) = ang_idx(il) == 0
            if (valence(ish, isp)) ang_idx(il) = ish
         end do
      end do

      mshell = maxval(bas%nsh_id)
      refocc(:, :) = 0.0_wp
      do isp = 1, mol%nid
         izp = mol%num(isp)
         do ish = 1, bas%nsh_id(isp)
            if (valence(ish, isp)) then
               refocc(ish, isp) = reference_occ(bas%cgto(ish, isp)%ang+1, izp)
            else
               refocc(ish, isp) = 0.0_wp
            end if
         end do
      end do

   end subroutine get_reference_occ


   !> Generator for the polynomial parameters for the distant dependent scaling (Not in CEH)
   subroutine get_shpoly(self, mol, bas, shpoly)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Polynomial parameters for distant dependent scaleing
      real(wp), intent(out) :: shpoly(:, :)

      integer :: isp, izp, ish

      shpoly(:, :) = 0.0_wp

   end subroutine get_shpoly


   !> Stub implementation of the diatomic frame scaling factor generator
   subroutine get_diat_scale(self, mol, bas, ksig, kpi, kdel)
      !> Instance of the Hamiltonian specification
      class(ceh_h0spec), intent(in) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Quadratic partial charge dependent shift
      real(wp), intent(out) :: ksig(:, :)
      !> Quadratic partial charge dependent shift
      real(wp), intent(out) :: kpi(:, :)
      !> Quadratic partial charge dependent shift
      real(wp), intent(out) :: kdel(:, :)

      integer :: isp, izp, jsp, jzp

      ksig(:, :) = 0.0_wp
      kpi(:, :) = 0.0_wp
      kdel(:, :) = 0.0_wp

      do isp = 1, mol%nid
         izp = mol%num(isp)
         do jsp = 1, mol%nid
            jzp = mol%num(jsp)
            ! Geometric mean
            ksig(isp, jsp) = (p_ceh_h0k(1,izp) * p_ceh_h0k(1,jzp))**0.5_wp 
            kpi (isp, jsp) = (p_ceh_h0k(2,izp) * p_ceh_h0k(2,jzp))**0.5_wp 
            kdel(isp, jsp) = (p_ceh_h0k(3,izp) * p_ceh_h0k(3,jzp))**0.5_wp 
         end do
      end do

   end subroutine get_diat_scale


   subroutine get_atomic_hardness(mol, bas, hardness)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Atom resolved hardness parameters
      real(wp), allocatable, intent(out) :: hardness(:, :)

      integer :: isp, izp

      allocate(hardness(1, mol%nid))
      hardness(:, :) = 0.0_wp
      do isp = 1, mol%nid
         izp = mol%num(isp)
         hardness(1, isp) = p_ceh_hubbard(izp)
      end do
   end subroutine get_atomic_hardness


   subroutine get_hubbard_derivs(mol, bas, hubbard_derivs)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set information
      type(basis_type), intent(in) :: bas
      !> Atom resolved Hubbard derivatives
      real(wp), allocatable, intent(out) :: hubbard_derivs(:, :)

      integer :: isp, izp

      ! no shell-resolution for the third order term
      allocate(hubbard_derivs(1, mol%nid)) 
      hubbard_derivs(:, :) = 0.0_wp
      do isp = 1, mol%nid
         izp = mol%num(isp)
         hubbard_derivs(1, isp) = p_ceh_hubbard_derivs(izp)
      end do
   end subroutine get_hubbard_derivs


   !> Build effective charges from the electronegativity-weighted CN
   subroutine get_effective_qat(mol, cn_en, qat, &
      & dcn_endr, dcn_endL, dqatdr, dqatdL)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Electronegativity weighted CN, shape: [nat]
      real(wp), intent(in) :: cn_en(:)
      !> Effective atomic charges, shape: [nat, spin]
      real(wp), intent(out) :: qat(:, :)
      !> Position gradient of cn_en, shape: [3, nat, nat]
      real(wp), intent(in), optional :: dcn_endr(:, :, :)
      !> Lattice vector gradient of cn_en, shape: [3, 3, nat]
      real(wp), intent(in), optional :: dcn_endL(:, :, :)
      !> Position gradient of qat, shape: [3, nat, nat, spin]
      real(wp), intent(out), optional :: dqatdr(:, :, :, :)
      !> Lattice vector gradient of qat, shape: [3, 3, nat, spin]
      real(wp), intent(out), optional :: dqatdL(:, :, :, :)

      integer :: iat, isp, izp, ispin

      qat(:, :) = 0.0_wp
      if (present(dqatdr)) dqatdr(:,:,:,:) = 0.0_wp
      if (present(dqatdL)) dqatdL(:,:,:,:) = 0.0_wp

      if (present(dqatdr) .and. present(dqatdL) .and. &
         & present(dcn_endr) .and. present(dcn_endL)) then
         do ispin = 1, size(qat, 2)
            do iat = 1, size(qat, 1)
               isp = mol%id(iat)
               izp = mol%num(isp)

               qat(iat, ispin) = p_ceh_en_to_q(izp)*cn_en(iat) &
               & + p_ceh_total_to_q * mol%charge/real(mol%nat, wp)
               dqatdr(:,:,iat, ispin) = p_ceh_en_to_q(izp) * dcn_endr(:,:,iat)
               dqatdL(:,:,iat, ispin) = p_ceh_en_to_q(izp) * dcn_endL(:,:,iat)
            end do
         end do
      else
         do ispin = 1, size(qat, 2)
            do iat = 1, size(qat, 1)
               isp = mol%id(iat)
               izp = mol%num(isp)

               qat(iat, ispin) = p_ceh_en_to_q(izp)*cn_en(iat) &
               & + p_ceh_total_to_q * mol%charge/real(mol%nat, wp)
            end do
         end do
      end if 
   
   end subroutine get_effective_qat


end module tblite_ceh_ceh
