/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2026 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_TABS_BASE_HPP
#define TGUI_TABS_BASE_HPP

#include <TGUI/Widget.hpp>
#include <TGUI/Renderers/TabsRenderer.hpp>
#include <TGUI/Text.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Tabs widget
    ///
    /// If you are looking for something that allow holds multiple panels,
    /// displayed one panel at a time then check out the TabContainer class.
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API TabsBase : public Widget
    {
    public:

        using Ptr = std::shared_ptr<TabsBase>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const TabsBase>; //!< Shared constant widget pointer

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        explicit TabsBase(const char* typeName, bool initRenderer);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer that may be shared with other widgets using the same renderer
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD TabsRenderer* getSharedRenderer() override;
        TGUI_NODISCARD const TabsRenderer* getSharedRenderer() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer
        /// @warning After calling this function, the widget has its own copy of the renderer and it will no longer be shared.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD TabsRenderer* getRenderer() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Enables or disables the widget
        /// @param enabled  Is the widget enabled?
        ///
        /// The disabled widget will no longer receive events and thus no longer send callbacks.
        /// All widgets are enabled by default.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setEnabled(bool enabled) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Adds a new tab
        ///
        /// @param text    The text of the tab that will be drawn on top of it
        /// @param select  Do you want the new tab to be selected immediately?
        ///
        /// @return  The index of the tab in the list
        ///
        /// @warning The index returned by this function may no longer be correct when a tab is removed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        std::size_t add(const String& text, bool select = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Inserts a new tab somewhere between the other tabs
        ///
        /// @param index   The index where the tab gets inserted. 0 means before the first tab, 1 means behind the first tab
        /// @param text    The text of the tab that will be drawn on top of it
        /// @param select  Do you want the new tab to be selected immediately?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void insert(std::size_t index, const String& text, bool select = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the text of one of the tabs
        ///
        /// @param index  The index of the tab. The first tab has index 0
        ///
        /// @return The text on the tab or an empty string when index was too high
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD String getText(std::size_t index) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the text of one of the tabs
        ///
        /// @param index  The index of the tab to be changed. The first tab has index 0
        /// @param text   The new text of the tab that will be drawn on top of it
        ///
        /// @return True when text was successfully changed, false when index was too high
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool changeText(std::size_t index, const String& text);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Selects the tab with a given text
        /// @param text  The text of the tab to select
        /// @return Whether a tab was selected, false is returned if tab doesn't exist or is invisible or disabled
        /// @see select(std::size_t)
        ///
        /// If there are multiple tabs with the same text then the first one will be selected.
        /// When false is returned, the selected tab will still be deselected.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool select(const String& text);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Selects the tab with a given index
        /// @param index  The index of the tab to select
        /// @return Whether a tab was selected, false is returned if the index was too high or if tab is invisible or disabled
        /// @see select(String)
        ///
        /// When false is returned, the selected tab will still be deselected.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool select(std::size_t index);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Deselects the selected tab
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void deselect();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes a tab with a given text
        /// @param text  The text on the tab to remove
        /// @return Whether a tab was removed, false is returned when the text didn't match any tab
        /// @see remove(std::size_t)
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool remove(const String& text);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes a tab with a given index
        /// @param index  The index of the tab to remove
        /// @return Whether a tab was removed, false is returned when the index was too high
        /// @see remove(String)
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool remove(std::size_t index);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes all tabs
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void removeAll();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the text that is drawn on the currently selected tab
        ///
        /// @return The text on the tab.
        ///         When no tab is selected then this function returns an empty string
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD String getSelected() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the index of the currently selected tab
        ///
        /// @return The index of the tab.
        ///         When no tab is selected then this function returns -1
        ///
        /// @warning The index returned by this function may no longer be correct when a tab is removed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD int getSelectedIndex() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the index of the tab below the mouse cursor
        /// @return The index of the tab below the mouse, or -1 when the mouse isn't on top of any tab
        /// @since TGUI 1.5
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD int getHoveredIndex() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether a tab is visible
        /// @param index   The index of the tab to show or hide
        /// @param visible Is the tab shown?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTabVisible(std::size_t index, bool visible);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the tab is shown or hidden
        /// @param index  The index of the tab
        /// @return Whether the tab is visible
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getTabVisible(std::size_t index) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether a tab is enabled
        /// @param index   The index of the tab to enable or disable
        /// @param enabled Is the tab enabled?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTabEnabled(std::size_t index, bool enabled);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the tab is enabled or disabled
        /// @param index  The index of the tab
        /// @return Whether the tab is enabled
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getTabEnabled(std::size_t index) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the amount of tabs
        ///
        /// @return Number of tabs
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD std::size_t getTabsCount() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the mouse position (which is relative to the parent widget) lies on top of the widget
        ///
        /// @return Is the mouse on top of the widget?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isMouseOnWidget(Vector2f pos) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rightMousePressed(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseNoLongerOnWidget() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Retrieves a signal based on its name
        ///
        /// @param signalName  Name of the signal
        ///
        /// @return Signal that corresponds to the name
        ///
        /// @throw Exception when the name does not match any signal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Signal& getSignal(String signalName) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Recalculates the size of each tab image.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual void recalculateTabsSize() = 0;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Update the colors of the text that is drawn on the tabs
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateTextColors();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Function called when one of the properties of the renderer is changed
        ///
        /// @param property  Name of the property that was changed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rendererChanged(const String& property) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Saves the widget as a tree node in order to save it to a file
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD std::unique_ptr<DataIO::Node> save(SavingRenderersMap& renderers) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Loads the widget from a tree of nodes
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void load(const std::unique_ptr<DataIO::Node>& node, const LoadingRenderersMap& renderers) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Called when the text size is changed (either by setTextSize or via the renderer)
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateTextSize() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    public:

        SignalString onTabSelect = {"TabSelected"}; //!< A tab that was selected. Optional parameter: selected item
        SignalString onTabRightClick = {"TabRightClicked"}; //!< Right mouse pressed on a tab. Optional parameter: selected item

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        int                m_selectedTab = -1;
        int                m_hoveringTab = -1;

        struct Tab
        {
            bool  visible;
            bool  enabled;
            float width;
            Text  text;
        };
        std::vector<Tab> m_tabs;

        Sprite m_spriteTab;
        Sprite m_spriteTabHover;
        Sprite m_spriteSelectedTab;
        Sprite m_spriteSelectedTabHover;
        Sprite m_spriteDisabledTab;

        // Cached renderer properties
        Borders   m_bordersCached;
        Color     m_borderColorCached;
        Color     m_borderColorHoverCached;
        Color     m_selectedBorderColorCached;
        Color     m_selectedBorderColorHoverCached;
        Color     m_backgroundColorCached;
        Color     m_backgroundColorHoverCached;
        Color     m_backgroundColorDisabledCached;
        Color     m_selectedBackgroundColorCached;
        Color     m_selectedBackgroundColorHoverCached;
        Color     m_textColorCached;
        Color     m_textColorHoverCached;
        Color     m_textColorDisabledCached;
        Color     m_selectedTextColorCached;
        Color     m_selectedTextColorHoverCached;
        float     m_distanceToSideCached = 0;
        float     m_roundedBorderRadiusCached = 0;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_TABS_BASE_HPP
