/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2026 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <TGUI/Widgets/BoxLayout.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    BoxLayout::BoxLayout(const char* typeName, bool initRenderer) :
        Group{typeName, initRenderer}
    {
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    BoxLayoutRenderer* BoxLayout::getSharedRenderer()
    {
        return downcast<BoxLayoutRenderer*>(Widget::getSharedRenderer());
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    const BoxLayoutRenderer* BoxLayout::getSharedRenderer() const
    {
        return downcast<const BoxLayoutRenderer*>(Widget::getSharedRenderer());
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    BoxLayoutRenderer* BoxLayout::getRenderer()
    {
        return downcast<BoxLayoutRenderer*>(Widget::getRenderer());
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BoxLayout::setSize(const Layout2d& size)
    {
        Group::setSize(size);

        updateWidgets();
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BoxLayout::add(const Widget::Ptr& widget, const String& widgetName)
    {
        insert(m_widgets.size(), widget, widgetName);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BoxLayout::insert(std::size_t index, const Widget::Ptr& widget, const String& widgetName)
    {
        // Move the widget to the right position
        if (index < m_widgets.size())
        {
            Group::add(widget, widgetName);

            m_widgets.pop_back();

            m_widgets.insert(m_widgets.begin() + static_cast<std::ptrdiff_t>(index), widget);
        }
        else // Just add the widget to the back
            Group::add(widget, widgetName);

        updateWidgets();
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    bool BoxLayout::remove(const Widget::Ptr& widget)
    {
        for (std::size_t i = 0; i < m_widgets.size(); ++i)
        {
            if (m_widgets[i] == widget)
                return remove(i);
        }

        return false;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    bool BoxLayout::remove(std::size_t index)
    {
        if (index >= m_widgets.size())
            return false;

        Group::remove(m_widgets[index]);

        updateWidgets();
        return true;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    Widget::Ptr BoxLayout::get(std::size_t index) const
    {
        if (index < m_widgets.size())
            return m_widgets[index];
        return nullptr;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    bool BoxLayout::setWidgetIndex(const Widget::Ptr& widget, std::size_t index)
    {
        const auto widgetIndexChanged = Group::setWidgetIndex(widget, index);
        if (widgetIndexChanged)
            updateWidgets();
        return widgetIndexChanged;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    void BoxLayout::rendererChanged(const String& property)
    {
        if (property == U"SpaceBetweenWidgets")
        {
            m_spaceBetweenWidgetsCached = getSharedRenderer()->getSpaceBetweenWidgets();
            updateWidgets();
        }
        else if (property == U"Padding")
        {
            Group::rendererChanged(property);

            // Update the space between widgets as the padding is used when no space was explicitly set
            m_spaceBetweenWidgetsCached = getSharedRenderer()->getSpaceBetweenWidgets();
            updateWidgets();
        }
        else
            Group::rendererChanged(property);
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
