/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2025 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_CHILD_WINDOW_HPP
#define TGUI_CHILD_WINDOW_HPP

#include <TGUI/CopiedSharedPtr.hpp>
#include <TGUI/Container.hpp>
#include <TGUI/Widgets/Button.hpp>
#include <TGUI/Renderers/ChildWindowRenderer.hpp>

#if !TGUI_EXPERIMENTAL_USE_STD_MODULE
    #include <limits>
#endif

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TGUI_MODULE_EXPORT namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Child window widget
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API ChildWindow : public Container
    {
    public:

        using Ptr = std::shared_ptr<ChildWindow>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const ChildWindow>; //!< Shared constant widget pointer

        static constexpr const char StaticWidgetType[] = "ChildWindow"; //!< Type name of the widget

        /// Title alignments, possible options for the setTitleAlignment function
        using TitleAlignment TGUI_DEPRECATED("Use tgui::HorizontalAlignment instead") = HorizontalAlignment;

        /// Title buttons (use bitwise OR to combine)
        enum TitleButton : unsigned int
        {
            None     = 0,      //!< No buttons
            Close    = 1 << 0, //!< Include a close button
            Maximize = 1 << 1, //!< Include a maximize button
            Minimize = 1 << 2  //!< Include a minimize button
        };

        /// Defines what the child window should do inside its close() function (which is called when the close button is pressed)
        /// @since TGUI 1.6
        enum class CloseBehavior
        {
            None,   //!< Nothing should happen after the onClose callback is called. The window remains visible (unless the onClose callback did something).
            Hide,   //!< childWindow->setVisible(false) is called after the onClose callback is called
            Remove  //!< parent->remove(childWindow) is called after the onClose callback is called
        };

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ChildWindow(const char* typeName = StaticWidgetType, bool initRenderer = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Copy constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ChildWindow(const ChildWindow&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Move constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ChildWindow(ChildWindow&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of copy assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ChildWindow& operator= (const ChildWindow&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of move assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ChildWindow& operator= (ChildWindow&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Creates a new child window widget
        ///
        /// @return The new child window
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static ChildWindow::Ptr create(const String& title = "", unsigned int titleButtons = TitleButton::Close);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Makes a copy of another child window
        ///
        /// @param childWindow  The other child window
        ///
        /// @return The new child window
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static ChildWindow::Ptr copy(const ChildWindow::ConstPtr& childWindow);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer that may be shared with other widgets using the same renderer
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD ChildWindowRenderer* getSharedRenderer() override;
        TGUI_NODISCARD const ChildWindowRenderer* getSharedRenderer() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the renderer, which gives access to functions that determine how the widget is displayed
        /// @return Temporary pointer to the renderer
        /// @warning After calling this function, the widget has its own copy of the renderer and it will no longer be shared.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD ChildWindowRenderer* getRenderer() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the position of the widget
        ///
        /// This function completely overwrites the previous position.
        /// See the move function to apply an offset based on the previous position instead.
        /// The default position of a transformable widget is (0, 0).
        ///
        /// @param position New position
        ///
        /// @see move, getPosition
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setPosition(const Layout2d& position) override;
        using Widget::setPosition;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the size of the child window
        ///
        /// @param size   Sets the new size of the child window
        ///
        /// This is the size of the entire child window, including the title bar and the borders.
        /// To set the size of the contents of the window (exluding the title bar and borders), use setClientSize instead.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setSize(const Layout2d& size) override;
        using Widget::setSize;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the space available for widgets inside the container
        /// @return Size without borders title bar
        ///
        /// For ChildWindow, the inner size is the same as the client size.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getInnerSize() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the client size of the child window
        ///
        /// @param size  New size of the child window contents
        ///
        /// This sets the size of the child window excluding the title bar and the borders.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        virtual void setClientSize(const Layout2d& size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the client size of the child window
        ///
        /// @return Size of the child window contents
        ///
        /// This sets the size of the child window excluding the title bar and the borders.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getClientSize() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the maximum size of the child window.
        ///
        /// @param size   Sets the new maximum size of the child window
        ///
        /// This function sets the maximum size of the entire window, including borders and titlebar.
        /// If the window is larger than the new maximum size, it will automatically be shrunk.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setMaximumSize(Vector2f size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the maximum size of the child window.
        ///
        /// @return Maximum size of the child window
        ///
        /// This size includes the title bar and the borders.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getMaximumSize() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the minimum size of the child window.
        ///
        /// @param size   Sets the new minimum size of the child window
        ///
        /// This function sets the minimum size of the entire window, including borders and titlebar.
        /// If the window is smaller than the new minimum size, it will automatically be enlarged.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setMinimumSize(Vector2f size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the minimum size of the child window.
        ///
        /// @return Minimum size of the child window
        ///
        /// This size includes the title bar and the borders.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getMinimumSize() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the title that is displayed in the title bar of the child window
        ///
        /// @param title  New title for the child window
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTitle(const String& title);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the title that is displayed in the title bar of the child window
        ///
        /// @return Title of the child window
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD const String& getTitle() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the character size of the title
        /// @param size  The new title text size.
        ///              If the size is set to 0 then the character size is determined by the height of the title bar.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTitleTextSize(unsigned int size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the character size of the title
        /// @return The current title text size
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD unsigned int getTitleTextSize() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the title alignment
        ///
        /// @param alignment  How should the title be aligned in the title bar?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTitleAlignment(HorizontalAlignment alignment);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the title alignment
        ///
        /// @return How the title is aligned in the title bar
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD HorizontalAlignment getTitleAlignment() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the title buttons
        ///
        /// @param buttons  Which buttons should be available in the title bar?
        ///
        /// By default ChildWindows only display a close button.
        /// The following example gives the ChildWindow both a minimize and close button.
        /// @code
        /// childWindow->setTitleButtons(ChildWindow::TitleButton::Minimize | ChildWindow::TitleButton::Close);
        /// @endcode
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTitleButtons(unsigned int buttons);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the title bar buttons
        ///
        /// @return Which buttons are available in the title bar
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD unsigned int getTitleButtons() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the behavior of closing the window
        /// @param behavior  Defines what the close() function does after calling the onClose callback
        ///
        /// The default close behavior is Remove.
        ///
        /// @since TGUI 1.6
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setCloseBehavior(CloseBehavior behavior);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the behavior of closing the window
        /// @return What the close() function does after calling the onClose callback
        ///
        /// The default close behavior is Remove.
        ///
        /// @since TGUI 1.6
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD CloseBehavior getCloseBehavior() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Try to close the window
        ///
        /// This will trigger the onClosing signal. If a callback function for this signal sets the abort parameter to true then
        /// the window will remain open. Otherwise the onClose signal is triggered and the chosen closing behavior is executed.
        ///
        /// If you want to "close" the window without those callbacks being triggered then you should just remove the child
        /// window from the parent to which it was added (i.e. parent->remove(childWindow)).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void close();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Closes the window
        ///
        /// This function is equivalent to removing the window from its parent. If you want to be receive a callback and have
        /// the ability to abort the operation then you should use the close() function instead.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_DEPRECATED("Use parent->remove(childWindow) instead") void destroy();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether the child window can be resized by dragging its borders or not
        ///
        /// @param resizable  Can the user change the size of the window by dragging one of the borders?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setResizable(bool resizable = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Checks whether the child window can be resized by dragging its borders or not
        ///
        /// @return Can the user change the size of the window by dragging one of the borders?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isResizable() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes whether the child window can be moved by dragging its title bar or not
        /// @param positionLocked  Is the child window frozen at its current position?
        ///
        /// Locking the position only affects user interaction, the setPosition function will still move the window.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setPositionLocked(bool positionLocked = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Checks whether the child window can be moved by dragging its title bar or not
        /// @return Is the child window frozen at its current position?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isPositionLocked() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the child window to be kept inside its parent
        ///
        /// @param enabled  When it's set to true, the child window will always be kept automatically inside its parent.
        ///                 It's set to false by default.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setKeepInParent(bool enabled = true);

#ifndef TGUI_REMOVE_DEPRECATED_CODE
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Tells whether the child window is kept inside its parent
        ///
        /// @return When it's set to true, the child window will always be kept automatically inside its parent.
        ///         It's set to false by default.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_DEPRECATED("Use getKeepInParent instead") TGUI_NODISCARD bool isKeptInParent() const;
#endif

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the child window is kept inside its parent
        ///
        /// @return When it's set to true, the child window will always be kept automatically inside its parent.
        ///         It's set to false by default.
        ///
        /// @since TGUI 1.3
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool getKeepInParent() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the distance between the position of the container and a widget that would be drawn inside
        ///        this container on relative position (0,0).
        ///
        /// @return Offset of the widgets in the container
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getChildWidgetsOffset() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// This function is called when the widget is added to a container.
        /// You should not call this function yourself.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setParent(Container* parent) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether the mouse position (which is relative to the parent widget) lies on top of the widget
        ///
        /// @return Is the mouse on top of the widget?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isMouseOnWidget(Vector2f pos) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool leftMousePressed(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void leftMouseReleased(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rightMousePressed(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rightMouseReleased(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseMoved(Vector2f pos) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void keyPressed(const Event::KeyEvent& event) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Called by the parent of the widget to check if keyPressed would process the event
        ///
        /// @param event  Key event that took place
        ///
        /// @return True if the event would be handled by the widget, false if the key event doesn't affect the widget
        ///
        /// @since TGUI 1.1
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        bool canHandleKeyPress(const Event::KeyEvent& event) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseNoLongerOnWidget() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void leftMouseButtonNoLongerDown() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Draw the widget to a render target
        ///
        /// @param target Render target to draw to
        /// @param states Current render states
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void draw(BackendRenderTarget& target, RenderStates states) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Updates the title bar texture, text and buttons after the title bar height has changed.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateTitleBarHeight();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Updates the mouse cursor for resizable child windows
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateResizeMouseCursor(Vector2f mousePos);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Retrieves a signal based on its name
        ///
        /// @param signalName  Name of the signal
        ///
        /// @return Signal that corresponds to the name
        ///
        /// @throw Exception when the name does not match any signal
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Signal& getSignal(String signalName) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Function called when one of the properties of the renderer is changed
        ///
        /// @param property  Name of the property that was changed
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void rendererChanged(const String& property) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Saves the widget as a tree node in order to save it to a file
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD std::unique_ptr<DataIO::Node> save(SavingRenderersMap& renderers) const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Loads the widget from a tree of nodes
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void load(const std::unique_ptr<DataIO::Node>& node, const LoadingRenderersMap& renderers) override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief This function is called when the mouse enters the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseEnteredWidget() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief This function is called when the mouse leaves the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void mouseLeftWidget() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Makes a copy of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Widget::Ptr clone() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    private:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // Helper function to connect the title button callbacks from the various constuctors and assignment operators
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void connectTitleButtonCallbacks();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    public:

        Signal            onMousePress = {"MousePressed"};  //!< The mouse went down on the widget
        SignalChildWindow onClose      = {"Closed"};        //!< The window was closed. Optional parameter: pointer to the window
        SignalChildWindow onMinimize   = {"Minimized"};     //!< The window was minimized. Optional parameter: pointer to the window
        SignalChildWindow onMaximize   = {"Maximized"};     //!< The window was maximized. Optional parameter: pointer to the window
        SignalChildWindow onEscapeKeyPress = {"EscapeKeyPressed"}; //!< The escape key was pressed while the child window was focused. Optional parameter: pointer to the window

        /// The window is about to be closed, unless the "abort" parameter is set to true.
        /// @code
        /// window->onClosing([](bool* abort) { *abort = true; });
        /// @endcode
        SignalTyped<bool*> onClosing = {"Closing"};

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        enum ResizeDirection
        {
            ResizeNone   = 0,
            ResizeLeft   = 1,
            ResizeTop    = 2,
            ResizeRight  = 4,
            ResizeBottom = 8
        };

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        Text           m_titleText;
        Vector2f       m_draggingPosition;
        Vector2f       m_maximumSize    = {std::numeric_limits<float>::infinity(), std::numeric_limits<float>::infinity()};
        Vector2f       m_minimumSize    = {0, 0};
        Layout*        m_decorationLayoutX = nullptr;
        Layout*        m_decorationLayoutY = nullptr;
        HorizontalAlignment m_titleAlignment = HorizontalAlignment::Center;
        unsigned int   m_titleButtons   = TitleButton::Close;
        unsigned int   m_titleTextSize  = 0;
        Cursor::Type   m_currentChildWindowMouseCursor = Cursor::Type::Arrow;
        CloseBehavior  m_closeBehavior = CloseBehavior::Remove;

        CopiedSharedPtr<Button> m_closeButton;
        CopiedSharedPtr<Button> m_minimizeButton;
        CopiedSharedPtr<Button> m_maximizeButton;

        bool m_mouseDownOnTitleBar = false;
        bool m_keepInParent = false;

        bool m_positionLocked = false;
        bool m_resizable = false;
        int m_resizeDirection = ResizeNone;

        Sprite m_spriteTitleBar;
        Sprite m_spriteBackground;

        // Cached renderer properties
        Borders m_bordersCached;
        Color   m_borderColorCached;
        Color   m_borderColorFocusedCached;
        Color   m_titleColorCached;
        Color   m_titleBarColorCached;
        Color   m_backgroundColorCached;
        float   m_titleBarHeightCached = 20;
        float   m_borderBelowTitleBarCached = 0;
        float   m_distanceToSideCached = 0;
        float   m_paddingBetweenButtonsCached = 0;
        float   m_minimumResizableBorderWidthCached = 10;
        bool    m_showTextOnTitleButtonsCached = false;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_CHILD_WINDOW_HPP
