#![cfg(all(feature = "python", feature = "pypi"))]
#![allow(clippy::disallowed_types)]

use anyhow::Result;
use assert_fs::prelude::*;
use insta::assert_snapshot;

use common::{uv_snapshot, TestContext};

mod common;

/// Lock a requirement from PyPI.
#[test]
fn lock_wheel_registry() -> Result<()> {
    let context = TestContext::new("3.12");
    let requirements_in = context.temp_dir.child("requirements.in");
    requirements_in.write_str("anyio==3.7.0")?;

    uv_snapshot!(context
        .compile()
        .arg("requirements.in")
        .arg("--generate-hashes")
        .arg("--unstable-uv-lock-file"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----
    # This file was autogenerated by uv via the following command:
    #    uv pip compile --cache-dir [CACHE_DIR] --exclude-newer 2024-03-25T00:00:00Z requirements.in --generate-hashes --unstable-uv-lock-file
    anyio==3.7.0 \
        --hash=sha256:275d9973793619a5374e1c89a4f4ad3f4b0a5510a2b5b939444bee8f4c4d37ce \
        --hash=sha256:eddca883c4175f14df8aedce21054bfca3adb70ffe76a9f607aef9d7fa2ea7f0
        # via -r requirements.in
    exceptiongroup==1.2.0 \
        --hash=sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14 \
        --hash=sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68
        # via anyio
    idna==3.6 \
        --hash=sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca \
        --hash=sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f
        # via anyio
    sniffio==1.3.1 \
        --hash=sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2 \
        --hash=sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc
        # via anyio
    typing-extensions==4.10.0 \
        --hash=sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475 \
        --hash=sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb
        # via anyio

    ----- stderr -----
    Resolved 5 packages in [TIME]
    "###);

    let lock = fs_err::read_to_string(context.temp_dir.join("uv.lock"))?;

    assert_snapshot!(
        lock, @r###"
    version = 1

    [[distribution]]
    name = "anyio"
    version = "3.7.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/c6/b3/fefbf7e78ab3b805dec67d698dc18dd505af7a18a8dd08868c9b4fa736b5/anyio-3.7.0.tar.gz"
    hash = "sha256:275d9973793619a5374e1c89a4f4ad3f4b0a5510a2b5b939444bee8f4c4d37ce"
    size = 142737

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/68/fe/7ce1926952c8a403b35029e194555558514b365ad77d75125f521a2bec62/anyio-3.7.0-py3-none-any.whl"
    hash = "sha256:eddca883c4175f14df8aedce21054bfca3adb70ffe76a9f607aef9d7fa2ea7f0"
    size = 80873

    [[distribution.dependencies]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [[distribution]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/8e/1c/beef724eaf5b01bb44b6338c8c3494eff7cab376fab4904cfbbc3585dc79/exceptiongroup-1.2.0.tar.gz"
    hash = "sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68"
    size = 26264

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/b8/9a/5028fd52db10e600f1c4674441b968cf2ea4959085bfb5b99fb1250e5f68/exceptiongroup-1.2.0-py3-none-any.whl"
    hash = "sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14"
    size = 16210

    [[distribution]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/bf/3f/ea4b9117521a1e9c50344b909be7886dd00a519552724809bb1f486986c2/idna-3.6.tar.gz"
    hash = "sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca"
    size = 175426

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/c2/e7/a82b05cf63a603df6e68d59ae6a68bf5064484a0718ea5033660af4b54a9/idna-3.6-py3-none-any.whl"
    hash = "sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f"
    size = 61567

    [[distribution]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/a2/87/a6771e1546d97e7e041b6ae58d80074f81b7d5121207425c964ddf5cfdbd/sniffio-1.3.1.tar.gz"
    hash = "sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc"
    size = 20372

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/e9/44/75a9c9421471a6c4805dbf2356f7c181a29c1879239abab1ea2cc8f38b40/sniffio-1.3.1-py3-none-any.whl"
    hash = "sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2"
    size = 10235

    [[distribution]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/16/3a/0d26ce356c7465a19c9ea8814b960f8a36c3b0d07c323176620b7b483e44/typing_extensions-4.10.0.tar.gz"
    hash = "sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb"
    size = 77558

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/f9/de/dc04a3ea60b22624b51c703a84bbe0184abcd1d0b9bc8074b5d6b7ab90bb/typing_extensions-4.10.0-py3-none-any.whl"
    hash = "sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475"
    size = 33926
    "###
    );

    // Install from the lockfile.
    uv_snapshot!(context.install().arg("--unstable-uv-lock-file").arg("anyio"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----

    ----- stderr -----
    Downloaded 5 packages in [TIME]
    Installed 5 packages in [TIME]
     + anyio==3.7.0
     + exceptiongroup==1.2.0
     + idna==3.6
     + sniffio==1.3.1
     + typing-extensions==4.10.0
    "###);

    Ok(())
}

/// Lock a requirement from PyPI.
#[test]
fn lock_sdist_registry() -> Result<()> {
    let context = TestContext::new("3.12");
    let requirements_in = context.temp_dir.child("requirements.in");
    requirements_in.write_str("netwell")?;

    uv_snapshot!(context
        .compile()
        .arg("requirements.in")
        .arg("--no-deps")
        .arg("--generate-hashes")
        .arg("--unstable-uv-lock-file"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----
    # This file was autogenerated by uv via the following command:
    #    uv pip compile --cache-dir [CACHE_DIR] --exclude-newer 2024-03-25T00:00:00Z requirements.in --no-deps --generate-hashes --unstable-uv-lock-file
    netwell==0.4.0 \
        --hash=sha256:cd6940614081790099b8fc1fdcf65cb930d54c647ed8bc9ba90dc781cf33926e
        # via -r requirements.in

    ----- stderr -----
    Resolved 1 package in [TIME]
    "###);

    let lock = fs_err::read_to_string(context.temp_dir.join("uv.lock"))?;

    assert_snapshot!(
        lock, @r###"
    version = 1

    [[distribution]]
    name = "netwell"
    version = "0.4.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/9d/01/4aacc00ffd2ed59595769e233baf1e72b3a4034ddabf65ff4ef34e1d487e/netwell-0.4.0.tar.gz"
    hash = "sha256:cd6940614081790099b8fc1fdcf65cb930d54c647ed8bc9ba90dc781cf33926e"
    size = 6671
    "###
    );

    // Install from the lockfile.
    uv_snapshot!(context.install().arg("--unstable-uv-lock-file").arg("netwell"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----

    ----- stderr -----
    Downloaded 1 package in [TIME]
    Installed 1 package in [TIME]
     + netwell==0.4.0
    "###);

    Ok(())
}

/// Lock a Git requirement.
#[test]
fn lock_sdist_git() -> Result<()> {
    let context = TestContext::new("3.12");
    let requirements_in = context.temp_dir.child("requirements.in");
    requirements_in.write_str("anyio @ git+https://github.com/agronholm/anyio@3.7.0")?;

    uv_snapshot!(context
        .compile()
        .arg("requirements.in")
        .arg("--generate-hashes")
        .arg("--unstable-uv-lock-file"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----
    # This file was autogenerated by uv via the following command:
    #    uv pip compile --cache-dir [CACHE_DIR] --exclude-newer 2024-03-25T00:00:00Z requirements.in --generate-hashes --unstable-uv-lock-file
    anyio @ git+https://github.com/agronholm/anyio@f7a880ffac4766efb39e6fb60fc28d944f5d2f65
        # via -r requirements.in
    exceptiongroup==1.2.0 \
        --hash=sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14 \
        --hash=sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68
        # via anyio
    idna==3.6 \
        --hash=sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca \
        --hash=sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f
        # via anyio
    sniffio==1.3.1 \
        --hash=sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2 \
        --hash=sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc
        # via anyio
    typing-extensions==4.10.0 \
        --hash=sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475 \
        --hash=sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb
        # via anyio

    ----- stderr -----
    Resolved 5 packages in [TIME]
    "###);

    let lock = fs_err::read_to_string(context.temp_dir.join("uv.lock"))?;

    assert_snapshot!(
        lock, @r###"
    version = 1

    [[distribution]]
    name = "anyio"
    version = "3.7.0"
    source = "git+https://github.com/agronholm/anyio?rev=3.7.0#f7a880ffac4766efb39e6fb60fc28d944f5d2f65"

    [distribution.sdist]
    url = "https://github.com/agronholm/anyio?rev=3.7.0#f7a880ffac4766efb39e6fb60fc28d944f5d2f65"

    [[distribution.dependencies]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [[distribution]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/8e/1c/beef724eaf5b01bb44b6338c8c3494eff7cab376fab4904cfbbc3585dc79/exceptiongroup-1.2.0.tar.gz"
    hash = "sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68"
    size = 26264

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/b8/9a/5028fd52db10e600f1c4674441b968cf2ea4959085bfb5b99fb1250e5f68/exceptiongroup-1.2.0-py3-none-any.whl"
    hash = "sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14"
    size = 16210

    [[distribution]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/bf/3f/ea4b9117521a1e9c50344b909be7886dd00a519552724809bb1f486986c2/idna-3.6.tar.gz"
    hash = "sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca"
    size = 175426

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/c2/e7/a82b05cf63a603df6e68d59ae6a68bf5064484a0718ea5033660af4b54a9/idna-3.6-py3-none-any.whl"
    hash = "sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f"
    size = 61567

    [[distribution]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/a2/87/a6771e1546d97e7e041b6ae58d80074f81b7d5121207425c964ddf5cfdbd/sniffio-1.3.1.tar.gz"
    hash = "sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc"
    size = 20372

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/e9/44/75a9c9421471a6c4805dbf2356f7c181a29c1879239abab1ea2cc8f38b40/sniffio-1.3.1-py3-none-any.whl"
    hash = "sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2"
    size = 10235

    [[distribution]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/16/3a/0d26ce356c7465a19c9ea8814b960f8a36c3b0d07c323176620b7b483e44/typing_extensions-4.10.0.tar.gz"
    hash = "sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb"
    size = 77558

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/f9/de/dc04a3ea60b22624b51c703a84bbe0184abcd1d0b9bc8074b5d6b7ab90bb/typing_extensions-4.10.0-py3-none-any.whl"
    hash = "sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475"
    size = 33926
    "###
    );

    // Install from the lockfile.
    uv_snapshot!(context.install().arg("--unstable-uv-lock-file").arg("anyio"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----

    ----- stderr -----
    Downloaded 4 packages in [TIME]
    Installed 5 packages in [TIME]
     + anyio==3.7.0 (from git+https://github.com/agronholm/anyio@f7a880ffac4766efb39e6fb60fc28d944f5d2f65)
     + exceptiongroup==1.2.0
     + idna==3.6
     + sniffio==1.3.1
     + typing-extensions==4.10.0
    "###);

    Ok(())
}

/// Lock a requirement from a direct URL to a wheel.
#[test]
fn lock_wheel_url() -> Result<()> {
    let context = TestContext::new("3.12");
    let requirements_in = context.temp_dir.child("requirements.in");
    requirements_in.write_str("anyio @ https://files.pythonhosted.org/packages/14/fd/2f20c40b45e4fb4324834aea24bd4afdf1143390242c0b33774da0e2e34f/anyio-4.3.0-py3-none-any.whl")?;

    uv_snapshot!(context
        .compile()
        .arg("requirements.in")
        .arg("--generate-hashes")
        .arg("--unstable-uv-lock-file"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----
    # This file was autogenerated by uv via the following command:
    #    uv pip compile --cache-dir [CACHE_DIR] --exclude-newer 2024-03-25T00:00:00Z requirements.in --generate-hashes --unstable-uv-lock-file
    anyio @ https://files.pythonhosted.org/packages/14/fd/2f20c40b45e4fb4324834aea24bd4afdf1143390242c0b33774da0e2e34f/anyio-4.3.0-py3-none-any.whl \
        --hash=sha256:048e05d0f6caeed70d731f3db756d35dcc1f35747c8c403364a8332c630441b8
        # via -r requirements.in
    exceptiongroup==1.2.0 \
        --hash=sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14 \
        --hash=sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68
        # via anyio
    idna==3.6 \
        --hash=sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca \
        --hash=sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f
        # via anyio
    sniffio==1.3.1 \
        --hash=sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2 \
        --hash=sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc
        # via anyio
    typing-extensions==4.10.0 \
        --hash=sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475 \
        --hash=sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb
        # via anyio

    ----- stderr -----
    Resolved 5 packages in [TIME]
    "###);

    let lock = fs_err::read_to_string(context.temp_dir.join("uv.lock"))?;

    assert_snapshot!(
        lock, @r###"
    version = 1

    [[distribution]]
    name = "anyio"
    version = "4.3.0"
    source = "direct+https://files.pythonhosted.org/packages/14/fd/2f20c40b45e4fb4324834aea24bd4afdf1143390242c0b33774da0e2e34f/anyio-4.3.0-py3-none-any.whl"

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/14/fd/2f20c40b45e4fb4324834aea24bd4afdf1143390242c0b33774da0e2e34f/anyio-4.3.0-py3-none-any.whl"
    hash = "sha256:048e05d0f6caeed70d731f3db756d35dcc1f35747c8c403364a8332c630441b8"

    [[distribution.dependencies]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [[distribution]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/8e/1c/beef724eaf5b01bb44b6338c8c3494eff7cab376fab4904cfbbc3585dc79/exceptiongroup-1.2.0.tar.gz"
    hash = "sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68"
    size = 26264

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/b8/9a/5028fd52db10e600f1c4674441b968cf2ea4959085bfb5b99fb1250e5f68/exceptiongroup-1.2.0-py3-none-any.whl"
    hash = "sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14"
    size = 16210

    [[distribution]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/bf/3f/ea4b9117521a1e9c50344b909be7886dd00a519552724809bb1f486986c2/idna-3.6.tar.gz"
    hash = "sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca"
    size = 175426

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/c2/e7/a82b05cf63a603df6e68d59ae6a68bf5064484a0718ea5033660af4b54a9/idna-3.6-py3-none-any.whl"
    hash = "sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f"
    size = 61567

    [[distribution]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/a2/87/a6771e1546d97e7e041b6ae58d80074f81b7d5121207425c964ddf5cfdbd/sniffio-1.3.1.tar.gz"
    hash = "sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc"
    size = 20372

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/e9/44/75a9c9421471a6c4805dbf2356f7c181a29c1879239abab1ea2cc8f38b40/sniffio-1.3.1-py3-none-any.whl"
    hash = "sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2"
    size = 10235

    [[distribution]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/16/3a/0d26ce356c7465a19c9ea8814b960f8a36c3b0d07c323176620b7b483e44/typing_extensions-4.10.0.tar.gz"
    hash = "sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb"
    size = 77558

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/f9/de/dc04a3ea60b22624b51c703a84bbe0184abcd1d0b9bc8074b5d6b7ab90bb/typing_extensions-4.10.0-py3-none-any.whl"
    hash = "sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475"
    size = 33926
    "###
    );

    // Install from the lockfile.
    uv_snapshot!(context.install().arg("--unstable-uv-lock-file").arg("anyio"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----

    ----- stderr -----
    Downloaded 4 packages in [TIME]
    Installed 5 packages in [TIME]
     + anyio==4.3.0 (from https://files.pythonhosted.org/packages/14/fd/2f20c40b45e4fb4324834aea24bd4afdf1143390242c0b33774da0e2e34f/anyio-4.3.0-py3-none-any.whl)
     + exceptiongroup==1.2.0
     + idna==3.6
     + sniffio==1.3.1
     + typing-extensions==4.10.0
    "###);

    Ok(())
}

/// Lock a requirement from a direct URL to an sdist.
#[test]
fn lock_sdist_url() -> Result<()> {
    let context = TestContext::new("3.12");
    let requirements_in = context.temp_dir.child("requirements.in");
    requirements_in.write_str("anyio @ https://files.pythonhosted.org/packages/db/4d/3970183622f0330d3c23d9b8a5f52e365e50381fd484d08e3285104333d3/anyio-4.3.0.tar.gz")?;

    uv_snapshot!(context
        .compile()
        .arg("requirements.in")
        .arg("--generate-hashes")
        .arg("--unstable-uv-lock-file"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----
    # This file was autogenerated by uv via the following command:
    #    uv pip compile --cache-dir [CACHE_DIR] --exclude-newer 2024-03-25T00:00:00Z requirements.in --generate-hashes --unstable-uv-lock-file
    anyio @ https://files.pythonhosted.org/packages/db/4d/3970183622f0330d3c23d9b8a5f52e365e50381fd484d08e3285104333d3/anyio-4.3.0.tar.gz \
        --hash=sha256:f75253795a87df48568485fd18cdd2a3fa5c4f7c5be8e5e36637733fce06fed6
        # via -r requirements.in
    exceptiongroup==1.2.0 \
        --hash=sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14 \
        --hash=sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68
        # via anyio
    idna==3.6 \
        --hash=sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca \
        --hash=sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f
        # via anyio
    sniffio==1.3.1 \
        --hash=sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2 \
        --hash=sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc
        # via anyio
    typing-extensions==4.10.0 \
        --hash=sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475 \
        --hash=sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb
        # via anyio

    ----- stderr -----
    Resolved 5 packages in [TIME]
    "###);

    let lock = fs_err::read_to_string(context.temp_dir.join("uv.lock"))?;

    assert_snapshot!(
        lock, @r###"
    version = 1

    [[distribution]]
    name = "anyio"
    version = "4.3.0"
    source = "direct+https://files.pythonhosted.org/packages/db/4d/3970183622f0330d3c23d9b8a5f52e365e50381fd484d08e3285104333d3/anyio-4.3.0.tar.gz"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/db/4d/3970183622f0330d3c23d9b8a5f52e365e50381fd484d08e3285104333d3/anyio-4.3.0.tar.gz"
    hash = "sha256:f75253795a87df48568485fd18cdd2a3fa5c4f7c5be8e5e36637733fce06fed6"

    [[distribution.dependencies]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [[distribution.dependencies]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [[distribution]]
    name = "exceptiongroup"
    version = "1.2.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/8e/1c/beef724eaf5b01bb44b6338c8c3494eff7cab376fab4904cfbbc3585dc79/exceptiongroup-1.2.0.tar.gz"
    hash = "sha256:91f5c769735f051a4290d52edd0858999b57e5876e9f85937691bd4c9fa3ed68"
    size = 26264

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/b8/9a/5028fd52db10e600f1c4674441b968cf2ea4959085bfb5b99fb1250e5f68/exceptiongroup-1.2.0-py3-none-any.whl"
    hash = "sha256:4bfd3996ac73b41e9b9628b04e079f193850720ea5945fc96a08633c66912f14"
    size = 16210

    [[distribution]]
    name = "idna"
    version = "3.6"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/bf/3f/ea4b9117521a1e9c50344b909be7886dd00a519552724809bb1f486986c2/idna-3.6.tar.gz"
    hash = "sha256:9ecdbbd083b06798ae1e86adcbfe8ab1479cf864e4ee30fe4e46a003d12491ca"
    size = 175426

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/c2/e7/a82b05cf63a603df6e68d59ae6a68bf5064484a0718ea5033660af4b54a9/idna-3.6-py3-none-any.whl"
    hash = "sha256:c05567e9c24a6b9faaa835c4821bad0590fbb9d5779e7caa6e1cc4978e7eb24f"
    size = 61567

    [[distribution]]
    name = "sniffio"
    version = "1.3.1"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/a2/87/a6771e1546d97e7e041b6ae58d80074f81b7d5121207425c964ddf5cfdbd/sniffio-1.3.1.tar.gz"
    hash = "sha256:f4324edc670a0f49750a81b895f35c3adb843cca46f0530f79fc1babb23789dc"
    size = 20372

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/e9/44/75a9c9421471a6c4805dbf2356f7c181a29c1879239abab1ea2cc8f38b40/sniffio-1.3.1-py3-none-any.whl"
    hash = "sha256:2f6da418d1f1e0fddd844478f41680e794e6051915791a034ff65e5f100525a2"
    size = 10235

    [[distribution]]
    name = "typing-extensions"
    version = "4.10.0"
    source = "registry+https://pypi.org/simple"

    [distribution.sdist]
    url = "https://files.pythonhosted.org/packages/16/3a/0d26ce356c7465a19c9ea8814b960f8a36c3b0d07c323176620b7b483e44/typing_extensions-4.10.0.tar.gz"
    hash = "sha256:b0abd7c89e8fb96f98db18d86106ff1d90ab692004eb746cf6eda2682f91b3cb"
    size = 77558

    [[distribution.wheel]]
    url = "https://files.pythonhosted.org/packages/f9/de/dc04a3ea60b22624b51c703a84bbe0184abcd1d0b9bc8074b5d6b7ab90bb/typing_extensions-4.10.0-py3-none-any.whl"
    hash = "sha256:69b1a937c3a517342112fb4c6df7e72fc39a38e7891a5730ed4985b5214b5475"
    size = 33926
    "###
    );

    // Install from the lockfile.
    uv_snapshot!(context.install().arg("--unstable-uv-lock-file").arg("anyio"), @r###"
    success: true
    exit_code: 0
    ----- stdout -----

    ----- stderr -----
    Downloaded 4 packages in [TIME]
    Installed 5 packages in [TIME]
     + anyio==4.3.0 (from https://files.pythonhosted.org/packages/db/4d/3970183622f0330d3c23d9b8a5f52e365e50381fd484d08e3285104333d3/anyio-4.3.0.tar.gz)
     + exceptiongroup==1.2.0
     + idna==3.6
     + sniffio==1.3.1
     + typing-extensions==4.10.0
    "###);

    Ok(())
}
