%feature("docstring") OT::AggregatedProcess
"Aggregation of several processes in one process.

Available constructor:
    AggregatedProcess(*collProc*)

Parameters
----------
collProc : sequence of :class:`~openturns.Process`
    Collection of processes which all have the same input dimension.

Notes
-----
If we note :math:`X_i: \\Omega \\times\\cD_i \\mapsto \\Rset^{d_i}` for :math:`0 \\leq i \\leq N` the collection of processes, where :math:`\\cD_i \\in \\Rset^n` for all :math:`i`.
Then the resulting aggregated process :math:`Y: \\Omega \\times\\cD_0 \\mapsto \\Rset^d` where :math:`d=\\sum_{i=0}^N d_i`. The mesh of the first process :math:`X_0` has been assigned to the process :math:`Y`.

Examples
--------
Create an aggregated process:

>>> import openturns as ot
>>> myMesher = ot.IntervalMesher([5, 10])
>>> lowerbound = [0.0, 0.0]
>>> upperBound = [2.0, 4.0]
>>> myInterval = ot.Interval(lowerbound, upperBound)
>>> myMesh = myMesher.build(myInterval)
>>> myProcess1 = ot.WhiteNoise(ot.Normal(), myMesh)
>>> myProcess2 = ot.WhiteNoise(ot.Triangular(), myMesh)
>>> myAggregatedProcess = ot.AggregatedProcess([myProcess1, myProcess2])

Draw one realization:

>>> myGraph = myAggregatedProcess.getRealization().drawMarginal(0)"

// ---------------------------------------------------------------------

%feature("docstring") OT::AggregatedProcess::getRealization
"Get one realization of the aggregated process.

Returns
-------
realization : :class:`~openturns.Field`
    Each process of the collection is realized on the common mesh defined on :math:`\\cD_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::AggregatedProcess::getContinuousRealization
"Get a continuous realization.

Returns
-------
realization : :class:`~openturns.Function`
    Each process of the collection is continuously realized on the common domain :math:`\\cD_0`."

// ---------------------------------------------------------------------

%feature("docstring") OT::AggregatedProcess::getMarginal
"Accessor the marginal processes.

Available usages:
    getMarginal(*index*)

    getMarginal(*indices*)

Parameters
----------
index : int
    Index of the selected marginal process.
indices : :class:`~openturns.Indices`
    List of indices of the selected marginal processes.

Notes
-----
The selected marginal processes are extracted if the list of indices does not mingle the processes of the initial collection: take care to extract all the marginal processes process by process.
For example, if :math:`X_0=(X_0^0, X_0^1)`, :math:`X_1=(X_1^0, X_1^1)` and :math:`X_2=(X_2^0, X_2^1, X_2^2)` then you can extract `Indices([1,0,2,4,6])` but not `Indices([1,2,0,4,6])`."

// ---------------------------------------------------------------------

%feature("docstring") OT::AggregatedProcess::setProcessCollection
"Set the collection of processes.

Parameters
----------
collProc : sequence of :class:`~openturns.Process`
    Collection of processes which all have the same input dimension.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::AggregatedProcess::getProcessCollection
"Get the collection of processes.

Returns
-------
collProc : :class:`~openturns.ProcessCollection`
    Collection of processes which all have the same input dimension.
"
