/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2025 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_GROW_HORIZONTAL_LAYOUT_HPP
#define TGUI_GROW_HORIZONTAL_LAYOUT_HPP

#include <TGUI/Widgets/BoxLayoutRatios.hpp>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TGUI_MODULE_EXPORT namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    /// @brief Container that automatically positions children beside each other.
    ///
    /// The width of child widgets isn't altered, while the height is changed to match that of the horizontal layout.
    /// With each widget that gets added, the width of the horizontal layout increases to fit the new widget.
    ///
    /// If you want the layout to have a fixed size and want children resized to fill the area then check HorizontalLayout instead.
    ///
    /// @since TGUI 1.7
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    class TGUI_API GrowHorizontalLayout : public BoxLayout
    {
    public:
        using Ptr = std::shared_ptr<GrowHorizontalLayout>; //!< Shared widget pointer
        using ConstPtr = std::shared_ptr<const GrowHorizontalLayout>; //!< Shared constant widget pointer

        static constexpr const char StaticWidgetType[] = "GrowHorizontalLayout"; //!< Type name of the widget

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// @brief Constructor
        /// @param typeName     Type of the widget
        /// @param initRenderer Should the renderer be initialized? Should be true unless a derived class initializes it.
        /// @see create
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GrowHorizontalLayout(const char* typeName = StaticWidgetType, bool initRenderer = true);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Copy constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GrowHorizontalLayout(const GrowHorizontalLayout&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Move constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GrowHorizontalLayout(GrowHorizontalLayout&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of copy assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GrowHorizontalLayout& operator=(const GrowHorizontalLayout&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Move assignment
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        GrowHorizontalLayout& operator=(GrowHorizontalLayout&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ~GrowHorizontalLayout() = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Creates a new horizontal layout widget
        ///
        /// @param height  Height of the horizontal layout
        ///
        /// @return The new horizontal layout
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static GrowHorizontalLayout::Ptr create(const Layout& height = {RelativeValue(1)});

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Makes a copy of another layout
        ///
        /// @param layout  The other layout
        ///
        /// @return The new layout
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD static GrowHorizontalLayout::Ptr copy(const GrowHorizontalLayout::ConstPtr& layout);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the space available for widgets inside the container
        /// @return Size of the container
        ///
        /// @warning The width returned by this function is always 0. This is to prevent child widgets from depending on the
        ///          width of the horizontal layout (which isn't allowed as the width of the layout depends on its children).
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getInnerSize() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Removes all widgets that were added to the container
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void removeAllWidgets() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // @brief Repositions and resize the widgets
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateWidgets() override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        // @brief Makes a copy of the widget
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Widget::Ptr clone() const override;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    protected:

        std::vector<Layout> m_widgetLayouts;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_GROW_HORIZONTAL_LAYOUT_HPP
