/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.netbeans.modules.options.editor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.ListCellRenderer;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import org.netbeans.api.editor.EditorRegistry;
import org.netbeans.api.options.OptionsDisplayer;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.spi.options.OptionsPanelController;

/**
 *
 * @author Dusan Balek
 */
@OptionsPanelController.KeywordsRegistration({
    @OptionsPanelController.Keywords(keywords = {"#KW_Hints"}, location = OptionsDisplayer.EDITOR, tabTitle="#CTL_Hints_DisplayName"),
    @OptionsPanelController.Keywords(keywords = {"#KW_Mark"}, location = OptionsDisplayer.EDITOR, tabTitle="#CTL_MarkOccurences_DisplayName"),
    @OptionsPanelController.Keywords(keywords = {"#KW_InlineHints"}, location = OptionsDisplayer.EDITOR, tabTitle="#CTL_InlineHints_DisplayName")
})
public final class FolderBasedOptionPanel extends JPanel implements ActionListener {
    
    private final FolderBasedController controller;
    private String lastSelectedMimeType = null;
    
    /** Creates new form FolderBasedOptionPanel */
    FolderBasedOptionPanel(FolderBasedController controller, Document filterDocument, boolean allowFiltering) {
        this.controller = controller;

        initComponents();

        filter.setDocument(filterDocument);

        if (!allowFiltering) {
            filter.setVisible(false);
            filterLabel.setVisible(false);
        }
        
        ListCellRenderer renderer = new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                if (value instanceof String)
                    value = EditorSettings.getDefault().getLanguageName((String)value);
                return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
            }
        };
        languageCombo.setRenderer(renderer);
        languageCombo.addActionListener(this);

        update();
    }

    void update () {
        DefaultComboBoxModel<String> model = new DefaultComboBoxModel<>();
        for (String mimeType : controller.getMimeTypes()) {
            model.addElement(mimeType);
        }
        languageCombo.setModel(model);
        
        lastSelectedMimeType = controller.getSavedSelectedLanguage();        
        if (lastSelectedMimeType != null && model.getSize() > 0) {
            languageCombo.setSelectedItem(lastSelectedMimeType);
            return;
        }

        JTextComponent pane = EditorRegistry.lastFocusedComponent();
        String preSelectMimeType = pane != null ? (String)pane.getDocument().getProperty("mimeType") : ""; // NOI18N
        languageCombo.setSelectedItem(preSelectMimeType);
        if (!preSelectMimeType.equals (languageCombo.getSelectedItem()) && model.getSize() > 0) {
            languageCombo.setSelectedIndex(0);
        }
    }
    
    String getSelectedLanguage() {
        return (String)languageCombo.getSelectedItem();
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        languageLabel = new javax.swing.JLabel();
        languageCombo = new javax.swing.JComboBox();
        optionsPanel = new javax.swing.JPanel();
        filter = new javax.swing.JTextField();
        filterLabel = new javax.swing.JLabel();

        setBorder(javax.swing.BorderFactory.createEmptyBorder(10, 10, 10, 10));

        languageLabel.setLabelFor(languageCombo);
        org.openide.awt.Mnemonics.setLocalizedText(languageLabel, org.openide.util.NbBundle.getMessage(FolderBasedOptionPanel.class, "LBL_Language")); // NOI18N

        languageCombo.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));

        optionsPanel.setOpaque(false);
        optionsPanel.setLayout(new java.awt.BorderLayout());

        filter.setColumns(10);
        filter.setText(org.openide.util.NbBundle.getMessage(FolderBasedOptionPanel.class, "FolderBasedOptionPanel.filter.text")); // NOI18N

        filterLabel.setLabelFor(filter);
        org.openide.awt.Mnemonics.setLocalizedText(filterLabel, org.openide.util.NbBundle.getMessage(FolderBasedOptionPanel.class, "FolderBasedOptionPanel.filterLabel.text")); // NOI18N

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addComponent(languageLabel)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(languageCombo, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addComponent(filterLabel)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(filter, javax.swing.GroupLayout.PREFERRED_SIZE, 114, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
            .addComponent(optionsPanel, javax.swing.GroupLayout.Alignment.TRAILING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(languageLabel)
                    .addComponent(languageCombo, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(filter, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(filterLabel))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(optionsPanel, javax.swing.GroupLayout.DEFAULT_SIZE, 288, Short.MAX_VALUE))
        );
    }// </editor-fold>//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTextField filter;
    private javax.swing.JLabel filterLabel;
    private javax.swing.JComboBox languageCombo;
    private javax.swing.JLabel languageLabel;
    private javax.swing.JPanel optionsPanel;
    // End of variables declaration//GEN-END:variables
 
    // Change in the combos
    public void actionPerformed(ActionEvent e) {
        optionsPanel.setVisible(false);
        optionsPanel.removeAll();
        String mimeType = (String)languageCombo.getSelectedItem();
        if (mimeType != null) {
            OptionsPanelController opc = controller.getController(mimeType);
            if (opc != null) {
                JComponent component = opc.getComponent(controller.getLookup());
                optionsPanel.add(component, BorderLayout.CENTER);
                optionsPanel.setVisible(true);
            }
        }

        searchEnableDisable();
        if (isShowing()) { //remember the last selected option only when panel is visible
            lastSelectedMimeType = mimeType;
        }
    }

    void setCurrentMimeType(String key) {
        languageCombo.setSelectedItem(key);
    }

    @Override
    public void removeNotify() {
        super.removeNotify();
        filter.setText("");
        lastSelectedMimeType = null;
    }

    void searchEnableDisable() {
        String mimeType = (String)languageCombo.getSelectedItem();
        
        filter.setEnabled(mimeType != null ? controller.supportsFilter(mimeType) : false);
    }
}
