// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include <cstdlib>
#include <iostream>
#include <string>
#include <vector>

#include "base/no_destructor.h"
#include "base/notreached.h"
#include "base/time/time_override.h"
#include "components/cast_certificate/cast_cert_reader.h"
#include "components/cast_certificate/cast_cert_test_helpers.h"
#include "components/cast_channel/cast_auth_util.h"
#include "components/cast_channel/fuzz_proto/fuzzer_inputs.pb.h"
#include "net/cert/x509_certificate.h"
#include "net/cert/x509_util.h"
#include "net/test/test_certificate_data.h"
#include "testing/libfuzzer/proto/lpm_interface.h"

namespace cast_channel {
namespace fuzz {
namespace {

const char kCertData[] = {
// Generated by //net/data/ssl/certificates:generate_fuzzer_cert_includes
#include "net/data/ssl/certificates/wildcard.inc"
};

base::NoDestructor<std::vector<std::string>> certs;

static bool InitializeOnce() {
  *certs = cast_certificate::ReadCertificateChainFromFile(
      cast_certificate::testing::GetCastCertificatesSubDirectory().AppendASCII(
          "chromecast_gen1.pem"));
  CHECK(certs->size() >= 1)
      << "We should always have at least one certificate.";
  return true;
}

void UpdateTime(TimeBoundCase c, const base::Time* time, int direction) {
  auto& mtime = const_cast<base::Time&>(*time);
  switch (c) {
    case TimeBoundCase::VALID:
      // Create bound that include the current date.
      mtime = base::Time::Now() + base::TimeDelta::FromDays(direction);
      break;
    case TimeBoundCase::INVALID:
      // Create a bound that excludes the current date.
      mtime = base::Time::Now() + base::TimeDelta::FromDays(-direction);
      break;
    case TimeBoundCase::OOB:
      // Create a bound so far in the past/future it's not valid.
      mtime = base::Time::Now() + base::TimeDelta::FromDays(direction * 10000);
      break;
    case TimeBoundCase::MISSING:
      // Remove any existing bound.
      mtime = base::Time();
      break;
    default:
      NOTREACHED();
  }
}

DEFINE_PROTO_FUZZER(CastAuthUtilInputs& input_union) {
  static bool init = InitializeOnce();
  CHECK(init);

  switch (input_union.input_case()) {
    case CastAuthUtilInputs::kAuthenticateChallengeReplyInput: {
      auto& input = *input_union.mutable_authenticate_challenge_reply_input();

      // If we have a DeviceAuthMessage, use it to override the cast_message()
      // payload with a more interesting value.
      if (input.has_auth_message()) {
        // Optimization: if the payload_binary() field is going to be
        // overwritten, insist that it has to be empty initially.  This cuts
        // down on how much time is spent generating identical arguments for
        // AuthenticateChallengeReply() from different values of |input|.
        if (input.cast_message().has_payload_binary())
          return;

        if (!input.auth_message().has_response()) {
          // Optimization.
          if (input.nonce_ok() || input.response_certs_ok() ||
              input.tbs_crls_size() || input.crl_certs_ok() ||
              input.crl_signatures_ok()) {
            return;
          }
        } else {
          auto& response = *input.mutable_auth_message()->mutable_response();

          // Maybe force the nonce to be the correct value.
          if (input.nonce_ok()) {
            // Optimization.
            if (response.has_sender_nonce())
              return;

            response.set_sender_nonce(input.nonce());
          }

          // Maybe force the response certs to be valid.
          if (input.response_certs_ok()) {
            // Optimization.
            if (!response.client_auth_certificate().empty() ||
                response.intermediate_certificate_size() > 0)
              return;

            response.set_client_auth_certificate(certs->front());
            response.clear_intermediate_certificate();
            for (std::size_t i = 1; i < certs->size(); i++) {
              response.add_intermediate_certificate(certs->at(i));
            }
          }

          // Maybe replace the crl() field in the response with valid data.
          if (input.tbs_crls_size() == 0) {
            // Optimization.
            if (input.crl_certs_ok() || input.crl_signatures_ok())
              return;
          } else {
            // Optimization.
            if (response.has_crl())
              return;

            cast::certificate::CrlBundle crl_bundle;
            for (const auto& tbs_crl : input.tbs_crls()) {
              cast::certificate::Crl& crl = *crl_bundle.add_crls();
              if (input.crl_certs_ok())
                crl.set_signer_cert(certs->at(0));
              if (input.crl_signatures_ok())
                crl.set_signature("");
              tbs_crl.SerializeToString(crl.mutable_tbs_crl());
            }
            crl_bundle.SerializeToString(response.mutable_crl());
          }
        }

        input.mutable_cast_message()->set_payload_type(CastMessage::BINARY);
        input.auth_message().SerializeToString(
            input.mutable_cast_message()->mutable_payload_binary());
      }

      // Build a well-formed cert with start and expiry times relative to the
      // current time.  The actual cert doesn't matter for testing purposes
      // because validation failures are ignored.
      scoped_refptr<net::X509Certificate> peer_cert =
          net::X509Certificate::CreateFromBytes(kCertData,
                                                base::size(kCertData));
      UpdateTime(input.start_case(), &peer_cert->valid_start(), -1);
      UpdateTime(input.expiry_case(), &peer_cert->valid_expiry(), +1);

      AuthContext context = AuthContext::CreateForTest(input.nonce());

      AuthenticateChallengeReply(input.cast_message(), *peer_cert, context);
      break;
    }
    default:
      NOTREACHED();
  }
}

}  // namespace
}  // namespace fuzz
}  // namespace cast_channel
