/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2011 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** No Commercial Usage
**
** This file contains pre-release code and may not be distributed.
** You may use this file in accordance with the terms and conditions
** contained in the Technology Preview License Agreement accompanying
** this package.
**
** GNU Lesser General Public License Usage
**
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**************************************************************************/

#include "stackframe.h"
#include "stackhandler.h"

#include <QtCore/QFileInfo>
#include <QtCore/QDebug>
#include <QtCore/QDir>
#include <QtCore/QTextStream>

#include <QtGui/QToolTip>
#include <QtGui/QFontMetrics>

namespace Debugger {
namespace Internal {

////////////////////////////////////////////////////////////////////////
//
// StackFrame
//
////////////////////////////////////////////////////////////////////////

StackFrame::StackFrame()
  : level(0), line(0), address(0)
{}

void StackFrame::clear()
{
    line = level = 0;
    function.clear();
    file.clear();
    from.clear();
    to.clear();
    address = 0;
}

bool StackFrame::isUsable() const
{
    return !file.isEmpty() && QFileInfo(file).isReadable();
}

QString StackFrame::toString() const
{
    QString res;
    QTextStream str(&res);
    str << StackHandler::tr("Address:") << ' ';
    str.setIntegerBase(16);
    str << address;
    str.setIntegerBase(10);
    str << ' '
        << StackHandler::tr("Function:") << ' ' << function << ' '
        << StackHandler::tr("File:") << ' ' << file << ' '
        << StackHandler::tr("Line:") << ' ' << line << ' '
        << StackHandler::tr("From:") << ' ' << from << ' '
        << StackHandler::tr("To:") << ' ' << to;
    return res;
}

QString StackFrame::toToolTip() const
{
    const QString filePath = QDir::toNativeSeparators(file);
    QString res;
    QTextStream str(&res);
    str << "<html><body><table>"
        << "<tr><td>" << StackHandler::tr("Address:") << "</td><td>0x";
    str.setIntegerBase(16);
    str <<  address;
    str.setIntegerBase(10);
    str << "</td></tr>"
        << "<tr><td>" << StackHandler::tr("Function:") << "</td><td>" << function << "</td></tr>"
        << "<tr><td>" << StackHandler::tr("File:") << "</td><td width="
        << QFontMetrics(QToolTip::font()).width(filePath) << ">" << filePath << "</td></tr>"
        << "<tr><td>" << StackHandler::tr("Line:") << "</td><td>" << line << "</td></tr>"
        << "<tr><td>" << StackHandler::tr("From:") << "</td><td>" << from << "</td></tr>"
        << "<tr><td>" << StackHandler::tr("To:") << "</td><td>" << to << "</td></tr>"
        << "</table></body></html>";
    return res;
}

QDebug operator<<(QDebug d, const  StackFrame &f)
{
    QString res;
    QTextStream str(&res);
    str << "level=" << f.level << " address=" << f.address;
    if (!f.function.isEmpty())
        str << ' ' << f.function;
    if (!f.file.isEmpty())
        str << ' ' << f.file << ':' << f.line;
    if (!f.from.isEmpty())
        str << " from=" << f.from;
    if (!f.to.isEmpty())
        str << " to=" << f.to;
    d.nospace() << res;
    return d;
}

} // namespace Internal
} // namespace Debugger
