/* ======================================================================== *
 * fastttyrec - utility for stripping long pauses from ttyrec files         *
 * Copyright (C) 2006 Thomas Schumm <phong@phong.org>                       *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify it  *
 * under the terms of the GNU General Public License as published by the    *
 * Free Software Foundation; either version 2 of the License, or (at your   *
 * option) any later version.                                               *
 *                                                                          *
 * This program is distributed in the hope that it will be useful, but      *
 * WITHOUT ANY WARRANTY; without even the implied warranty of               *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        *
 * General Public License for more details.                                 *
 *                                                                          *
 * You should have received a copy of the GNU General Public License along  *
 * with this program; if not, write to the Free Software Foundation, Inc.,  *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA                    *
 * ======================================================================== */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

struct timeRec {
    unsigned int sec;
    unsigned int usec;
    unsigned int len;
};

void parseArgs(int argc, char *argv[], struct timeRec *rec) { /* {{{ */
    /*
     * fasttyrec takes exactly one argument - a maximum duration for
     * pauses in the output ttyrec file
     */
    int x, mid;
    char temp[256];

    if (argc != 2) {
        fprintf(stderr, "Usage: fastttyrec maxtime < infile > outfile\n");
        exit(1);
    }

    if (strlen(argv[1]) >= sizeof(temp)) {
        fprintf(stderr, "Error parsing command line.\n");
        exit(1);
    }

    strncpy(temp, argv[1], sizeof(temp));
    temp[255] = '\0';

    x = 0;
    while (temp[x] != '\0' && temp[x] != '.') {
        if (temp[x] < '0' || temp[x] > '9') {
            fprintf(stderr, "Error parsing command line.\n");
            exit(1);
        }
        x++;
    }

    if (temp[x] == '\0') {
        rec->sec = atoi(temp);
        rec->usec = 0;
        return;
    }

    mid = x;
    temp[mid] = '\0';

    x++;
    while (temp[x] != '\0') {
        if (temp[x] < '0' || temp[x] > '9') {
            fprintf(stderr, "Error parsing command line.\n");
            exit(1);
        }
        x++;
    }

    temp[mid+6+1] = '\0';
    rec->sec = atoi(temp);
    rec->usec = atoi(temp + mid + 1);
    for (x=0; x<6-strlen(temp+mid+1); x++)
        rec->usec *= 10;

    return;
} /* }}} */

void parseRec(unsigned char rec[], struct timeRec *dest) { /* {{{ */
    /*
     * ttyrec files consist of a series of records.  Each record
     * consists of a 12 byte header (describing the timing information
     * and the length of the data component of the record,)
     * Values are stored little-endian.  Or is it big-endian?  I forget
     * which.  This code handles it though.
     *
     * The 12 byte header is 3 unsigned four byte integers.  The first two are
     * a timestamp, the first being whole seconds, the second being the
     * fractional part in microseconds.  The third integer is the length of
     * the data portion.  The difference between the timestamps of two
     * consecutive records is effectively the pause between them.
     */
    dest->sec  =  (rec[3] << 24) |  (rec[2] << 16) | (rec[1] << 8) | rec[0];
    dest->usec =  (rec[7] << 24) |  (rec[6] << 16) | (rec[5] << 8) | rec[4];
    dest->len  = (rec[11] << 24) | (rec[10] << 16) | (rec[9] << 8) | rec[8];

    return;
} /* }}} */

void packRec(unsigned char rec[], struct timeRec *src) { /* {{{ */
    rec[0]  = (src->sec       ) & 0xff;
    rec[1]  = (src->sec  >>  8) & 0xff;
    rec[2]  = (src->sec  >> 16) & 0xff;
    rec[3]  = (src->sec  >> 24) & 0xff;
    rec[4]  = (src->usec      ) & 0xff;
    rec[5]  = (src->usec >>  8) & 0xff;
    rec[6]  = (src->usec >> 16) & 0xff;
    rec[7]  = (src->usec >> 24) & 0xff;
    rec[8]  = (src->len       ) & 0xff;
    rec[9]  = (src->len  >>  8) & 0xff;
    rec[10] = (src->len  >> 16) & 0xff;
    rec[11] = (src->len  >> 24) & 0xff;
} /* }}} */

int main(int argc, char *argv[]) {
    unsigned char rec[12];
    unsigned char buf[4096];
    struct timeRec max, prev, cur, diff, fudge;
    int didFirst = 0;
    int r;

    parseArgs(argc, argv, &max);

    prev.sec = prev.usec = 0;
    fudge.sec = fudge.usec = 0;

    errno = 0;
    while (fread(rec, 1, 12, stdin) == 12) {

        parseRec(rec, &cur);

        if (didFirst) {
            diff.sec = cur.sec - prev.sec;
            /* determine if the difference between consecutive time
             * markers is greater than the maximum permitted */
            if (prev.usec > cur.usec) {
                diff.sec--;
                diff.usec = cur.usec + 1000000 - prev.usec;
            } else {
                diff.usec = cur.usec - prev.usec;
            }
            if (diff.sec > max.sec || (diff.sec == max.sec && diff.usec > max.usec)) {
                /* The gap between the last two records was greater than the
                 * max, so we adjust our fudge factor */
                fudge.sec += diff.sec - max.sec;
                if (diff.usec < max.usec) {
                    fudge.sec -= 1;
                    fudge.usec += 1000000;
                }
                fudge.usec += diff.usec - max.usec;
                fudge.sec += fudge.usec/1000000;
                fudge.usec %= 1000000;
            }
        }

        /* fprintf(stderr, "%d.%06d\n", fudge.sec, fudge.usec); */
        prev.sec = cur.sec;
        prev.usec = cur.usec;

        /* all records after an adjusted gap must be shifted backwards */
        cur.sec -= fudge.sec;
        if (cur.usec < fudge.usec) {
            cur.sec--;
            cur.usec += 1000000;
        }
        cur.usec -= fudge.usec;

        /* write out the modified record */
        packRec(rec, &cur);
        r = fwrite(rec, 1, 12, stdout);
        if (r != 12) {
            perror("Error reading writing to stdout");
            exit(1);
        }

        /* copy over the data portion of the record unmodified */
        while (cur.len > sizeof(buf)) {
            r = fread(buf, 1, sizeof(buf), stdin);
            if (r != sizeof(buf)) {
                perror("Error reading from stdin");
                exit(1);
            }
            r = fwrite(buf, 1, sizeof(buf), stdout);
            cur.len -= sizeof(buf);
        }
        r = fread(buf, 1, cur.len, stdin);
        if (r != cur.len) {
            perror("Error reading from stdin");
            exit(1);
        }
        r = fwrite(buf, 1, cur.len, stdout);

        didFirst = 1;
    }

    if (errno != 0) {
        perror("Error reading from stdin");
        exit(1);
    }

    return 0;
}
