# This file is part of dftd4.
# SPDX-Identifier: LGPL-3.0-or-later
#
# dftd4 is free software: you can redistribute it and/or modify it under
# the terms of the Lesser GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# dftd4 is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# Lesser GNU General Public License for more details.
#
# You should have received a copy of the Lesser GNU General Public License
# along with dftd4.  If not, see <https://www.gnu.org/licenses/>.

from typing import Union

import numpy as np
import pytest
from pytest import approx

try:
    import dftd4.pyscf as disp
    import pyscf
    from pyscf import gto, scf, pbc
except ModuleNotFoundError:
    pyscf = None


@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
@pytest.mark.parametrize("ecp", [None, "ccecp"])
def test_energy_r2scan_d4(ecp: Union[None, str]) -> None:
    mol = gto.M(
        atom="""
             C   -0.755422531  -0.796459123  -1.023590391
             C    0.634274834  -0.880017014  -1.075233285
             C    1.406955202   0.199695367  -0.653144334
             C    0.798863737   1.361204515  -0.180597909
             C   -0.593166787   1.434312023  -0.133597923
             C   -1.376239198   0.359205222  -0.553258516
             I   -1.514344238   3.173268101   0.573601106
             H    1.110906949  -1.778801728  -1.440619836
             H    1.399172302   2.197767355   0.147412751
             H    2.486417780   0.142466525  -0.689380574
             H   -2.454252250   0.422581120  -0.512807958
             H   -1.362353593  -1.630564523  -1.348743149
             S   -3.112683203   6.289227834   1.226984439
             H   -4.328789697   5.797771251   0.973373089
             C   -2.689135032   6.703163830  -0.489062886
             H   -1.684433029   7.115457372  -0.460265708
             H   -2.683867206   5.816530502  -1.115183775
             H   -3.365330613   7.451201412  -0.890098894
             """,
        ecp=ecp,
    )

    d4 = disp.DFTD4Dispersion(mol, xc="r2SCAN")
    assert d4.kernel()[0] == approx(-0.005001101058518388, abs=1.0e-7)

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_energy_tpss_d4s() -> None:
    mol = gto.M(
        atom="""
             Na  -1.855282634   3.586705153  -2.417637293 
             H    4.401780235   0.023388444  -4.954577493 
             O   -2.987060334   4.762520654   1.270433015 
             H    0.799808860   1.411034556  -5.046553216 
             F   -4.206474694   1.842757675   4.550380848 
             H   -3.543561218  -3.188356651   1.462400217 
             H    2.700321601   1.068184525  -1.732346503 
             O    3.731140888  -2.070015433   2.231609376 
             N   -1.753068192   0.359514171   1.053234061 
             H    5.417557885  -1.578818300   1.753940027 
             H   -2.234628682  -2.138565050   4.109222857 
             Cl   1.015658662  -3.219521545  -3.360509630 
             B    2.421192557   0.266264350  -3.918624743 
             B   -3.025260988   2.536678890   2.316649847 
             N   -2.004389486  -2.292351369   2.197828073 
             Al   1.122265541  -1.369420070   0.484550554 
             """
    )

    d4 = disp.DFTD4Dispersion(mol, xc="TPSS", model="d4s")
    assert d4.kernel()[0] == approx(-0.016049411775539424, abs=1.0e-7)

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_gradient_b97m_d4() -> None:
    mol = gto.M(
        atom="""
             H    0.002144194   0.361043475   0.029799709
             C    0.015020592   0.274789738   1.107648016
             C    1.227632658   0.296655040   1.794629427
             C    1.243958826   0.183702791   3.183703934
             C    0.047958213   0.048915002   3.886484583
             C   -1.165135654   0.026954348   3.200213281
             C   -1.181832083   0.139828643   1.810376587
             H    2.155807907   0.399177037   1.249441585
             H    2.184979344   0.198598553   3.716170761
             H    0.060934662  -0.040672756   4.964014252
             H   -2.093220602  -0.078628959   3.745125056
             H   -2.122845437   0.123257119   1.277645797
             Br  -0.268325907  -3.194209024   1.994458950
             C    0.049999933  -5.089197474   1.929391171
             F    0.078949601  -5.512441335   0.671851563
             F    1.211983937  -5.383996300   2.498664481
             F   -0.909987405  -5.743747328   2.570721738
             """
    )
    ref = np.array(
        [
            [+2.98598566e-06, +5.58662750e-05, -2.26040542e-04],
            [+1.75816159e-05, +3.76346114e-04, -5.59737686e-04],
            [+5.30037419e-04, +3.39528029e-04, -2.58687563e-04],
            [+5.28657666e-04, +2.71291979e-04, +3.25797992e-04],
            [+2.86698935e-05, +2.42198161e-04, +6.35784343e-04],
            [-4.99037369e-04, +2.74343468e-04, +3.55926359e-04],
            [-5.08919994e-04, +3.36480299e-04, -2.53165937e-04],
            [+1.90422032e-04, +3.91738111e-05, -1.05867702e-04],
            [+1.88137796e-04, +1.62741316e-05, +1.07526135e-04],
            [+6.73432246e-06, +2.93353505e-06, +2.23028509e-04],
            [-1.97182105e-04, +1.82664904e-05, +1.26683921e-04],
            [-2.07136902e-04, +4.89015922e-05, -1.12337721e-04],
            [-2.01762905e-04, -1.23133705e-03, -2.26387771e-04],
            [+7.59912392e-06, -1.01235495e-04, -7.86554908e-06],
            [+3.02563455e-05, -2.19783887e-04, -2.31677736e-04],
            [+2.31670341e-04, -2.08064229e-04, +9.53780006e-05],
            [-1.48713265e-04, -2.61183219e-04, +1.11642946e-04],
        ]
    )

    d4 = disp.DFTD4Dispersion(mol, xc="b97m")
    assert d4.kernel()[1] == approx(ref, abs=1.0e-7)

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_gradient_blyp_d4s() -> None:
    mol = gto.M(
        atom="""
             H   -1.795376258  -3.778664229  -1.078835583
             S   -2.682788333   0.388926662   1.662148652
             B    0.114846497   1.488579332   3.656603965
             O   -1.079988795  -0.162591216  -4.557030658
             Mg   0.603028329   4.088161496  -0.025893731
             H   -1.225340893  -1.799813824  -3.707731733
             H   -1.334609820  -4.248190824   2.727919027
             H   -0.162780825   2.412679941   5.690306951
             Si   2.878024440  -0.331205250   1.883113735
             H    0.684893279   0.327902040  -4.205476937
             B   -1.209197735  -2.872537625   0.940642042
             Li  -3.255726045   2.212410929  -2.867155493
             F   -1.831474682   5.205272937  -2.269762706
             H    4.908858657  -1.925765619   2.990699194
             H    1.268062422  -2.604093417   0.551628052
             S    4.119569763   1.598928667  -1.391174777
             """,
        spin=1
    )
    ref = np.array(
        [  
            [-1.04361222e-04, -1.65054791e-04, -1.36662175e-04],
            [-1.41500522e-03, +1.89282651e-04, +2.16639105e-04],
            [-1.18067839e-04, +4.50543787e-04, +1.50087553e-03],
            [+3.37690080e-04, -4.10348598e-04, -3.02311767e-04],
            [+4.39892308e-04, +1.54862493e-03, +1.33655085e-04],
            [+1.31259180e-06, -7.51721105e-05, -1.39848135e-04],
            [-4.61111364e-05, -1.65382677e-04, +1.81820530e-04],
            [-1.94292825e-05, +7.21791149e-05, +1.79879351e-04],
            [+1.14226323e-03, -6.08455689e-04, +6.24007890e-04],
            [+6.95738570e-05, -1.86718359e-05, -1.25837081e-04],
            [-1.66091884e-04, -1.03519307e-03, -1.71797180e-04],
            [-1.29925668e-03, +6.18658801e-05, -6.30138324e-04],
            [-1.58991399e-04, +5.73306273e-04, -2.35799582e-04],
            [+2.90056077e-04, -2.14985916e-04, +1.62430848e-04],
            [+6.43808246e-05, -3.35585457e-04, -2.45131168e-04],
            [+9.82145702e-04, +1.33047503e-04, -1.01178292e-03],
        ]
    )

    d4 = disp.DFTD4Dispersion(mol, xc="BLYP", model="d4s")
    assert d4.kernel()[1] == approx(ref, abs=1.0e-7)


@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_energy_hf() -> None:
    mol = gto.M(
        atom="""
             N  -1.57871857  -0.04661102   0.00000000
             N   1.57871857   0.04661102   0.00000000
             H  -2.15862174   0.13639605   0.80956529
             H  -0.84947130   0.65819321   0.00000000
             H  -2.15862174   0.13639605  -0.80956529
             H   2.15862174  -0.13639605  -0.80956529
             H   0.84947130  -0.65819321   0.00000000
             H   2.15862174  -0.13639605   0.80956529
             """
    )
    mf = disp.energy(scf.RHF(mol))
    assert mf.kernel() == approx(-110.91742452859162, abs=1.0e-8)
    assert "dispersion" in mf.scf_summary

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_energy_hf_d4s() -> None:
    mol = gto.M(
        atom="""
             N  -1.57871857  -0.04661102   0.00000000
             N   1.57871857   0.04661102   0.00000000
             H  -2.15862174   0.13639605   0.80956529
             H  -0.84947130   0.65819321   0.00000000
             H  -2.15862174   0.13639605  -0.80956529
             H   2.15862174  -0.13639605  -0.80956529
             H   0.84947130  -0.65819321   0.00000000
             H   2.15862174  -0.13639605   0.80956529
             """
    )
    mf = disp.energy(scf.RHF(mol), model="d4s")
    assert mf.kernel() == approx(-110.91765211773482, abs=1.0e-8)
    assert "dispersion" in mf.scf_summary

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_gradient_hf() -> None:
    mol = gto.M(
        atom="""
             O  -1.65542061  -0.12330038   0.00000000
             O   1.24621244   0.10268870   0.00000000
             H  -0.70409026   0.03193167   0.00000000
             H  -2.03867273   0.75372294   0.00000000
             H   1.57598558  -0.38252146  -0.75856129
             H   1.57598558  -0.38252146   0.75856129
             """
    )
    ref = np.array(
        [
            [+1.72438133e-02, +5.08406920e-02, -1.89235207e-15],
            [+3.80018285e-02, -4.60223790e-02, +1.94297606e-16],
            [-3.05058266e-02, -1.26478132e-02, +1.04061735e-15],
            [+6.92338575e-03, -3.82898692e-02, +8.54978540e-16],
            [-1.58316004e-02, +2.30596847e-02, +2.18908543e-02],
            [-1.58316004e-02, +2.30596847e-02, -2.18908543e-02],
        ]
    )
    grad = disp.energy(scf.RHF(mol)).run().nuc_grad_method()
    assert grad.kernel() == approx(ref, abs=1.0e-7)

@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_gradient_hf_d4s() -> None:
    mol = gto.M(
        atom="""
             O  -1.65542061  -0.12330038   0.00000000
             O   1.24621244   0.10268870   0.00000000
             H  -0.70409026   0.03193167   0.00000000
             H  -2.03867273   0.75372294   0.00000000
             H   1.57598558  -0.38252146  -0.75856129
             H   1.57598558  -0.38252146   0.75856129
             """
    )
    ref = np.array(
        [
            [+1.72591013e-02, +5.08609363e-02, -3.88151022e-16],
            [+3.80150780e-02, -4.60423874e-02, +1.26616497e-15],
            [-3.05314461e-02, -1.26530890e-02, +2.80179917e-16],
            [+6.92409507e-03, -3.83029588e-02, +1.05415315e-16],
            [-1.58334141e-02, +2.30687495e-02, +2.19006284e-02],
            [-1.58334141e-02, +2.30687495e-02, -2.19006284e-02],
        ]
    )
    grad = disp.energy(scf.RHF(mol), model="d4s").run().nuc_grad_method()
    assert grad.kernel() == approx(ref, abs=1.0e-7)


@pytest.mark.skipif(pyscf is None, reason="requires pyscf")
def test_pbc():
    mol = gto.M(
        atom="""
             O  -1.6256  -0.0413   0.3705
             H  -0.7061  -0.0938   0.0934
             H  -2.0618  -0.7328  -0.1359
             """,
        basis="def2-tzvp",
    )

    pmol = pbc.gto.M(
        atom="""
             O  -1.6256  -0.0413   0.3705
             H  -0.7061  -0.0938   0.0934
             H  -2.0618  -0.7328  -0.1359
             """,
        basis="def2-tzvp",
        a=[[3, 0, 0], [0, 3, 0], [0, 0, 3]],
    )

    xc = 'pbe'

    e_mol_disp = disp.DFTD4Dispersion(mol, xc=xc).kernel()[0]
    e_pbc_disp = disp.DFTD4Dispersion(pmol, xc=xc).kernel()[0]

    assert e_mol_disp != e_pbc_disp
