/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { GEMINI_DIR, Storage, ClearcutLogger, Config, ExtensionInstallEvent, } from '@google/gemini-cli-core';
import * as fs from 'node:fs';
import * as path from 'node:path';
import * as os from 'node:os';
import { simpleGit } from 'simple-git';
import { SettingScope, loadSettings } from '../config/settings.js';
import { getErrorMessage } from '../utils/errors.js';
import { recursivelyHydrateStrings } from './extensions/variables.js';
import { isWorkspaceTrusted } from './trustedFolders.js';
import { resolveEnvVarsInObject } from '../utils/envVarResolver.js';
import { randomUUID } from 'node:crypto';
export const EXTENSIONS_DIRECTORY_NAME = path.join(GEMINI_DIR, 'extensions');
export const EXTENSIONS_CONFIG_FILENAME = 'gemini-extension.json';
export const INSTALL_METADATA_FILENAME = '.gemini-extension-install.json';
export class ExtensionStorage {
    extensionName;
    constructor(extensionName) {
        this.extensionName = extensionName;
    }
    getExtensionDir() {
        return path.join(ExtensionStorage.getUserExtensionsDir(), this.extensionName);
    }
    getConfigPath() {
        return path.join(this.getExtensionDir(), EXTENSIONS_CONFIG_FILENAME);
    }
    static getUserExtensionsDir() {
        const storage = new Storage(os.homedir());
        return storage.getExtensionsDir();
    }
    static async createTmpDir() {
        return await fs.promises.mkdtemp(path.join(os.tmpdir(), 'gemini-extension'));
    }
}
export function getWorkspaceExtensions(workspaceDir) {
    // If the workspace dir is the user extensions dir, there are no workspace extensions.
    if (path.resolve(workspaceDir) === path.resolve(os.homedir())) {
        return [];
    }
    return loadExtensionsFromDir(workspaceDir);
}
async function copyExtension(source, destination) {
    await fs.promises.cp(source, destination, { recursive: true });
}
export async function performWorkspaceExtensionMigration(extensions) {
    const failedInstallNames = [];
    for (const extension of extensions) {
        try {
            const installMetadata = {
                source: extension.path,
                type: 'local',
            };
            await installExtension(installMetadata);
        }
        catch (_) {
            failedInstallNames.push(extension.config.name);
        }
    }
    return failedInstallNames;
}
export function loadExtensions(workspaceDir = process.cwd()) {
    const settings = loadSettings(workspaceDir).merged;
    const disabledExtensions = settings.extensions?.disabled ?? [];
    const allExtensions = [...loadUserExtensions()];
    if ((isWorkspaceTrusted(settings) ?? true) &&
        // Default management setting to true
        !(settings.experimental?.extensionManagement ?? true)) {
        allExtensions.push(...getWorkspaceExtensions(workspaceDir));
    }
    const uniqueExtensions = new Map();
    for (const extension of allExtensions) {
        if (!uniqueExtensions.has(extension.config.name) &&
            !disabledExtensions.includes(extension.config.name)) {
            uniqueExtensions.set(extension.config.name, extension);
        }
    }
    return Array.from(uniqueExtensions.values());
}
export function loadUserExtensions() {
    const userExtensions = loadExtensionsFromDir(os.homedir());
    const uniqueExtensions = new Map();
    for (const extension of userExtensions) {
        if (!uniqueExtensions.has(extension.config.name)) {
            uniqueExtensions.set(extension.config.name, extension);
        }
    }
    return Array.from(uniqueExtensions.values());
}
export function loadExtensionsFromDir(dir) {
    const storage = new Storage(dir);
    const extensionsDir = storage.getExtensionsDir();
    if (!fs.existsSync(extensionsDir)) {
        return [];
    }
    const extensions = [];
    for (const subdir of fs.readdirSync(extensionsDir)) {
        const extensionDir = path.join(extensionsDir, subdir);
        const extension = loadExtension(extensionDir);
        if (extension != null) {
            extensions.push(extension);
        }
    }
    return extensions;
}
export function loadExtension(extensionDir) {
    if (!fs.statSync(extensionDir).isDirectory()) {
        console.error(`Warning: unexpected file ${extensionDir} in extensions directory.`);
        return null;
    }
    const installMetadata = loadInstallMetadata(extensionDir);
    let effectiveExtensionPath = extensionDir;
    if (installMetadata?.type === 'link') {
        effectiveExtensionPath = installMetadata.source;
    }
    const configFilePath = path.join(effectiveExtensionPath, EXTENSIONS_CONFIG_FILENAME);
    if (!fs.existsSync(configFilePath)) {
        console.error(`Warning: extension directory ${effectiveExtensionPath} does not contain a config file ${configFilePath}.`);
        return null;
    }
    try {
        const configContent = fs.readFileSync(configFilePath, 'utf-8');
        let config = recursivelyHydrateStrings(JSON.parse(configContent), {
            extensionPath: extensionDir,
            '/': path.sep,
            pathSeparator: path.sep,
        });
        if (!config.name || !config.version) {
            console.error(`Invalid extension config in ${configFilePath}: missing name or version.`);
            return null;
        }
        config = resolveEnvVarsInObject(config);
        const contextFiles = getContextFileNames(config)
            .map((contextFileName) => path.join(effectiveExtensionPath, contextFileName))
            .filter((contextFilePath) => fs.existsSync(contextFilePath));
        return {
            path: effectiveExtensionPath,
            config,
            contextFiles,
            installMetadata,
        };
    }
    catch (e) {
        console.error(`Warning: error parsing extension config in ${configFilePath}: ${getErrorMessage(e)}`);
        return null;
    }
}
function loadInstallMetadata(extensionDir) {
    const metadataFilePath = path.join(extensionDir, INSTALL_METADATA_FILENAME);
    try {
        const configContent = fs.readFileSync(metadataFilePath, 'utf-8');
        const metadata = JSON.parse(configContent);
        return metadata;
    }
    catch (_e) {
        return undefined;
    }
}
function getContextFileNames(config) {
    if (!config.contextFileName) {
        return ['GEMINI.md'];
    }
    else if (!Array.isArray(config.contextFileName)) {
        return [config.contextFileName];
    }
    return config.contextFileName;
}
/**
 * Returns an annotated list of extensions. If an extension is listed in enabledExtensionNames, it will be active.
 * If enabledExtensionNames is empty, an extension is active unless it is in list of disabled extensions in settings.
 * @param extensions The base list of extensions.
 * @param enabledExtensionNames The names of explicitly enabled extensions.
 * @param workspaceDir The current workspace directory.
 */
export function annotateActiveExtensions(extensions, enabledExtensionNames, workspaceDir) {
    const settings = loadSettings(workspaceDir).merged;
    const disabledExtensions = settings.extensions?.disabled ?? [];
    const annotatedExtensions = [];
    if (enabledExtensionNames.length === 0) {
        return extensions.map((extension) => ({
            name: extension.config.name,
            version: extension.config.version,
            isActive: !disabledExtensions.includes(extension.config.name),
            path: extension.path,
        }));
    }
    const lowerCaseEnabledExtensions = new Set(enabledExtensionNames.map((e) => e.trim().toLowerCase()));
    if (lowerCaseEnabledExtensions.size === 1 &&
        lowerCaseEnabledExtensions.has('none')) {
        return extensions.map((extension) => ({
            name: extension.config.name,
            version: extension.config.version,
            isActive: false,
            path: extension.path,
        }));
    }
    const notFoundNames = new Set(lowerCaseEnabledExtensions);
    for (const extension of extensions) {
        const lowerCaseName = extension.config.name.toLowerCase();
        const isActive = lowerCaseEnabledExtensions.has(lowerCaseName);
        if (isActive) {
            notFoundNames.delete(lowerCaseName);
        }
        annotatedExtensions.push({
            name: extension.config.name,
            version: extension.config.version,
            isActive,
            path: extension.path,
        });
    }
    for (const requestedName of notFoundNames) {
        console.error(`Extension not found: ${requestedName}`);
    }
    return annotatedExtensions;
}
/**
 * Clones a Git repository to a specified local path.
 * @param gitUrl The Git URL to clone.
 * @param destination The destination path to clone the repository to.
 */
async function cloneFromGit(gitUrl, destination) {
    try {
        // TODO(chrstnb): Download the archive instead to avoid unnecessary .git info.
        await simpleGit().clone(gitUrl, destination, ['--depth', '1']);
    }
    catch (error) {
        throw new Error(`Failed to clone Git repository from ${gitUrl}`, {
            cause: error,
        });
    }
}
export async function installExtension(installMetadata, cwd = process.cwd()) {
    const config = new Config({
        sessionId: randomUUID(),
        targetDir: process.cwd(),
        cwd: process.cwd(),
        model: '',
        debugMode: false,
    });
    const logger = ClearcutLogger.getInstance(config);
    let newExtensionConfig = null;
    let localSourcePath;
    try {
        const settings = loadSettings(cwd).merged;
        if (!isWorkspaceTrusted(settings)) {
            throw new Error(`Could not install extension from untrusted folder at ${installMetadata.source}`);
        }
        const extensionsDir = ExtensionStorage.getUserExtensionsDir();
        await fs.promises.mkdir(extensionsDir, { recursive: true });
        if (!path.isAbsolute(installMetadata.source) &&
            (installMetadata.type === 'local' || installMetadata.type === 'link')) {
            installMetadata.source = path.resolve(cwd, installMetadata.source);
        }
        let tempDir;
        if (installMetadata.type === 'git') {
            tempDir = await ExtensionStorage.createTmpDir();
            await cloneFromGit(installMetadata.source, tempDir);
            localSourcePath = tempDir;
        }
        else if (installMetadata.type === 'local' ||
            installMetadata.type === 'link') {
            localSourcePath = installMetadata.source;
        }
        else {
            throw new Error(`Unsupported install type: ${installMetadata.type}`);
        }
        try {
            newExtensionConfig = await loadExtensionConfig(localSourcePath);
            if (!newExtensionConfig) {
                throw new Error(`Invalid extension at ${installMetadata.source}. Please make sure it has a valid gemini-extension.json file.`);
            }
            const newExtensionName = newExtensionConfig.name;
            const extensionStorage = new ExtensionStorage(newExtensionName);
            const destinationPath = extensionStorage.getExtensionDir();
            const installedExtensions = loadUserExtensions();
            if (installedExtensions.some((installed) => installed.config.name === newExtensionName)) {
                throw new Error(`Extension "${newExtensionName}" is already installed. Please uninstall it first.`);
            }
            await fs.promises.mkdir(destinationPath, { recursive: true });
            if (installMetadata.type === 'local' || installMetadata.type === 'git') {
                await copyExtension(localSourcePath, destinationPath);
            }
            const metadataString = JSON.stringify(installMetadata, null, 2);
            const metadataPath = path.join(destinationPath, INSTALL_METADATA_FILENAME);
            await fs.promises.writeFile(metadataPath, metadataString);
        }
        finally {
            if (tempDir) {
                await fs.promises.rm(tempDir, { recursive: true, force: true });
            }
        }
        logger?.logExtensionInstallEvent(new ExtensionInstallEvent(newExtensionConfig.name, newExtensionConfig.version, installMetadata.source, 'success'));
        return newExtensionConfig.name;
    }
    catch (error) {
        // Attempt to load config from the source path even if installation fails
        // to get the name and version for logging.
        if (!newExtensionConfig && localSourcePath) {
            newExtensionConfig = await loadExtensionConfig(localSourcePath);
        }
        logger?.logExtensionInstallEvent(new ExtensionInstallEvent(newExtensionConfig?.name ?? '', newExtensionConfig?.version ?? '', installMetadata.source, 'error'));
        throw error;
    }
}
export async function loadExtensionConfig(extensionDir) {
    const configFilePath = path.join(extensionDir, EXTENSIONS_CONFIG_FILENAME);
    if (!fs.existsSync(configFilePath)) {
        return null;
    }
    try {
        const configContent = fs.readFileSync(configFilePath, 'utf-8');
        const config = recursivelyHydrateStrings(JSON.parse(configContent), {
            extensionPath: extensionDir,
            '/': path.sep,
            pathSeparator: path.sep,
        });
        if (!config.name || !config.version) {
            return null;
        }
        return config;
    }
    catch (_) {
        return null;
    }
}
export async function uninstallExtension(extensionName, cwd = process.cwd()) {
    const installedExtensions = loadUserExtensions();
    if (!installedExtensions.some((installed) => installed.config.name === extensionName)) {
        throw new Error(`Extension "${extensionName}" not found.`);
    }
    removeFromDisabledExtensions(extensionName, [SettingScope.User, SettingScope.Workspace], cwd);
    const storage = new ExtensionStorage(extensionName);
    return await fs.promises.rm(storage.getExtensionDir(), {
        recursive: true,
        force: true,
    });
}
export function toOutputString(extension) {
    let output = `${extension.config.name} (${extension.config.version})`;
    output += `\n Path: ${extension.path}`;
    if (extension.installMetadata) {
        output += `\n Source: ${extension.installMetadata.source} (Type: ${extension.installMetadata.type})`;
    }
    if (extension.contextFiles.length > 0) {
        output += `\n Context files:`;
        extension.contextFiles.forEach((contextFile) => {
            output += `\n  ${contextFile}`;
        });
    }
    if (extension.config.mcpServers) {
        output += `\n MCP servers:`;
        Object.keys(extension.config.mcpServers).forEach((key) => {
            output += `\n  ${key}`;
        });
    }
    if (extension.config.excludeTools) {
        output += `\n Excluded tools:`;
        extension.config.excludeTools.forEach((tool) => {
            output += `\n  ${tool}`;
        });
    }
    return output;
}
export async function updateExtensionByName(extensionName, cwd = process.cwd()) {
    const installedExtensions = loadUserExtensions();
    const extension = installedExtensions.find((installed) => installed.config.name === extensionName);
    if (!extension) {
        throw new Error(`Extension "${extensionName}" not found. Run gemini extensions list to see available extensions.`);
    }
    return await updateExtension(extension, cwd);
}
export async function updateExtension(extension, cwd = process.cwd()) {
    if (!extension.installMetadata) {
        throw new Error(`Extension ${extension.config.name} cannot be updated.`);
    }
    if (extension.installMetadata.type === 'link') {
        throw new Error(`Extension is linked so does not need to be updated`);
    }
    const originalVersion = extension.config.version;
    const tempDir = await ExtensionStorage.createTmpDir();
    try {
        await copyExtension(extension.path, tempDir);
        await uninstallExtension(extension.config.name, cwd);
        await installExtension(extension.installMetadata, cwd);
        const updatedExtensionStorage = new ExtensionStorage(extension.config.name);
        const updatedExtension = loadExtension(updatedExtensionStorage.getExtensionDir());
        if (!updatedExtension) {
            throw new Error('Updated extension not found after installation.');
        }
        const updatedVersion = updatedExtension.config.version;
        return {
            name: extension.config.name,
            originalVersion,
            updatedVersion,
        };
    }
    catch (e) {
        console.error(`Error updating extension, rolling back. ${getErrorMessage(e)}`);
        await copyExtension(tempDir, extension.path);
        throw e;
    }
    finally {
        await fs.promises.rm(tempDir, { recursive: true, force: true });
    }
}
export function disableExtension(name, scope, cwd = process.cwd()) {
    if (scope === SettingScope.System || scope === SettingScope.SystemDefaults) {
        throw new Error('System and SystemDefaults scopes are not supported.');
    }
    const settings = loadSettings(cwd);
    const settingsFile = settings.forScope(scope);
    const extensionSettings = settingsFile.settings.extensions || {
        disabled: [],
    };
    const disabledExtensions = extensionSettings.disabled || [];
    if (!disabledExtensions.includes(name)) {
        disabledExtensions.push(name);
        extensionSettings.disabled = disabledExtensions;
        settings.setValue(scope, 'extensions', extensionSettings);
    }
}
export function enableExtension(name, scopes) {
    removeFromDisabledExtensions(name, scopes);
}
/**
 * Removes an extension from the list of disabled extensions.
 * @param name The name of the extension to remove.
 * @param scope The scopes to remove the name from.
 */
function removeFromDisabledExtensions(name, scopes, cwd = process.cwd()) {
    const settings = loadSettings(cwd);
    for (const scope of scopes) {
        const settingsFile = settings.forScope(scope);
        const extensionSettings = settingsFile.settings.extensions || {
            disabled: [],
        };
        const disabledExtensions = extensionSettings.disabled || [];
        extensionSettings.disabled = disabledExtensions.filter((extension) => extension !== name);
        settings.setValue(scope, 'extensions', extensionSettings);
    }
}
export async function updateAllUpdatableExtensions(cwd = process.cwd()) {
    const extensions = loadExtensions(cwd).filter((extension) => !!extension.installMetadata);
    return await Promise.all(extensions.map((extension) => updateExtension(extension, cwd)));
}
//# sourceMappingURL=extension.js.map