#!/usr/bin/env python3
#BEGIN_LEGAL
#
#Copyright (c) 2025 Intel Corporation
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.
#  
#END_LEGAL
"""
Python wrapper function generator for XED API.

This utility generates Python-friendly wrapper functions for XED's C API.
Creates functions with the _py suffix that provide more convenient interfaces
for Python bindings (ctypes or similar).

Usage: python gen_py_wrappers.py --xeddir <xed_source_dir> --gendir <output_dir>
"""
# Type hints imports
from __future__ import annotations  # Enables deferred type hints
from typing import TYPE_CHECKING, Optional, TextIO
if TYPE_CHECKING: # Only for type hints, prevent recursive imports
    from generator import all_generator_info_t

import argparse
import re
import sys
from pathlib import Path
import codegen

FUNC_SPECIFIER = 'XED_DLL_EXPORT'
PY_FUNC_SUFFIX = '_py'

def setup() -> argparse.Namespace:
    """Sets up argument parsing."""
    parser = argparse.ArgumentParser(description='Generate Py wrappers lists')
    parser.add_argument('--xeddir',  
                        default='.', 
                        help='XED Directory')
    parser.add_argument('--gendir',  
                        default='.', 
                        help='Output directory')
    args = parser.parse_args()
    return args

def extract_return_type(func_match: list[str]) -> str:
    """
    Extracts the return type from a function signature.
    
    Args:
        func_match (list): A list containing the entire match for a XED function.
    
    Returns:
        str: The return type without any function specifiers.
    """
    return_decl: str = func_match[0].strip()
    long_return_types: list[str] = ['const ', 'const*', 'unsigned ', 'char']

    # Find the earliest match for multi-word return types
    indices: list[int] = [return_decl.find(sub) for sub in long_return_types if return_decl.find(sub) != -1]

    if indices:
        idx: int = min(indices)
        return_type: str = return_decl[idx:]
    else:
        return_type = return_decl.split()[-1]

    return ' '.join(return_type.split())  # Remove extra spaces


def gen_func_ifdef(orig_func_name: str) -> str:
    """
    Generates an ifdef guard for operand accessor functions.
    
    Args:
        orig_func_name (str): The original function name.
    
    Returns:
        str: The ifdef preprocessor directive for the function.
    """
    operand_pattern: str = r'xed3_operand_[gs]et_'
    xed_operand: str = re.sub(operand_pattern, '', orig_func_name).upper()
    return f'#if defined(XED_OPERAND_{xed_operand}_DEFINED)\n'

def write_file_header_comment(file_path: Path, cfe: codegen.xed_file_emitter_t) -> None:
    """Writes a comment in the output file indicating the processed header file."""
    file_comment = f'/* Export functions from: {file_path.name} */'
    cfe.write('/' * len(file_comment) + '\n')
    cfe.write(file_comment + '\n')
    cfe.write('/' * len(file_comment) + '\n')
    cfe.write(f'#include "{file_path.name}"\n')

def process_header_file(file_path: Path, scanned_files: set[str], generated_functions: set[str], cfe = codegen.xed_file_emitter_t) -> None:
    """Processes a header file and generates exported functions."""
    filename = file_path.name
    if filename in scanned_files:
        return
    scanned_files.add(filename)
    try:
        with file_path.open('r') as file:
            content = file.read()
    except Exception as e:
        print(f"Error reading {file_path}: {e}", file=sys.stderr)
        return

    func_pattern = r'\n(?!\s*#)(?!\s*//)\s*(\w[\w\s\*]+)\s+(\w+)\s*\((.*?)\)\s*[\{\;]'
    func_matches: list[tuple] = re.findall(func_pattern, content, re.DOTALL)

    if not func_matches:
        return

    write_file_header_comment(file_path, cfe)

    # Safe declaration is needed if the file is not autogenerated on build phase.
    # We lock it to False and keep the special operand treatments if needed in the future
    # The 'safe_declaration' flag is used to determine whether the function declaration
    # should be guarded with additional preprocessor checks. This is typically necessary
    # if the wrapper file is not automatically generated during the build phase.
    # Currently, this flag is set to False, but it can be enabled for special cases
    # involving operand treatments if required in the future.
    safe_decl: bool = False # filename in ['xed-operand-accessors.h']
    
    for func_match in func_matches:
        return_type: str = extract_return_type(func_match)
        orig_func_name: str = func_match[1].strip()
        orig_args_with_types: str = func_match[2].strip()

        if '...' in orig_args_with_types \
            or return_type in ['typedef', 'return', '__asm__'] \
            or orig_func_name in generated_functions:
            continue # skip

        generated_functions.add(orig_func_name)

        # Extract API args w/o types
        if orig_args_with_types != 'void':
            args = ', '.join([arg.split()[-1] for arg in orig_args_with_types.split(',')])
        else:
            args = ''

        func_return_token = 'return ' if 'void' not in return_type else ''

        # Generate py wrapper
        func_decl_py: str = f'{return_type} {orig_func_name}{PY_FUNC_SUFFIX}({orig_args_with_types})'

        xed_function: str = f'{FUNC_SPECIFIER} {func_decl_py} {{\n'
        xed_function += f'    {func_return_token}{orig_func_name}({args});\n'
        xed_function += '}\n'

        # Add ifdef guards for certain functions
        if safe_decl:
            xed_function = gen_func_ifdef(orig_func_name) + xed_function
            xed_function += '#endif\n'
        xed_function += '\n'
        cfe.write(xed_function)

def get_src_headers(xeddir: str, agi: Optional[all_generator_info_t]) -> list[Path]:
    """Collects all source header files, excluding predefined headers."""
    # Files to be excluded during header processing
    EXCLUDED_HEADERS = {'xed-util.h', 'xed-portability.h', 'xed-encode-direct.h', 'xed-encode-check.h'}
    headers = []
    if agi:
        # collect dynamically generated headers
        gen_hdrs = [Path(h) for h in agi.hdr_files if 'enum' in h] # Generated enums
        gen_hdrs.append(Path('obj/xed-operand-accessors.h'))
        gen_hdrs.append(Path('obj/xed-operand-ctype-map.h'))
        headers.extend(gen_hdrs)
        # collect predefined, static headers (some are excluded)
        static_hdrs = Path(xeddir, 'include', 'public', 'xed')
        headers.extend(h for h in static_hdrs.glob('*.h') if h.name not in EXCLUDED_HEADERS)
    else:
        # Collect from the obj/kit "include" directory
        headers_path = Path(xeddir, 'obj/wkit/include/xed')
        for file in headers_path.rglob('*.h'):
            if file.name not in EXCLUDED_HEADERS:
                headers.append(file)
    return sorted(headers, key=lambda x: x.name)

def gen(options: argparse.Namespace, agi: Optional[all_generator_info_t]) -> str:
    """Main function to generate exported functions."""
    scanned_files = set()
    generated_functions = set()

    cfp = codegen.xed_file_emitter_t(options.xeddir,
                                     options.gendir,
                                     'xed-export-functions.c',
                                     is_private=False)
    cfp.start()
    headers: list[Path] = get_src_headers(options.xeddir, agi)
    for header_path in headers:
        process_header_file(header_path, scanned_files, generated_functions, cfp)

    cfp.close()
    return cfp.full_file_name

if __name__ == "__main__":
    args = setup()
    output_file_path = gen(args, None)
    print(f"Generated file: {output_file_path}")
    sys.exit(0)
