"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exportsForTesting = void 0;
exports.findNvim = findNvim;
const node_child_process_1 = require("node:child_process");
const node_path_1 = require("node:path");
const node_fs_1 = require("node:fs");
const versionRegex = /^(\d+)\.(\d+)\.(\d+)(?:-(.+))?$/;
const nvimVersionRegex = /^[nN][vV][iI][mM]\s+v?(.+)$/m;
const buildTypeRegex = /^Build\s+type:\s+(.+)$/m;
const luaJitVersionRegex = /^LuaJIT\s+(.+)$/m;
const windows = process.platform === 'win32';
function parseVersion(version) {
    if (typeof version !== 'string') {
        throw new TypeError('Invalid version format: not a string');
    }
    const match = version.match(versionRegex);
    if (!match) {
        return undefined;
    }
    const [, major, minor, patch, prerelease] = match;
    if (major === undefined || minor === undefined || patch === undefined) {
        throw new TypeError(`Invalid version string: "${version}"`);
    }
    const majorNumber = Number(major);
    const minorNumber = Number(minor);
    const patchNumber = Number(patch);
    const versionParts = [majorNumber, minorNumber, patchNumber];
    if (prerelease !== undefined) {
        versionParts.push(prerelease);
    }
    else {
        versionParts.push('zzz');
    }
    return versionParts;
}
/**
 * Compares two versions.
 * @param a - First version to compare.
 * @param b - Second version to compare.
 * @returns -1 if a < b, 0 if a == b, 1 if a > b.
 * @throws {TypeError} If the versions are not valid.
 *
 * Format could be:
 * - 0.9.1
 * - 0.10.0-dev-658+g06694203e-Homebrew
 */
function compareVersions(a, b) {
    var _a, _b;
    const versionA = parseVersion(a);
    const versionB = parseVersion(b);
    if (versionA === undefined) {
        throw new TypeError(`Invalid version: "${a}"`);
    }
    if (versionB === undefined) {
        return 1;
    }
    const length = Math.min(versionA.length, versionB.length);
    for (let i = 0; i < length; i = i + 1) {
        const partA = (_a = versionA[i]) !== null && _a !== void 0 ? _a : 0;
        const partB = (_b = versionB[i]) !== null && _b !== void 0 ? _b : 0;
        if (partA < partB) {
            return -1;
        }
        if (partA > partB) {
            return 1;
        }
    }
    if (versionB.length > versionA.length) {
        return -1;
    }
    return 0;
}
function normalizePath(path) {
    return (0, node_path_1.normalize)(windows ? path.toLowerCase() : path);
}
function getPlatformSearchDirs() {
    const paths = new Set();
    const { PATH, USERPROFILE, LOCALAPPDATA, PROGRAMFILES, HOME } = process.env;
    PATH === null || PATH === void 0 ? void 0 : PATH.split(node_path_1.delimiter).forEach(p => paths.add(normalizePath(p)));
    // Add common Neovim installation paths not always in the system's PATH.
    if (windows) {
        // Scoop common install location
        if (USERPROFILE) {
            paths.add(normalizePath(`${USERPROFILE}/scoop/shims`));
        }
        paths.add(normalizePath('C:/ProgramData/scoop/shims'));
        // Winget common install location
        // See https://github.com/microsoft/winget-cli/blob/master/doc/specs/%23182%20-%20Support%20for%20installation%20of%20portable%20standalone%20apps.md
        if (LOCALAPPDATA) {
            paths.add(normalizePath(`${LOCALAPPDATA}/Microsoft/WindowsApps`));
            paths.add(normalizePath(`${LOCALAPPDATA}/Microsoft/WinGet/Packages`));
        }
        if (PROGRAMFILES) {
            paths.add(normalizePath(`${PROGRAMFILES}/Neovim/bin`));
            paths.add(normalizePath(`${PROGRAMFILES} (x86)/Neovim/bin`));
            paths.add(normalizePath(`${PROGRAMFILES}/WinGet/Packages`));
            paths.add(normalizePath(`${PROGRAMFILES} (x86)/WinGet/Packages`));
        }
    }
    else {
        // Common paths for Unix-like systems
        [
            '/usr/local/bin',
            '/usr/bin',
            '/opt/homebrew/bin',
            '/home/linuxbrew/.linuxbrew/bin',
            '/snap/nvim/current/usr/bin',
        ].forEach(p => paths.add(p));
        if (HOME) {
            paths.add(normalizePath(`${HOME}/bin`));
            paths.add(normalizePath(`${HOME}/.linuxbrew/bin`));
        }
    }
    return paths;
}
/**
 * Tries to find a usable `nvim` binary on the current system.
 *
 * @param opt.minVersion See {@link FindNvimOptions.minVersion}
 * @param opt.orderBy See {@link FindNvimOptions.orderBy}
 * @param opt.firstMatch See {@link FindNvimOptions.firstMatch}
 * @param opt.paths See {@link FindNvimOptions.paths}
 * @param opt.dirs See {@link FindNvimOptions.dirs}
 */
function findNvim(opt = {}) {
    var _a, _b, _c;
    const platformDirs = getPlatformSearchDirs();
    const nvimExecutable = windows ? 'nvim.exe' : 'nvim';
    const normalizedPathsFromUser = ((_a = opt.paths) !== null && _a !== void 0 ? _a : []).map(normalizePath);
    const allPaths = new Set([
        ...normalizedPathsFromUser,
        ...((_b = opt.dirs) !== null && _b !== void 0 ? _b : []).map(dir => normalizePath((0, node_path_1.join)(dir, nvimExecutable))),
        ...[...platformDirs].map(dir => (0, node_path_1.join)(dir, nvimExecutable)),
    ]);
    const matches = new Array();
    const invalid = new Array();
    for (const nvimPath of allPaths) {
        if ((0, node_fs_1.existsSync)(nvimPath) || normalizedPathsFromUser.includes(nvimPath)) {
            try {
                (0, node_fs_1.accessSync)(nvimPath, node_fs_1.constants.X_OK);
                // TODO: fallback to `echo 'print(vim.version())' | nvim -l -` if parsing --version fails.
                const nvimVersionFull = (0, node_child_process_1.execFileSync)(nvimPath, ['--version']).toString();
                const nvimVersionMatch = nvimVersionRegex.exec(nvimVersionFull);
                const buildTypeMatch = buildTypeRegex.exec(nvimVersionFull);
                const luaJitVersionMatch = luaJitVersionRegex.exec(nvimVersionFull);
                if (nvimVersionMatch && buildTypeMatch && luaJitVersionMatch) {
                    if ('minVersion' in opt &&
                        compareVersions((_c = opt.minVersion) !== null && _c !== void 0 ? _c : '0.0.0', nvimVersionMatch[1]) === 1) {
                        invalid.push({
                            nvimVersion: nvimVersionMatch[1],
                            path: nvimPath,
                            buildType: buildTypeMatch[1],
                            luaJitVersion: luaJitVersionMatch[1],
                        });
                    }
                    else {
                        matches.push({
                            nvimVersion: nvimVersionMatch[1],
                            path: nvimPath,
                            buildType: buildTypeMatch[1],
                            luaJitVersion: luaJitVersionMatch[1],
                        });
                        if (opt.firstMatch) {
                            return {
                                matches,
                                invalid,
                            };
                        }
                    }
                }
            }
            catch (e) {
                invalid.push({
                    path: nvimPath,
                    error: e,
                });
            }
        }
    }
    if (opt.orderBy === undefined || opt.orderBy === 'desc') {
        matches.sort((a, b) => { var _a, _b; return compareVersions((_a = b.nvimVersion) !== null && _a !== void 0 ? _a : '0.0.0', (_b = a.nvimVersion) !== null && _b !== void 0 ? _b : '0.0.0'); });
    }
    return {
        matches,
        invalid,
    };
}
// .mocharc.js sets NODE_ENV=test.
if (process.env.NODE_ENV === 'test') {
    // These functions are intentionally not exported. After `nvim` is found, clients can use Nvim's
    // own `vim.version` module, so node-client shouldn't expose a half-baked "semver" implementation.
    exports.exportsForTesting = {
        parseVersion,
        compareVersions,
        normalizePath,
    };
}
