/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TGUI - Texus' Graphical User Interface
// Copyright (C) 2012-2026 Bruno Van de Velde (vdv_b@tgui.eu)
//
// This software is provided 'as-is', without any express or implied warranty.
// In no event will the authors be held liable for any damages arising from the use of this software.
//
// Permission is granted to anyone to use this software for any purpose,
// including commercial applications, and to alter it and redistribute it freely,
// subject to the following restrictions:
//
// 1. The origin of this software must not be misrepresented;
//    you must not claim that you wrote the original software.
//    If you use this software in a product, an acknowledgment
//    in the product documentation would be appreciated but is not required.
//
// 2. Altered source versions must be plainly marked as such,
//    and must not be misrepresented as being the original software.
//
// 3. This notice may not be removed or altered from any source distribution.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef TGUI_SPRITE_HPP
#define TGUI_SPRITE_HPP

#include <TGUI/Texture.hpp>
#include <TGUI/Vector2.hpp>
#include <TGUI/Rect.hpp>
#include <TGUI/Color.hpp>
#include <TGUI/RenderStates.hpp>

#include <vector>
#include <memory>

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace tgui
{
    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    /// @internal
    class TGUI_API Sprite
    {
    public:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief The way the image should be scaled
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        enum class ScalingType
        {
            Normal,     //!< The image is not split and scaled normally
            Horizontal, //!< Image is split in Left, Middle and Right parts. Left and Right keep ratio, Middle gets stretched
            Vertical,   //!< Image is split in Top, Middle and Bottom parts. Top and Bottom keep ratio, Middle gets stretched
            NineSlice   //!< Image is split in 9 parts. Corners keep size, sides are stretched in one direction, middle is stretched in both directions
        };

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    public:

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Default constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite() = default;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Constructor that immediately sets the texture
        ///
        /// @param texture  Texture to use in the sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite(const Texture& texture);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Copy constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite(const Sprite&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Move constructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite(Sprite&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Destructor
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        ~Sprite();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Overload of copy assignment operator
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite& operator=(const Sprite&);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Move assignment
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        Sprite& operator=(Sprite&&) noexcept;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the texture
        ///
        /// @param texture  New texture
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setTexture(const Texture& texture);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the texture used by this sprite
        ///
        /// @return Texture of the sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD const Texture& getTexture() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns whether a texture was set
        ///
        /// @return Has a valid texture been assigned to this sprite?
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isSet() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the size that the image will have when drawing
        ///
        /// @param size  Size of the sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setSize(Vector2f size);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the size that the image has when drawing
        ///
        /// @return Size of the sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getSize() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the opacity of the texture
        ///
        /// @param opacity  The opacity of the texture. 0 means completely transparent, while 1 (default) means fully opaque
        ///
        /// The alpha component of the color specified with setColor is multiplied with this factor.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setOpacity(float opacity);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the opacity of the texture
        ///
        /// @return The opacity of the texture. 0 means completely transparent, while 1 (default) means fully opaque
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD float getOpacity() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Changes the part of the sprite that should be drawn
        ///
        /// @param visibleRect Visible part of the sprite
        ///
        /// Set this to (0, 0, 0, 0) to show the entire sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setVisibleRect(const FloatRect& visibleRect);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the part of the sprite that is drawn
        ///
        /// @return Visible part of the sprite
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD FloatRect getVisibleRect() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets the position of the sprite
        ///
        /// @param position  New sprite position
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setPosition(Vector2f position);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets the position of the sprite
        ///
        /// @return Sprite position
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD Vector2f getPosition() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Sets rotation of the sprite
        ///
        /// @param angle  Degrees to rotate counterclockwise
        ///
        /// Rotation can currently only be a multiple of 90 degrees.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void setRotation(float angle);

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Gets rotation of the sprite
        ///
        /// @return Degrees rotated counterclockwise
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD float getRotation() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Checks if a certain pixel is transparent
        ///
        /// @param pos  Coordinate of the pixel
        ///
        /// @return True when the pixel is transparent, false when it is not
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD bool isTransparentPixel(Vector2f pos) const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @brief Returns the way in which the image is being scaled
        ///
        /// @return Scaling type
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD ScalingType getScalingType() const;

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// Returns the internal SVG texture for drawing.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD const std::shared_ptr<BackendTexture>& getSvgTexture() const
        {
            return m_svgTexture;
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// Returns the internal vertices for drawing.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD const std::vector<Vertex>& getVertices() const
        {
            return m_vertices;
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// Returns the internal vertices for drawing.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        TGUI_NODISCARD const std::vector<unsigned int>& getIndices() const
        {
            return m_indices;
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        /// @internal
        /// Updates the location of the vertices. In case of an SVG texture, this function also performs the rasterization again.
        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        void updateVertices();

        /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    private:

        Vector2f    m_size;
        Texture     m_texture;
        std::shared_ptr<BackendTexture> m_svgTexture;
        std::vector<Vertex> m_vertices;
        std::vector<unsigned int> m_indices;

        FloatRect   m_visibleRect;

        Color       m_vertexColor = Color::White;
        float       m_opacity = 1;
        float       m_rotation = 0;
        Vector2f    m_position;

        ScalingType m_scalingType = ScalingType::Normal;
    };

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // TGUI_SPRITE_HPP
