[comment {-*- tcl -*- doctools manpage}]
[manpage_begin "Security and access control" n ""]
[copyright "2003-2010, Ashok P. Nadkarni"]
[moddesc "Security commands"]
[titledesc "Commands related to authentication and access control"]
[require twapi_security]
[description]
[para]

This module provides procedures related to security and access
control in Windows operating systems.
[para]
This documentation is reference material that assumes familiarity with
Windows security terms and concepts.
For more introductory material and guide, see the [emph "Windows Security"]
chapter in the
[uri https://www.magicsplat.com/book "Tcl on Windows"] online book.

[section "Security Identifiers"]
Each trustee (user, group, logon session etc.) is identified by a
[emph "security identifier"] (SID).
[uri #equal_sids [cmd equal_sids]] compares two SIDs for equality
while
[uri #sids_from_same_domain [cmd sids_from_same_domain]]
checks if two SIDs belong to the same domain.
[uri #well_known_sid [cmd well_known_sid]]
returns well known SIDs as defined by Microsoft and
[uri #is_well_known_sid [cmd is_well_known_sid]]
checks if a given SID is one of these well known SIDs.
[uri #get_sid_domain [cmd get_sid_domain]]
returns the SID of the domain containing a specified SID.

[section "Locally Unique Identifiers"]

A [emph "Locally Unique Identifier (LUID)"] is a string generated by the
command [uri \#new_luid [cmd new_luid]]
that is guaranteed to be unique on that system
for that boot session. Unique means it will not be
generated by some other call to that command.

[section "Access Tokens and Privileges"]
An access token is associated with a process or a thread and contains
various elements that determine the security context for the process or thread.

[para]
The commands [uri \#open_process_token [cmd open_process_token]]
and [uri \#open_thread_token [cmd open_thread_token]]
retrieve the access tokens associated with a process and thread
respectively. An access token for a specific user account may be
obtained using [uri \#open_user_token [cmd open_user_token]].
 The returned tokens
can be passed to other functions to retrieve or
set information for the process or thread's privileges and other
security related information.
When no longer required, the tokens must be passed
to [uri \#close_token [cmd close_token]] in order to release resources.

[para]
The command [uri #get_token_info [cmd get_token_info]] retrieves
information from a token. Individual pieces of the token can also
be retrieved via commands [uri \#get_token_user [cmd get_token_user]],
[uri \#get_token_default_dacl [cmd get_token_default_dacl]],
[uri \#get_token_origin [cmd get_token_origin]],
[uri \#get_token_groups [cmd get_token_groups]],
[uri \#get_token_primary_group [cmd get_token_primary_group]],
[uri \#get_token_owner [cmd get_token_owner]],
[uri \#get_token_tssession [cmd get_token_tssession]],
[uri \#get_token_groups_and_attrs [cmd get_token_groups_and_attrs]]
and
[uri \#get_token_restricted_groups_and_attrs [cmd get_token_restricted_groups_and_attrs]].

[para]
The commands [uri \#get_token_privileges [cmd get_token_privileges]],
[uri \#check_enabled_privileges [cmd check_enabled_privileges]]
and [uri \#get_token_privileges_and_attrs [cmd get_token_privileges_and_attrs]]
retrieve information related
to the privileges associated with a token. The commands
[uri \#enable_token_privileges [cmd enable_token_privileges]],
[uri \#disable_token_privileges [cmd disable_token_privileges]] and
[uri \#disable_all_token_privileges [cmd disable_all_token_privileges]]
allow manipulation of the same.
Similarly, [uri \#enable_privileges [cmd enable_privileges]]
and [uri \#disable_privileges [cmd disable_privileges]] can
be used to enable and disable privileges for the current process.
[uri \#eval_with_privileges [cmd eval_with_privileges]]
evaluates a script after enabling specified
privileges.
[para]
The command [uri \#get_privilege_description [cmd get_privilege_description]]
returns a description of a privilege. An empty string is returned if
the specified privilege is not a known privilege.

[para]
The commands [uri \#get_token_type [cmd get_token_type]]
and [uri \#get_token_impersonation_level [cmd get_token_impersonation_level]]
return information related to impersonation.

[para]
[uri \#get_token_source [cmd get_token_source]] returns the source of a token.
[uri \#get_token_statistics [cmd get_token_statistics]]
returns various other pieces of information, such
as expiration, memory usage etc.

[para]
The commands [uri \#map_luid_to_privilege [cmd map_luid_to_privilege]]
and [uri \#map_privilge_to_luid [cmd map_privilge_to_luid]]
map LUID's to privilege names.
The command [uri \#is_valid_luid_syntax [cmd is_valid_luid_syntax]]
can be used to check the syntactic validity of a LUID.

[section Impersonation]
Windows allows a thread or process to impersonate a user different from
the account in which it was started (given the proper privileges). The
functions for impersonation and reverting back are
[uri \#impersonate_token [cmd impersonate_token]],
[uri \#impersonate_user [cmd impersonate_user]],
[uri \#impersonate_self [cmd impersonate_self]],
[uri \#set_thread_token [cmd set_thread_token]]
and [uri \#revert_to_self [cmd revert_to_self]].
For impersonating named pipe clients, use the
[uri namedpipe.html#impersonate_namedpipe_client [cmd impersonate_namedpipe_client]]
command from the  [cmd twapi_namedpipe] package.
[para]
The command [uri \#get_current_user [cmd get_current_user]]
returns the user account for the thread
taking into consideration any ongoing impersonation.

[para]
Impersonation may be done at several levels. These levels control the degree
to which the impersonator takes on the identity of the impersonated client.
The following constants define the different levels:
[list_begin opt]
[opt_def [const anonymous]] The impersonating process cannot
obtain identification information about the client or impersonate it.
[opt_def [const identification]] The impersonating process cannot impersonate
the client (ie. access resources using the client's credentials) but can
obtain identity and privilege information about the client. This can be
used for controlling access to its own objects (as opposed to operating
system resources).
[opt_def [const impersonation]] The impersonating process can impersonate
the client only on the local system. The client credentials cannot be
used to access other resources on the network.
[opt_def [const delegation]] The impersonating process can use the client's
credentials for accessing network resources.
[list_end]

[section "Access Rights"]

Access rights in the Windows API take the form of a bit mask with each
bit signifying a specific desired or granted right. Access rights may
be specified to TWAPI functions in the form of a single element or
list of elements, each of which is an integer or a symbolic constant.
The access rights bit mask passed to the underlying function is formed
by OR'ing together the individual elements in the list. In the case of a
symbolic constant, the access right is mapped to the appropriate bits before
being OR'ed with the other elements.

[para]
The following symbolic constants define generic access rights that are
mapped by each object type into its own specific rights: [const generic_read],
[const generic_write], [const generic_execute] and [const generic_all].

[para]
The following symbolic constants define access rights that are common
to all securable object types: [const delete], [const read_control],
[const write_dac], [const write_owner], [const synchronize],
[const standard_rights_read], [const standard_rights_write],
[const standard_rights_execute],
[const standard_rights_required] and [const standard_rights_all].
Some APIs will also accept [const maximum_allowed] to indicate the
the maximum permitted access to the resource for the process is desired.
Refer to the Windows SDK for a definition of these access rights.

[para]
In addition to the standard rights,
specific object types may have other access rights associated with them as
described in the following table.

[list_begin opt]

[opt_def COM]
[const com_rights_execute],
[const com_rights_execute_local],
[const com_rights_execute_remote],
[const com_rights_activate_local],
[const com_rights_activate_remote]


[opt_def Desktops]
[const desktop_readobjects],
[const desktop_createwindow],
[const desktop_createmenu],
[const desktop_hookcontrol],
[const desktop_journalrecord],
[const desktop_journalplayback],
[const desktop_enumerate],
[const desktop_writeobjects],
[const desktop_switchdesktop]

[opt_def Events]
[const event_all_access], [const event_modify_state]

[opt_def Files]
[const file_all_access],
[const file_generic_read], [const file_generic_write],
[const file_generic_execute], [const file_read_data], [const file_write_data],
[const file_append_data], [const file_read_ea], [const file_write_ea],
[const file_execute], [const file_delete_child], [const file_read_attributes],
[const file_write_attributes]

[opt_def Mutexes]
[const mutex_all_access], [const mutex_modify_state]

[opt_def "Named pipes"] [const file_all_access],
[const file_read_data], [const file_write_data],
[const file_create_pipe_instance], [const file_read_attributes],
[const file_write_attributes]

[opt_def Processes]
[const process_all_access],
[const process_terminate], [const process_create_thread],
[const process_set_sessionid], [const process_vm_operation],
[const process_vm_read], [const process_vm_write], [const process_dup_handle],
[const process_create_process], [const process_set_quota],
[const process_set_information],
[const process_query_information], [const process_suspend_resume]

[opt_def "Registry"]
[const key_all_access],
[const key_read], [const key_write], [const key_execute],
[const key_query_value], [const key_set_value],
[const key_create_sub_key], [const key_enumerate_sub_keys],
[const key_notify], [const key_create_link],
[const key_wow64_32key], [const key_wow64_64key], [const key_wow64_res]

[opt_def Semaphores]
[const semaphore_all_access], [const semaphore_modify_state]

[opt_def "Services"] [const service_all_access],
[const service_query_config], [const service_change_config],
[const service_query_status], [const service_enumerate_dependents],
[const service_start], [const service_stop], [const service_pause_continue],
[const service_interrogate], [const service_user_defined_control]

[opt_def Threads]
[const thread_all_access],
[const thread_terminate], [const thread_suspend_resume],
[const thread_get_context], [const thread_set_context],
[const thread_set_information], [const thread_query_information],
[const thread_set_thread_token], [const thread_impersonate],
[const thread_direct_impersonation]

[opt_def Timers]
[const timer_all_access],
[const timer_query_state], [const timer_modify_state].

[opt_def Tokens]
[const token_all_access],
[const token_assign_primary], [const token_duplicate],
[const token_impersonate], [const token_query], [const token_query_source],
[const token_adjust_privileges], [const token_adjust_groups],
[const token_adjust_default], [const token_adjust_sessionid].

[opt_def "Window stations"]
[const winsta_enumdesktops],
[const winsta_readattributes],
[const winsta_accessclipboard],
[const winsta_createdesktop],
[const winsta_writeattributes],
[const winsta_accessglobalatoms],
[const winsta_exitwindows],
[const winsta_enumerate],
[const winsta_readscreen],
[const winsta_all].

[list_end]


[para]
Note that some of the symbolic access rights are a combination of other
access rights. For example, [const file_all_access] includes all other
rights associated with a file object. When an access rights list is returned
from a command, it includes both the individual symbolic rights as well as
all valid symbols for combined rights.

[section "Access Control Entries"]

Several functions deal with access control entries (ACE) either directly or as
part of the more complex structures. A new ACE may be created through
the [uri \#new_ace [cmd new_ace]] command.
The structure of an ACE should be treated as opaque and
the fields should be manipulated only through the access control entry
functions. An access control entry contains the fields described below.

[para]
The type of an ACE may be one of the following values:
[const allow], [const deny], [const audit],
[const alarm], [const allow_compound], [const allow_object],
[const deny_object], [const audit_object], [const alarm_object],
[const allow_callback], [const deny_callback], [const allow_callback_object],
[const deny_callback_object], [const audit_callback], [const alarm_callback],
[const audit_callback_object], [const alarm_callback_object],
[const mandatory_label]. Currently, only
the [const allow], [const deny] and [const audit] type ACE's are supported.
The type of an ACE may be manipulated through the
[uri \#get_ace_type [cmd get_ace_type]] and
[uri \#set_ace_type [cmd set_ace_type]] commands.

[para]
Each ACE has an SID associated with it that indicates the user or group
that the ACE applies to. This may be retrieved or set using the
[uri \#get_ace_sid [cmd get_ace_sid]]
and [uri \#set_ace_sid [cmd set_ace_sid]] functions.

[para]
The [uri \#accessrights "access rights"]
granted or restricted by an ACE can be manipulated through
the [uri \#get_ace_rights [cmd get_ace_rights]]
and [uri \#set_ace_rights [cmd set_ace_rights]] functions.

[para]
An ACE may be recursively applied to objects within a container. This is
controlled through the [uri \#get_ace_inheritance [cmd get_ace_inheritance]]
and [uri \#set_ace_inheritance [cmd set_ace_inheritance]]
commands.

[para]
The command [uri \#sort_aces [cmd sort_aces]]
may be used to sort a list of ACE elements into
the order recommended by Windows.

[para]
The command
[uri \#get_ace_text [cmd get_ace_text]] returns a textual description of an ACE.

[section "Access Control List"]
An access control list (ACL) is an ordered list of
[uri \#accesscontrolentries "access control entries"]
that is applied to a resource.
The command [uri \#new_acl [cmd new_acl]]
creates a new ACL with an optional set of ACE
elements. For the common case of granting access only specific accounts,
[uri #new_restricted_dacl  [cmd new_restricted_dacl]] provides a
simpler and more convenient interface.


[para]
The commands [uri \#get_acl_aces [cmd get_acl_aces]]
and [uri \#set_acl_aces [cmd set_acl_aces]] may be
used to retrieve or set the access control entries in the ACL. Alternatively,
the commands [uri \#prepend_acl_aces [cmd prepend_acl_aces]]
and [uri \#append_acl_aces [cmd append_acl_aces]] may be
used to place additional ACEs in front of or after, respectively, of the
current entries in an ACL.

[para]
Access control lists are of two types - a discretionary access control list
(DACL) controls the operations that may be invoked on a resource based on
the identity of the accessor, while a system access control list (SACL)
controls the audit events that are generated when the resource is accessed.
A SACL should contain only ACE elements of types prefixed with [const audit]
or [const mandatory_label]
while a DACL should only contain ACE elements of other types.

[para]
The list of ACE elements in an ACL may be resorted to conform to the
Windows recommended order through the [uri \#sort_acl_aces [cmd sort_acl_aces]]
command.

[section "Security Descriptors"]
A security descriptor is associated with a resource and ties together various
information that controls access to the resource including the SID of the
owner of the resource, the SID of the primary group, a discretionary access
control list (DACL) and a system access control list (SACL).

[para]
A new security descriptor may be created through the command
[uri \#new_security_descriptor [cmd new_security_descriptor]].

[para]
The commands
[uri \#get_security_descriptor_owner [cmd get_security_descriptor_owner]],
[uri \#get_security_descriptor_group [cmd get_security_descriptor_group]],
[uri \#set_security_descriptor_owner [cmd set_security_descriptor_owner]]
and [uri \#set_security_descriptor_group [cmd set_security_descriptor_group]]
retrieve and set the owner and group
attributes of a security descriptor.

[para]
The commands [uri \#get_security_descriptor_dacl [cmd get_security_descriptor_dacl]],
[uri \#get_security_descriptor_sacl [cmd get_security_descriptor_sacl]],
[uri \#set_security_descriptor_dacl [cmd set_security_descriptor_dacl]]
and [uri \#set_security_descriptor_sacl [cmd set_security_descriptor_sacl]]
retrieve and set the DACL and SACL
fields of a security descriptor.
It is important to distinguish between the absence of a DACL (or SACL) in
a security descriptor and an empty DACL (or SACL). The former results in
everyone being allowed to access the object whereas the latter results in
no one being allowed to access the object. When modifying or creating
a security descriptor, pass the literal string [const null] to indicate
that the security descriptor has no DACL (or SACL). An empty DACL (or SACL)
on the other hand is indicated simply by passing an ACL that does not
contain any ACEs.

[para]
The command
[uri \#get_security_descriptor_control [cmd get_security_descriptor_control]]
returns the control flags associated with the security descriptor.

[para]
Note that not all fields in a security descriptor need to be initialized.
For example, to change the owner of a resource, you only need to create
a descriptor with the owner field initialized either
at creation time or through a subsequent call
to [uri \#set_security_descriptor_owner [cmd set_security_descriptor_owner]].
 Then this can be passed to
the appropriate function to set a resource's security with an option
indicating that only the owner field should be changed.

[para]
The security information associated with each resource type can be retrieved
and modified through functions
[uri \#get_resource_security_descriptor [cmd get_resource_security_descriptor]]
and [uri \#set_resource_security_descriptor [cmd set_resource_security_descriptor]].

[para]
The command
[uri \#get_security_descriptor_text [cmd get_security_descriptor_text]]
returns a textual description of a security descriptor. Commands
[uri #security_descriptor_to_sddl [cmd security_descriptor_to_sddl]] and
[uri #sddl_to_security_descriptor [cmd sddl_to_security_descriptor]] can
be used to convert security descriptors to SDDL format.
Commands 
[uri #encode_security_descriptor [cmd encode_security_descriptor]] and
[uri #decode_security_descriptor [cmd decode_security_descriptor]]
encode and decode security descriptors to and from
self-relative binary formats. This is useful in cases where security
descriptors need to be stored in that format, for example in the registry.

[section "Logon Sessions"]

A logon session is identified by an [uri \#uniqueidentifiers "LUID"] and
provides the context for a particular instance of an authenticated security
principal where the principal may be a user, a computer, a service etc.
A single principal, for example a user, may have multiple logon sessions
concurrently present at any time. The command
[uri \#find_logon_sessions [cmd find_logon_sessions]]
allows enumeration of logon sessions. The details associated with
each can be obtained through the
[uri \#get_logon_session_info [cmd get_logon_session_info]] command.


[section "Windows Credentials"]

The Windows credentials manager maintains a cache of credentials for
a user account or logon session. The [cmd twapi_security] package
provides some additional commands to those already
present in the [cmd twapi_base] package.
[para]
The [uri #credentials [cmd credentials]] returns a list containing the content
of the credentials cache.
The [uri #cred_get [cmd cred_get]] command returns cached
credentials for a credential target.
The [uri #cred_delete [cmd cred_delete]] command deletes cached
credentials. The commands
[uri #cred_marshal [cmd cred_marshal]],
[uri #cred_unmarshal [cmd cred_unmarshal]] and
[uri #cred_is_marshaled [cmd cred_is_marshaled]] can be used for marshaling
of credentials. The command [uri #cred_parse_username [cmd cred_parse_username]]
can be used to extract the user and domain names from a fully qualified user name.

[section "User Account Control"]

Windows Vista and later versions
include the
[uri http://windows.microsoft.com/en-us/windows-vista/What-is-User-Account-Control "User Account Control"]
(UAC) feature that runs even adminstrator account processes with reduced
privileges unless explicitly elevated by the user.
The command 
[uri #get_token_elevation [cmd get_token_elevation]] checks whether a
particular process token is elevated or not. The command
[uri #get_token_linked_token [cmd get_token_linked_token]] returns
the token handle for the token that is linked to an elevated or limited token.

[para]
Similarly, the
[uri http://msdn.microsoft.com/en-us/library/bb625957.aspx "Windows Integrity"]
mechanism attempts to distinguish between applications at different levels 
of trust and control access to resources accordingly. The commands
[uri #get_token_integrity [cmd get_token_integrity]] and
[uri #set_token_integrity [cmd set_token_integrity]] allow retrieving and
setting the integrity level associated with a process token. Alternatively,
the higher level commands
[uri process.html#get_process_integrity get_process_integrity] and
[uri process.html#set_process_integrity set_process_integrity] can also
be used for this purpose.

[para]
The associated
policy for maintaining integrity levels can be read and modified using
[uri #get_token_integrity_policy [cmd get_token_integrity_policy]] and
[uri #set_token_integrity_policy [cmd set_token_integrity_policy]].

[para]
The integrity level associated with a resource can be retrieved and
set using the commands
[uri #get_resource_integrity [cmd get_resource_integrity]] and
[uri #set_resource_integrity [cmd set_resource_integrity]] respectively.
The integrity level stored in a security descriptor can be retrieved and
set using
[uri #get_security_descriptor_integrity [cmd get_security_descriptor_integrity]]
and
[uri #set_security_descriptor_integrity [cmd set_security_descriptor_integrity]].

[para]
The above security mechanims also introduced compatibility issues for
applications written for older versions of Windows which expected to be
able to write to resources at higher integrity levels, such as the system
directory, that were now restricted. To allow these applications to run,
Windows silently virtualizes protected resources and redirects
reads and writes to these resources from such applications. The
[uri #get_token_virtualization [cmd get_token_virtualization]]
can be used to detect if a process is virtualized in this fashion and
[uri #set_token_virtualization [cmd set_token_virtualization]] can
be used to enable or disable the state.

[para]
For a tutorial and discussion of these issues, refer to the
[uri http://msdn.microsoft.com/en-us/library/aa905330.aspx "MSDN article"].

[section Commands]
[list_begin definitions]

[call [cmd append_acl_aces] [arg ACL] [arg ACELIST]]
Appends the list of ACEs in [arg ACELIST] to the entries in the given ACL.
Returns the modified ACL.

[call [cmd check_enabled_privileges] [arg TOKEN] [arg PRIVILEGES] [opt [cmd -any]]]
Without the [cmd -any] option,
the command returns 1 if the given list of privileges is enabled in the
token, and 0 otherwise. If the option [cmd -any] is specified, the command
returns 1 if at least one of the privileges is enabled, and 0 otherwise.
[arg PRIVLIST] is a list of privilege names or LUIDs.

[call [cmd close_token] [arg TOKEN]]
Frees resources associated with an access token.

[call [cmd cred_delete] [arg TARGET] [opt [arg TYPE]]]

Deletes the credentials identified by [arg TARGET] from the credential cache
associated with the logon session of the curent security token.
The [arg TYPE] argument is one of [const generic], [const domain_password],
[const domain_certificate], [const domain_visible_password],
[const generic_certificate] or [const domain_extended].

[call [cmd cred_get] [arg TARGET] [opt [arg TYPE]]]
Returns the credentials identified by [arg TARGET] from the credential set
associated with the logon session of the curent security token.
The [arg TYPE] argument is one of [const generic], [const domain_password],
[const domain_certificate], [const domain_visible_password],
[const generic_certificate] or [const domain_extended].

The return value is a dictionary with the following keys:

[list_begin opt]
[opt_def [const attributes]] A list of credential attributes. Each attribute is
a triplet consisting of a application-specific keyword, an integer flags
(currently always 0) and a binary string whose format is application-dependent.
[opt_def [const comment]] Any associated descriptive comment for the credential
that might be added by the user.
[opt_def [const lastwritten]] The last time the credentials were written
as the number of 100ns since Jan 1, 1601
(see [uri base.html#large_system_time_to_secs_since_1970 [cmd large_system_time_to_secs_since_1970]]).
[opt_def persist] One of the values [const session], [const local_machine]
or [const enterprise] depending on whether the credentials are persisted
for the current logon session only, or are persisted on the local system
and are visible to future logon session for the user on that system,
or are persisted on the local system and are visible to logon session
for that user from other systems.
[opt_def [const target]] The name of the target for which the credential
is applicable.
[opt_def [const targetalias]] An alias for the target.
[opt_def [const type]] The credential type which is one of
[const generic], [const domain_password], [const domain_certificate], 
[const domain_visible_password]
[const generic_certificate], [const domain_extended.]
[opt_def [const username]] The user name associated with the target
credential.
[list_end]
The dictionary may contain other elements. These should be ignored
and are liable to change.
[nl]
For more information, see the description of the
[uri https://docs.microsoft.com/en-us/windows/win32/api/wincred/ns-wincred-credentialw "CREDENTIALS"] structure in the Windows SDK.

[call [cmd cred_is_marshaled] [arg STR]]
Returns a boolean true value if the string [arg STR] represents a credential
in marshaled form and false otherwise.

[call [cmd cred_marshal] [arg TYPE] [arg CRED]]
Transforms the passed credential [arg CRED] into text string. [arg TYPE] should
be either [const certificate] or [const user]. In former case, [arg CRED] should
be the binary hash of the certificate as returned in the [const sha1_hash] property
of the [uri certs.html#cert_property [cmd cert_property]] command. In the latter
case, it should be a string specifying the user name.

[call [cmd cred_parse_username] [arg FQUN]]
Parses a fully qualified user name [arg FQUN] and returns it as a user name and
domain pair.

[call [cmd cred_unmarshal] [arg MCRED]]
Reverts a marshalled credential back to its original form. The return value
is a pair consisting of the type and value. See [uri #cred_marshal [cmd cred_marshal]].

[call [cmd credentials] [opt [arg TARGETPATTERN]]]
Returns a list of current credentials from the credential cache.
If [arg TARGETPATTERN] is not specified, all credentials are returned.
Otherwise, only those with targets matching [arg TARGETPATTERN] are returned.
[arg TARGETPATERN] is a name prefix optionally terminating in a wildcard
[const *] character.
[nl]
The returned list elements are dictionaries containing credentials with
keys as described for the
[uri base.html#cred_get [cmd cred_get]] command.

[list_begin opt]
[opt_def [const comment]] Any associated descriptive comment for the credential
that might be added by the user.
[opt_def [const lastwritten]] The last time the credentials were written
as the number of 100ns since Jan 1, 1601
(see [uri base.html#large_system_time_to_secs_since_1970 [cmd large_system_time_to_secs_since_1970]]).
[opt_def persist] One of the values [const session], [const local_machine]
or [const enterprise] depending on whether the credentials are persisted
for the current logon session only, or are persisted on the local system
and are visible to future logon session for the user on that system,
or are persisted on the local system and are visible to logon session
for that user from other systems.
[opt_def [const target]] The name of the target for which the credential
is applicable.
[opt_def [const targetalias]] An alias for the target.
[opt_def [const type]] The credential type which is one of
[const generic], [const domain_password], [const domain_certificate], 
[const domain_visible_password]
[const generic_certificate], [const domain_extended.]
[opt_def [const username]] The user name associated with the target
credential.
[list_end]
The dictionary may contain other elements. These should be ignored
and are liable to change.
[nl]
For more information, see the description of the
[uri http://msdn.microsoft.com/en-us/library/windows/desktop/aa374788(v=vs.85).aspx "CREDENTIALS"] structure in the Windows SDK.

[call [cmd decode_security_descriptor] [arg SECDBIN]]
Decodes a self-relative security descriptor in binary format and returns
it in the corresponding canonical Tcl format.

[call [cmd disable_all_token_privileges] [arg TOKEN]]
This command disables all privileges in the token.
An error will be raised if the privilege does not exist or cannot be disabled
because of lack of rights to the token handle.
[nl]
The command returns a list of privileges that were modified by the command
because they were not previously disabled. This list can be passed to
[uri \#enable_token_privileges [cmd enable_token_privileges]]
to undo the privilege modifications made by this call.

[call [cmd disable_privileges] [arg PRIVLIST]]

This command disables one or more privileges in the current process.
[arg PRIVLIST]
is a list of privileges each of which may be either a symbolic privilege
such as [const SeLoadDriverPrivilege] or a LUID corresponding to a privilege.
[nl]
An error will be raised if the privilege does not exist or cannot be disabled
because of lack of rights to the token handle.
[nl]
The command returns a list of privileges that were modified by the command
because they were not previously disabled. This list can be passed to
[uri \#enable_privileges [cmd enable_privileges]]
to undo the privilege modifications made by this call.

[call [cmd disable_token_privileges] [arg TOKEN] [arg PRIVLIST]]

This command disables one or more privileges in the token. [arg PRIVLIST]
is a list of privileges each of which may be either a symbolic privilege
such as [const SeLoadDriverPrivilege] or a LUID corresponding to a privilege.
[nl]
An error will be raised if the privilege does not exist or cannot be disabled
because of lack of rights to the token handle.
[nl]
The command returns a list of privileges that were modified by the command
because they were not previously disabled. This list can be passed to
[uri \#enable_token_privileges [cmd enable_token_privileges]]
to restore the token privileges to their original value.

[call [cmd enable_privileges] [arg PRIVLIST]]

This command enables one or more privileges in the current process.
[arg PRIVLIST]
is a list of privileges each of which may be either a symbolic privilege
such as [const SeLoadDriverPrivilege] or a LUID corresponding to a privilege.
[nl]
An error will be raised if a privilege does not exist or cannot be enabled
because of lack of rights to the token handle.
[nl]
The command returns a list of privileges that were modified by the command
because they were not previously enabled. This list can be passed to
[uri \#disable_privileges [cmd disable_privileges]]
to undo the privilege modifications made by this call.

[call [cmd enable_token_privileges] [arg TOKEN] [arg PRIVLIST]]

This command enables one or more privileges in the token. [arg PRIVLIST]
is a list of privileges each of which may be either a symbolic privilege
such as [const SeLoadDriverPrivilege] or a LUID corresponding to a privilege.
[nl]
An error will be raised if a privilege does not exist or cannot be enabled
because of lack of rights to the token handle.
[nl]
The command returns a list of privileges that were modified by the command
because they were not previously enabled. This list can be passed to
[uri \#disable_token_privileges [cmd disable_token_privileges]]
to undo the privilege modifications made by this call.

[call [cmd encode_security_descriptor] [arg SECD]]
Encodes a security descriptor into a self-relative binary format and
returns it.

[call [cmd equal_sids] [arg SIDA] [arg SIDB]]
If both arguments are valid SIDs, returns [const 1] if they are equal
and [const 0] otherwise. Generates an error if either is not a valid SID.

[call [cmd eval_with_privileges] [arg SCRIPT] [arg PRIVLIST]]
The command enables the privileges specified in [arg PRIVLIST], if they were
not already enabled, and then evaluates [arg SCRIPT] in the caller's context.
The command returns the result of [arg SCRIPT] after restoring
the privileges in effect when the command was called.

[call [cmd find_logon_sessions] [opt [arg options]]]
Returns a list of [uri \#uniqueidentifiers LUID] values that identify the
logon sessions on the system. By default, all logon sessions are returned.
One or more of the following options may be specified to restrict
the set of logon sessions that are returned.

[list_begin opt]
[opt_def [cmd -user] [arg ACCOUNT]]
Only returns logon sessions for the specified user. [arg ACCOUNT] may
be either the account name or the SID for the user.
[opt_def [cmd -type] [arg SESSIONTYPELIST]]
Only returns logon sessions that match one of the specified types.
[arg SESSIONTYPELIST] is a list of
integer logon session types defined in the
Windows SDK or the following equivalent tokens:
[const interactive], [const network],
[const batch], [const service], [const proxy], [const unlockworkstation],
[const networkclear], [const newcredentials], [const remoteinteractive],
[const cachedinteractive], [const cachedremoteinteractive] or
[const cachedunlockworkstation].
[opt_def [cmd -tssession] [arg TSSESSION]]
Only returns those logon sessions belonging to the terminal services
session identified by [arg TSSESSION].
[list_end]

[call [cmd get_ace_inheritance] [arg ACE]]
Returns a flat list of "[arg option] [arg value]" pairs that indicate
the inheritance characteristics of the ACE. The [arg option] elements are
[cmd -self], [cmd -recursecontainers], [cmd -recurseobjects] and
[cmd -recurseonelevelonly] as described in the documentation for
[cmd new_ace]. In addition, the read-only option [cmd -inherited] indicates
whether the ACE has been inherited from an ancestor or not.

[call [cmd get_ace_rights] [arg ACE] [opt "[cmd -resourcetype] [arg RESOURCETYPE]"] [opt [cmd -raw]]]
Returns a list of access rights contained within an ACE. This is a list
of symbolic constants and bit masks as described in [sectref "Access Rights"].
The option [cmd -resourcetype] may be used to map the access rights to symbolic
constants for a specific type of resource. [arg RESOURCETYPE] must be
one of [const file], [const pipe], [const policy], [const process], 
[const registry], [const service], [const thread], [const token] or
[const winsta].
[nl]
If the [cmd -raw] option is specified, the access rights are returned as a
single hex bitmask.

[call [cmd get_ace_sid] [arg ACE]]
Returns the SID  of the given ACE.

[call [cmd get_ace_text] [arg ACE] [opt "[cmd -resourcetype] [arg RESOURCETYPE]"]]
Returns a textual description of the specified ACE. The option
[cmd -resourcetype] may be specified to indicate that the ACE is for
a particular resource type. [arg RESOURCETYPE] must be one of the
resource type values defined for the
[uri \#get_ace_rights [cmd get_ace_rights]] command. This command is
primarily useful for debugging and troubleshooting purposes.

[call [cmd get_ace_type] [arg ACE]]
Returns the type of the given ACE. See [sectref "Access Control Entries"] for
the possible types.

[call [cmd get_acl_aces] [arg ACL]]
Returns the ordered list of ACE elements in the given ACL.

[call [cmd get_acl_rev] [arg ACL]]
Returns the revision level of an ACL. Note that this is controlled by the
ACEs in the ACL and cannot be set directly.

[call [cmd get_current_user] [opt [arg FORMATOPTION]]]

Returns the user account for the current thread. If the thread is
impersonating, this command returns the account being impersonated. 
Note this is [emph not] the
same as the [cmd {$::env(USERNAME)}] variable as the latter does not take
impersonation into account.
By default,
the command returns the name of the user account in NT 4.0 SAM-compatible
format. One of the following options may be specified to return
the name in a different format:
[list_begin opt]
[opt_def [cmd -fullyqualifieddn]]
Fully qualified Active Directory distinguished name.
[opt_def [cmd -samcompatible]]
NT 4.0 SAM account name (default).
[opt_def [cmd -display]]
Active Directory 
human-friendly display name.
[opt_def [cmd -uniqueid]]
Active Directory GUID format name.
This is NOT the same as the SID.
[opt_def [cmd -canonical]]
Active Directory canonical format.
[opt_def [cmd -userprincipal]]
User principal format.
[opt_def [cmd -canonicalex]]
Active Directory canonical format except that the
rightmost [const /] is replaced by a newline.
[opt_def [cmd -serviceprincipal]]
Generalized service principal name.
[opt_def [cmd -dnsdomain]]
DNS domain followed by the SAM user name.
[opt_def [cmd -sid]]
User SID instead of name.
[list_end]
[nl]
Note that all options except [cmd -samcompatible] and [cmd -sid]
require that the system be part of a Active Directory domain, else
an error is raised that no mapping to that format exists.

[call [cmd get_logon_session_info] [arg SESSIONLUID] [opt [arg options]]]
Returns details for the logon session identified by [arg SESSIONLUID].
The information is returned as a flat list of option value pairs and
depends on the specified options:

[list_begin opt]
[opt_def [cmd -all]]
Returns the values for all options.
[opt_def [cmd -authpackage]]
Returns the name of the package that authenticated the session.
[opt_def [cmd -dnsdomain]]
Returns the DNS name for the owner of the logon session.
[opt_def [cmd -logondomain]]
Returns the name of the domain used to authenticate the session.
[opt_def [cmd -logonid]]
Returns the [uri \#uniqueidentifiers LUID] for the session
(same as [arg SESSIONLUID]).
[opt_def [cmd -logonserver]]
Returns the name of the server used to authenticate the session.
[opt_def [cmd -logontime]]
Returns the time session owner logged on.
[opt_def [cmd -type]]
Returns the logon session type
(see [uri \#find_logon_sessions [cmd find_logon_sessions]]).
[opt_def [cmd -usersid]]
Returns the SID for the session owner.
[opt_def [cmd -tssession]]
Returns the terminal session identifier for the logon session.
[opt_def [cmd -user]]
Returns the user name for the session owner.
[opt_def [cmd -userprincipal]]
Returns the user principal name for the session owner.
[list_end]

[call [cmd get_privilege_description] [arg PRIVILEGE]]
Returns a string description of the specified privilege.

[call [cmd get_resource_integrity] [arg RESTYPE] [arg NAME] [opt [arg options]]]
Retrieves the integrity level associated with a resource. 
[arg RESTYPE] specifies the resource type and if option [cmd -handle] is
not specified, must be one of [const file],
[const service] (Windows service), [const printer], [const registry],
[const share] (Network shares) or [const kernelobj]
(kernel objects such as mutexes, semaphores and events).
If option [cmd -handle] is specified, then [arg RESTYPE] may also
be one of [const windowstation], [const directoryservice],
[const directoryserviceall], [const providerdefined], [const wmiguid]
or [const registrywow6432key] but not [const share].
[nl]
[arg NAME] specifies the name of the resource, for example, the path of
a file unless option [cmd -handle] is specified in which case it is the
handle to the resource.

[nl] On systems that
do not support integrity levels, the command returns an empty list.
Otherwise, a list with two elements is returned. The first element of
the list is the integrity value associated with the resource, by
default represented as an integer value 
between 0 and 65535. If the option [cmd -raw] is specified, the integrity
level is returned as the raw SID value as stored internally by Windows in the
token. Alternatively, if the [cmd -label] option is specified the corresponding
integrity level range label is returned which is one of
[const untrusted] (0-4095), [const low] (4096-8191),
[const medium] (8192-12287, [const high] (12288-16383) or
[const system] (anything higher).
[nl]
The second element of the returned list is a list containing zero or
or more of the following values: 
[const system_mandatory_label_no_write_up],
[const system_mandatory_label_no_read_up],
[const system_mandatory_label_no_execute_up]. These control the operations
allowed on the resource by processes with lower integrity levels. Refer
to the Windows SDK for more details.

[call [cmd get_resource_security_descriptor] [arg RESTYPE] [arg NAME] [opt [arg options]]]

Retrieves the security descriptor attached to a operating system resource.
[arg RESTYPE] specifies the resource type and if option [cmd -handle] is
not specified, must be one of [const file],
[const service] (Windows service), [const printer], [const registry],
[const share] (Network shares) or [const kernelobj]
(kernel objects such as mutexes, semaphores and events).
If option [cmd -handle] is specified, then [arg RESTYPE] may also
be one of [const windowstation], [const directoryservice],
[const directoryserviceall], [const providerdefined], [const wmiguid]
or [const registrywow6432key] but not [const share].
[nl]
[arg NAME] specifies the name of the resource, for example, the path of
a file unless option [cmd -handle] is specified in which case it is the
handle to the resource.
[nl]
Without any options being specified, the returned security descriptor
will contain only the owner, group and DACL fields. The SACL field will
not contain valid data. The following options may be specified to control
which fields are returned in the security descriptor:

[list_begin opt]
[opt_def [cmd -all]] Includes all fields in the security descriptor.
This requires the caller to have the
[const SeSecurityPrivilege] privilege enabled.
[opt_def [cmd -dacl]] Includes the DACL in the returned
security descriptor.
[opt_def [cmd -group]] Includes the group information in the returned
security descriptor.
[opt_def [cmd -owner]] Includes the owner information in the returned
security descriptor.
[opt_def [cmd -sacl]] Includes the SACL in the returned
security descriptor. This requires the caller to have the
[const SeSecurityPrivilege] privilege enabled.
[list_end]

[call [cmd get_security_descriptor_control] [arg SECD]]
Returns the control flags contained in the given security
descriptor [arg SECD].
The flags are returned as a list of zero or more tokens from the
following:
[const owner_defaulted],
[const group_defaulted],
[const dacl_present],
[const dacl_defaulted],
[const sacl_present],
[const sacl_defaulted],
[const dacl_auto_inherited],
[const sacl_auto_inherited],
[const dacl_protected],
[const sacl_protected],
[const rm_control_valid],
[const self_relative].
Refer to the Windows Platform SDK for details.

[call [cmd get_security_descriptor_dacl] [arg SECD]]
Returns the DACL contained in the given security descriptor [arg SECD].

[call [cmd get_security_descriptor_group] [arg SECD]]
Returns the group SID for the given security descriptor [arg SECD].

[call [cmd get_security_descriptor_integrity] [arg SECD] [opt [cmd "-label|-raw"]]]
Returns the integrity level stored in the given security descriptor [arg SECD].
Refer to
[uri #get_resource_integrity [cmd get_resource_integrity]] for the 
option description and format
of the returned value.

[call [cmd get_security_descriptor_owner] [arg SECD]]
Returns the owner SID for the given security descriptor [arg SECD].

[call [cmd get_security_descriptor_sacl] [arg SECD]]
Returns the SACL contained in the given security descriptor [arg SECD].

[call [cmd get_security_descriptor_text] [arg SECD] [opt "[cmd -resourcetype] [arg RESOURCETYPE]"]]
Returns a textual description of the specified security descriptor [arg SECD].
The option
[cmd -resourcetype] may be specified to indicate that the security
descriptor is for
a particular resource type. [arg RESOURCETYPE] must be one of the
resource type values defined for the
[uri \#get_ace_rights [cmd get_ace_rights]] command. This command is
primarily useful for debugging and troubleshooting purposes.

[call [cmd get_sid_domain] [arg SID]]
Returns the SID of the domain containing the specified SID.

[call [cmd get_token_default_dacl] [arg TOKEN]]
Returns the DACL stored in the token which will be used as the defaulted
DACL when a new resource is created with no security descriptor specified.

[call [cmd get_token_elevation] [arg TOKEN]]
On systems that support Windows UAC, this command returns one of
[const default], [const full] or [const limited] depending on whether the
token corresponds to a process is running with default privileges 
for the account, full (elevated)
privileges for the account or limited (filtered) privileges.
On systems that do not support UAC, the command always returns [const default].
[nl]
To directly access the elevation level of a process, you can also
use the
[uri process.html#get_process_elevation [cmd get_process_elevation]] command
or the [cmd -elevation] option of the
[uri process.html#get_process_info [cmd get_process_info]] command.

[call [cmd get_token_groups] [arg TOKEN] [opt [cmd -name]]]
Returns the list of group SID's for the token. If [cmd -name]
is specified, returns the group names instead of the SID's.

[call [cmd get_token_groups_and_attrs] [arg TOKEN]]
Returns the list of group SID's for the token together with the attributes
for each group. The returned list is of the form
[arg "SID ATTRIBUTELIST SID ATTRIBUTELIST..."] where [arg ATTRIBUTELIST]
is a list of zero or more of the following attributes:
[const enabled], [const enabled_by_default], [const logon_id],
[const mandatory], [const owner], [const resource], [const integrity],
[const integrity_enabled] and
[const use_for_deny_only]. Refer to the Windows Platform SDK for details.

[call [cmd get_token_impersonation_level] [arg TOKEN]]
Gets the current impersonation level for the token. This may have
one of the following values: [const anonymous], [const identification],
[const impersonation], or [const delegation]. See [sectref Impersonation]
for details.

[call [cmd get_token_info] [arg TOKEN] [opt [arg options]]]
Returns a dictionary containing the token fields specified in
[arg options]. The following fields may be requested:
[list_begin opt]
[opt_def [cmd -disabledprivileges]] Returns the list of privileges
in the token that are currently disabled.
[opt_def [cmd -elevation]] Returns the elevation level for the token
(see [uri process.html#get_process_elevation [cmd get_process_elevation]]).
[opt_def [cmd -enabledprivileges]] Returns the list of privileges
that are enabled in the token.
[opt_def [cmd -groupattrs]]
Returns a dictionary whose keys are group SID's contained in the token,
with the corresponding value being a list of attributes for the group.
[opt_def [cmd -groups]] Returns the list of the names of
groups included in the token.
For domain accounts, an error is raised if the
domain is unreachable or if an SID has no corresponding name.
Use [cmd -groupsids] to avoid this.
[opt_def [cmd -groupsids]] Returns the list of SID's of the
groups included in the token.
[opt_def [cmd -integrity]] Returns the numeric integrity level associated
with the token
(see [uri process.html#get_process_integrity [cmd get_process_integrity]]).
[opt_def [cmd -integritylabel]] Returns the label for
the range for the integrity level associated with the token
(see [uri process.html#get_process_integrity [cmd get_process_integrity]]).
[opt_def [cmd -linkedtoken]]
If [arg TOKEN] is an elevated or limited token, this command returns the
corresponding limited or elevated token linked to it.
[opt_def [cmd -logonsession]]
Returns the [uri #uniqueidentifiers LUID] of the
[uri #logonsessions "logon session"] associated with the token.
[opt_def [cmd -logonsessionsid]]
Returns the SID of the
[uri #logonsessions "logon session"] associated with the token.
[opt_def [cmd -primarygroup]] Returns the primary group for the token.
For domain accounts, an error is raised if the
domain is unreachable. Use [cmd -primarygroupsid] to avoid this.
[opt_def [cmd -primarygroupsid]] Returns the SID of primary group 
for the token.
[opt_def [cmd -privileges]] Returns the list of enabled and disabled
process privileges in the same format as
[uri security.html\#get_token_privileges [cmd get_token_privileges]].
Deprecated, use [cmd -enabledprivileges] and [cmd -disabledprivileges]
instead.
[opt_def [cmd -restrictedgroupattrs]]
Returns a dictionary whose keys are SID's for the restricted 
groups contained in the token,
with the corresponding value being a list of attributes for the group.
See [uri #get_token_groups_and_attrs [cmd get_token_groups_and_attrs]]
for details.
[opt_def [cmd -restrictedgroups]] Returns the list of restricted groups in
the token.
For domain accounts, an error is raised if the
domain is unreachable. Use [cmd -restrictedgroupattrs] to avoid this.
[opt_def [cmd -tssession]]
Returns the terminal server session id of the process.
[opt_def [cmd -usersid]]
Returns the SID of the user account associated with the token.
[opt_def [cmd -virtualized]]
Returns true if the process associated with the token is virtualized
and false otherwise. See [uri process.html#virtualized_process [cmd virtualized_process]].

[list_end]


[call [cmd get_token_integrity] [arg TOKEN] [opt "[cmd -raw] | [cmd -label]"]]
On systems that support Windows UAC, this command returns the integrity
level associated with the token. The integrity value is an integer value 
between 0 and 65535. If the option [cmd -raw] is specified, the integrity
level is returned as the raw SID value as stored internally by Windows in
the token. Alternatively, if the [cmd -label] is specified the corresponding
integrity level range label is returned which is one of
[const untrusted] (0-4095), [const low] (4096-8191),
[const medium] (8192-8447), [const mediumplus] (8448-12287),
[const high] (12288-16383) or
[const system] (anything higher).
[nl]
On systems that do not support integrity levels, this command always returns
an appropriately formatted value corresponding to the medium integrity level.
[nl]
To directly access the integrity level of a process, you can also
use the
[uri process.html#get_process_integrity [cmd get_process_integrity]] command
or the [cmd -integrity] option of the
[uri process.html#get_process_info [cmd get_process_info]] command.

[call [cmd get_token_integrity_policy] [arg TOKEN]]
On systems that support Windows UAC, this command returns the integrity policy
associated with the token. The integrity policy is a list containing zero or
more values from table below.
[list_begin opt]
[opt_def [const no_write_up]] If present, writes to objects at a
higher integrity level are denied.
[opt_def [const new_process_min]] If present, new processes will be created
at an integrity level that is the minimum of the integrity level of the
token and the integrity level associated with the executable file
of the new process.
[list_end]
[nl]
On systems that do not support integrity levels, this command always returns
an empty list indicating no integrity policy is in effect.

[call [cmd get_token_linked_token] [arg TOKEN]]
If [arg TOKEN] is an elevated or limited token, this command returns the
corresponding limited or elevated token linked to it.

[call [cmd get_token_origin] [arg TOKEN]]
Returns the LUID of the logon session that created the logon session
containing token [arg TOKEN] except if the token's logon resulted from
network authentication in which case the command returns
a zero LUID.

[call [cmd get_token_owner] [arg TOKEN] [opt [cmd -name]]]
Returns the SID of the entity (user, group, etc.)
that will be used as the owner for objects
created using this token. If [cmd -name] is specified, the name
of the entity is returned instead of the SID.

[call [cmd get_token_primary_group] [arg TOKEN] [opt [cmd -name]]]
Returns the SID of the group that will be used as the
primary group for objects
created using this token. If [cmd -name] is specified, the name
of the group is returned instead of the SID.

[call [cmd get_token_privileges] [arg TOKEN] [opt [cmd -all]]]
Returns the list of enabled privileges in the token. If the option
[cmd -all] is specified, returns a list of two sublists. The first
containing the enabled privileges and the second the disabled privileges.

[call [cmd get_token_privileges_and_attrs] [arg TOKEN]]
Returns the list of privileges in the token together with the attributes
for each. The returned list is of the form
[arg "PRIVILEGE ATTRIBUTELIST PRIVILEGE ATTRIBUTELIST..."]
where [arg ATTRIBUTELIST]
is a list of zero or more of the following attributes: [const enabled],
[const enabled_by_default] and [const used_for_access].

[call [cmd get_token_restricted_groups_and_attrs] [arg TOKEN]]
Returns the list of restricted group SID's for the token together with the 
attributes for each group. The returned list is of the form
[arg "SID ATTRIBUTELIST SID ATTRIBUTELIST..."] where [arg ATTRIBUTELIST]
is a list of zero or more of the following attributes:
[const enabled], [const enabled_by_default], [const logon_id],
[const mandatory], [const owner], [const resource], [const integrity],
[const integrity_enabled] and
[const use_for_deny_only]. Refer to the Windows Platform SDK for details.

[call [cmd get_token_source] [arg TOKEN]]
Returns the source of an access token.

[call [cmd get_token_statistics] [arg TOKEN]]
Returns various token statistics as a list of field value pairs. The following
fields are returned:
[list_begin opt]
[opt_def authluid]
The [uri \#uniqueidentifiers "LUID"] for the session represented by
the token.
[opt_def expiration]
Expiration time for the token. Windows does not currently support
expiration for access tokens.
[opt_def dynamicavailable]
Number of free bytes available for storing default protection and primary
group identifier for the token.
[opt_def dynamiccharged]
Number of bytes allocated for storing default protection and primary
group identifier for the token.
[opt_def groupcount]
Number of group SID's included in the token.
[opt_def impersonationlevel]
The [uri \#impersonation "impersonation level"] of the token. This value
is only valid if the field [cmd type] has the value [const impersonation].
[opt_def luid]
An [uri \#uniqueidentifiers "LUID"] that identifies the instance of the token.
[opt_def modificationluid]
An [uri \#uniqueidentifiers "LUID"] that changes every time the token
is modified.
[opt_def privilegecount]
The number of privileges (including disabled ones) that are included in
the token.
[opt_def type]
The type of the token - [const primary] or [const impersonation].
[list_end]

[call [cmd get_token_tssession] [arg TOKEN]]
Returns the terminal server session id of the process.

[call [cmd get_token_type] [arg TOKEN]]
Returns the type of the token: either [const primary], or
[const impersonation].

[call [cmd get_token_user] [arg TOKEN] [opt [cmd -name]]]
Returns the SID of the user account associated with the token. If [cmd -name]
is specified, returns the name of the account instead of its SID.

[call [cmd get_token_virtualization] [arg TOKEN]]
Returns true if the token indicates a virtualized process where the system
redirects writes to protected file and registry locations and false otherwise.
On systems that do not support UAC, the command will always return false.

[nl]
To directly retrieve the virtualization status of a process, you can also
use the
[uri process.html#virtualized_process [cmd virtualized_process]] command
or the [cmd -virtualized] option of the
[uri process.html#get_process_info [cmd get_process_info]] command.


[call [cmd impersonate_self] [arg IMPERSONATIONLEVEL]]
Assigns an impersonation token to the current thread based on the security
context of the process. The security characteristics of the thread can
then be changed (for example by modifying the privileges of its token)
without impacting the security context of the remaining threads in the process.
[nl]
[arg IMPERSONATIONLEVEL] may have any of the values allowed for impersonation
levels (see [sectref Impersonation]).

[call [cmd impersonate_token] [arg TOKEN]]
Changes the security context of the calling thread to that of the user
account associated with the specified token. [arg TOKEN] is a token
returned from a command such as [cmd open_user_token], [cmd open_process_token]
or [cmd open_thread_token].
[nl]
[emph "The caller must be careful that exceptions raised by this command are handled correctly. It should not continue with normal processing as though the security context was changed to that of the token."]


[call [cmd impersonate_user] [arg USERNAME] [arg PASSWORD] [opt [arg options]]]
Changes the security context of the calling thread to that of the specified
user account. This is essentially a wrapper around [cmd open_user_token]
and [cmd impersonate_token]. Refer to the documentation of
[cmd open_user_token] for a description of the arguments.
[nl]
[emph "The caller must be careful that exceptions raised by this command are handled correctly. It should not continue with normal processing as though the security context was changed to that of the user."]


[call [cmd is_valid_luid_syntax] [arg LUID]]

Returns if [arg LUID] is a syntactically valid LUID, else 0.

[call [cmd is_well_known_sid] [arg SID] [arg KNOWNSIDNAME]]
Returns [const 1] if [arg SID] is a well-known SID corresponding to
[arg KNOWNSIDNAME]. See [uri #well_known_sid [cmd well_known_sid]]
for values of [arg KNOWNSIDNAME].

[call [cmd map_luid_to_privilege] [arg LUID] [opt "[cmd -system] [arg SYSTEMNAME]"] [opt [cmd -mapunknown]]]

Returns the privilege name corresponding to the given LUID.
If [arg LUID] does not correspond to
a system defined privilege, an exception is raised unless the
[cmd -mapunknown] option is specified. In this case, the command will
return a string of the form [const Privilege-][arg LUID]. As a special
case, if [arg LUID] is a privilege name (and not a LUID), returns it as is.
The option [cmd -system]
may be used to specify a system as described in [sectref "Standard Options"].

[call [cmd map_privilege_to_luid] [arg PRIVILEGE] [opt "[cmd -system] [arg SYSTEMNAME]"] [opt [cmd -mapunknown]]]

Returns the LUID corresponding to the given privilege name.
If [arg PRIVILEGE] does not correspond to
a system defined privilege name or a constructed privilege name of the
form [const Privilege-][arg LUID], an exception is raised.
As a special case,
if [arg PRIVILEGE] is itself a LUID, it is returned.
The option [cmd -system]
may be used to specify a system as described in [sectref "Standard Options"].

[call [cmd new_ace] [arg ACETYPE] [arg ACCOUNT] [arg ACCESSRIGHTS] [opt [arg options]]]

Creates and returns a new ACE. [arg ACETYPE] is the type of the ACE
and should be one of
supported values described in [sectref "Access Control Entries"].
The user or group that the ACE pertains to is given by [arg ACCOUNT]
which may be either a name or a SID.
[arg ACCESSRIGHTS] determines the rights to be granted or denied by the ACE
and should be a list of one or more constants described in
[sectref "Access Rights"].

[nl]
The following options may be specified when creating an ACE:
[list_begin opt]
[opt_def [cmd -auditsuccess] [arg BOOLEAN]]
Specified that the ACE should log permitted accesses to a resource.
Ignored if the ACE type is not [const audit]. Defaults to [const true].
[opt_def [cmd -auditfailure] [arg BOOLEAN]]
Specified that the ACE should log failed accesses to a resource.
Ignored if the ACE type is not [const audit]. Defaults to [const true].
[opt_def [cmd -recursecontainers] [arg BOOLEAN]] If the specified value is 1,
the ACE is also applied to all descendents of the object to which the ACE is
attached that are themselves containers. For example, in the case of a file
ACE, this would indicate that the ACE also apply to all subdirectories.
If 0 (default), the ACE does not apply to descendents that are containers.
[opt_def [cmd -recurseobjects] [arg BOOLEAN]] If the specified value is 1,
the ACE is also applied to all descendents of the object to which the ACE is
attached that are [emph not] themselves containers.
For example, in the case of a file
ACE, this would indicate that the ACE also apply to all files under
all subdirectories below a directory but not to the subdirectories themselves.
If 0 (default), the ACE does not apply to descendents that are not containers.
[opt_def [cmd -recurseonelevelonly] [arg BOOLEAN]] If the specified value
is 0 (default), the [cmd -recursecontainers] and [cmd -recurseobjects] options
have effect at all levels below the container object to which the ACE is
applied. If the value is 1, the ACE is only applied to the immediate
children of the container object. This option has no effect if neither
[cmd -recurseobjects] nor [cmd -recursecontainers] is specified.
[opt_def [cmd -self] [arg BOOLEAN]] If the specified value is 1 (default),
the ACE applies to the object to which it is attached.
If the specified value is 0, then
the created ACE does not apply to the object with which it is attached.
In this case, it only applies to the descendents of that object as
indicated by the other options.

[list_end]

[call [cmd new_acl] [opt [arg ACELIST]]]
Returns a new ACL initialized. [arg ACELIST] may be optionally specified
to prepopulate the ACL with a list of ACE elements.

[call [cmd new_luid]]
Returns a [uri \#uniqueidentifiers LUID].

[call [cmd new_restricted_dacl] [arg ACCOUNTS] [arg RIGHTS] [opt [arg options]]]
Returns a new DACL where the access rights specified by [arg RIGHTS]
are granted to the accounts included in [arg ACCOUNTS]. Other accounts
are implicitly denied. Accounts
may be specified using either the account names or SID's. [arg RIGHTS]
is a list of symbolic constants and bit masks as described
in [sectref "Access Rights"].
Any additional options are passed to [uri #new_ace [cmd new_ace]]
for creating the ACE's in the DACL.

[call [cmd new_security_descriptor] [opt [arg options]]]
Returns a new [uri \#securitydescriptors "security descriptor"].
The following options may be specified to initialize various fields
of the security descriptor:

[list_begin opt]
[opt_def [cmd -owner] [arg ACCOUNT]]
The owner field of the security descriptor is set to the SID for the
specified account. [arg ACCOUNT] may be the SID itself or the name of
the user or group account.

[opt_def [cmd -group] [arg ACCOUNT]]
The primary group field of the security descriptor is set to the SID for the
specified account. [arg ACCOUNT] may be the SID itself or the name of
the group.

[opt_def [cmd -dacl] [arg ACL]]
The DACL field of the security descriptor is set to [arg ACL]. If [arg ACL]
is the string [const \"null\"], no DACL will be present in the security descriptor.

[opt_def [cmd -sacl] [arg ACL]]
The SACL field of the security descriptor is set to [arg ACL]. If [arg ACL]
is the string [const \"null\"],
no SACL will be present in the security descriptor.

[list_end]

[call [cmd open_process_token] [opt "[cmd -pid] [arg PID] | [cmd -hprocess] [arg HPROCESS]"] [opt "[cmd -access] [arg DESIREDACCESS]"]]
Returns a process token that can passed to other functions for querying and
modifying security related data for a process. By default, the current
process is assumed. The options [cmd -hprocess] or [cmd -pid] options
may be used to retrieve the token for another process by passing a handle
to it or its PID.
[nl]
The [cmd -access] option may be used to specify a list of desired
access rights. If this option is
not specified, access rights default to [const token_query].
See [sectref "Access Rights"] for a description of the format of this
list and for a description of standard access rights.
[nl]
In addition, the
following symbols may be specified in the list:
[const token_adjust_default], [const token_adjust_groups],
[const token_adjust_privileges], [const token_adjust_sessionid],
[const token_assign_primary], [const token_duplicate],
[const token_execute], [const token_impersonate],
[const token_query], [const token_query_source],
[const token_read], [const token_write] and [const token_all_access].
See the documentation of [syscmd OpenProcessToken]
in the Windows SDK for a description of each of these.
[nl]
When no longer required, this token must be passed to
[uri \#close_token [cmd close_token]].

[call [cmd open_thread_token] [opt "[cmd -tid] [arg TID] | [cmd -hthread] [arg HTHREAD]"] [opt "[cmd -access] [arg DESIREDACCESS]"] [opt "[cmd -self] [arg BOOLEAN]"]]
Returns a thread access token that can passed to other functions for querying
and modifying security related data for a thread. By default, the current
thread is assumed. The options [cmd -hthread] or [cmd -tid] options
may be used to retrieve the token for another thread by passing a handle
to it or its TID.
[nl]
Note that the command will generate an error if the thread does not have
a token. In this case, [cmd ::errorCode] facility and code are
set to [const WINDOWS] and [const 1008] respectively.
The caller can retrieve the process token instead.
[nl]
The [cmd -access] option specifies the desired access mode.
See [uri \#open_process_token [cmd open_process_token]]
for more details.
[nl]
The [cmd -self] option controls whether the access check is to
be made against the security context of the thread or
against the security context of the process.
If specified as false, the security context of the thread is used, else
that of the process.
[nl]
When no longer required, this token must be passed to
[uri \#close_token [cmd close_token]].

[call [cmd open_user_token] [arg USERNAME] [arg PASSWORD] [opt [arg options]]]

Returns a token corresponding to a logon session for the [arg USERNAME]
account. This token may be used for impersonation or starting processes
under a different account. When no
longer required, this token must be passed to
[uri \#close_token [cmd close_token]].
[nl]
[arg USERNAME] is the name of the account and may be specified either as just
the account name, or in UPN format as [arg user@DNS_domain_name].
[nl]
The following options may be specified:
[list_begin opt]
[opt_def [cmd -domain] [arg DOMAIN]] Specifies the domain for the user. This
option must not be specified if [arg USERNAME] is in UPN format
[arg user@DNS_domain_name] which includes a domain name. If [arg USERNAME]
is not in UPN format and [cmd -domain] is not specified, the account
validation is done only using the local account database.
[opt_def [cmd -type] [arg LOGONTYPE]] Indicates the logon type.
[arg LOGONTYPE] must be one of [const batch], [const interactive],
[const network], [const network_cleartext], [const new_credentials],
[const service] or [const unlock]. Refer to the Windows SDK documentation
of [cmd LogonUser] for details.
[opt_def [cmd -provider] [arg PROVIDER]] Selects a logon provider.
[arg PROVIDER] must be one of [const winnt50],
[const winnt40] or [const winnt35]. If this option is not specified, the
operating system picks the standard (version dependent) logon provider for
the system. Specify [const winnt40] to select the NTLM provider and
[const winnt50] (only valid for Windows 2000 and up)
for the negotiate logon provider.
[list_end]

[call [cmd prepend_acl_aces] [arg ACL] [arg ACELIST]]
Prepends the list of ACEs in [arg ACELIST] to the entries in the given ACL.
Returns the modified ACL.

[call [cmd reset_thread_token]]
Ends the current thread's impersonation state.

[call [cmd revert_to_self]]
Calling this function will restore the original security context for the
process thereby ending any impersonation initiated by prior impersonation
commands such as [cmd impersonate_user].
[nl]
Any exceptions raised by this
command will result in the process continuing to run in the impersonation
context. This is dangerous from a security perspective and the caller should
normally immediately exit.

[call [cmd sddl_to_security_descriptor] [arg SECD]]
Parses a
[uri http://msdn.microsoft.com/en-us/library/aa379567%28v=VS.85%29.aspx SDDL]
string and returns the corresponding security descriptor.

[call [cmd security_descriptor_to_sddl] [arg SECD]]
Returns the specified security descriptor in
[uri http://msdn.microsoft.com/en-us/library/aa379567%28v=VS.85%29.aspx SDDL]
string format.

[call [cmd set_ace_inheritance] [arg ACE] [opt [arg options]]]
Returns a new ACE constructed from [arg ACE] with inheritance attributes
defined by [arg options]. These may be one or more of
[cmd -self], [cmd -recursecontainers], [cmd -recurseobjects] and
[cmd -recurseonelevelonly] as described in the documentation for
[cmd new_ace].

[call [cmd set_ace_rights] [arg ACE] [arg ACCESSRIGHTS]]
Returns a new ACE containing the access rights specified in [arg ACCESSRIGHTS].
This should be a list
of symbolic constants and bit masks as described in [sectref "Access Rights"].

[call [cmd set_ace_sid] [arg ACE] [arg ACCOUNT]]
Returns a new ACE constructed from [arg ACE] but with its SID field
set to be the SID for account [arg ACCOUNT]. [arg ACCOUNT] may be either
the name or the SID of a user or group account.

[call [cmd set_ace_type] [arg ACE] [arg ACETYPE]]
Returns a new ACE constructed from [arg ACE] but with the type [arg ACETYPE].
See [sectref "Access Control Entries"] for valid values of [arg ACETYPE].

[call [cmd set_acl_aces] [arg ACL] [arg ACELIST]]
Sets the list of ACEs within an ACL to those given in [arg ACELIST]. Returns
the modified ACL.


[call [cmd set_resource_integrity] [arg RESTYPE] [arg NAME] [arg INTEGRITYLEVEL] [arg INTEGRITYPOLICY] [opt [arg options]]]
Sets the integrity level associated with a resource. See
[uri #set_security_descriptor_integrity [cmd set_security_descriptor_integrity]]
for a description of the [arg INTEGRITYLEVEL] and [arg INTEGRITYPOLICY]
parameters and the options [cmd -recurseobjects] and [cmd -recursecontainers].
See [uri #set_resource_security_descriptor [cmd set_resource_security_descriptor]]
for description of [arg RESTYPE], [arg NAME] and the [cmd -handle] option.

[call [cmd set_resource_security_descriptor] [arg RESTYPE] [arg NAME] [arg SECD] [opt [arg options]]]

Modifies the security descriptor attached to a operating system resource.
[arg RESTYPE] specifies the resource type and if option [cmd -handle] is
not specified, must be one of [const file],
[const service] (Windows service), [const printer], [const registry],
[const share] (Network shares) or [const kernelobj]
(kernel objects such as mutexes, semaphores and events).
If option [cmd -handle] is specified, then [arg RESTYPE] may also
be one of [const windowstation], [const directoryservice],
[const directoryserviceall], [const providerdefined], [const wmiguid]
or [const registrywow6432key] but must not be [const share].
[nl]
[arg NAME] specifies the name of the resource, for example, the path of
a file unless option [cmd -handle] is specified in which case it is the
handle to the resource.
[nl]
The fields of the resource's security descriptor are modified based
on the contents of [arg SECD]. At lease one of the options
[cmd -all], [cmd -owner], [cmd -group], [cmd -dacl],
[cmd -sacl] or [cmd -mandatory_label]
must be specified.
The specific fields modified
depend on the options specified:

[list_begin opt]
[opt_def [const -all]] All fields in the security descriptor are modified.
This requires the caller to have the
[const SeSecurityPrivilege] privilege enabled.
[opt_def [const -owner]] The resource's owner is modified.
[opt_def [const -group]] The resource's group is modified.
[opt_def [const -dacl]] Changes the DACL attached to the resource.
[opt_def [const -sacl]] Changes the SACL attached to the resource.
This requires the caller to have the
[const SeSecurityPrivilege] privilege enabled.
[opt_def [const -protect_dacl]] If this option is specified, the resource
will not inherit any DACL's from its parent. Otherwise, the parent's DACL's
flow down to the resource. This option only has effect if the [const -dacl]
option is specified. This option cannot be
specified with [const -unprotect_dacl]. If neither is specified, the
current DACL inherit settings are preserved.
[opt_def [const -protect_sacl]] If this option is specified, the resource
will not inherit any SACL's from its parent. Otherwise, the parent's SACL's
flow down to the resource. This option only has effect if the [const -sacl]
option is specified. This option cannot be
specified with [const -unprotect_sacl]. If neither is specified, the
current SACL inherit settings are preserved.
[opt_def [const -unprotect_dacl]] If this option is specified, the resource
will inherit any DACL's from its parent.
This option only has effect if the [const -dacl]
option is specified. This option cannot be
specified with [const -protect_dacl]. If neither is specified, the
current DACL inherit settings are preserved.
[opt_def [const -unprotect_sacl]] If this option is specified, the resource
will inherit any SACL's from its parent.
This option only has effect if the [const -sacl]
option is specified. This option cannot be
specified with [const -protect_sacl]. If neither is specified, the
current SACL inherit settings are preserved.
[list_end]

[call [cmd set_security_descriptor_dacl] [arg SECD] [arg ACL] [opt [arg DEFAULTED]]]
Returns a security descriptor identical to [arg SECD]
except for the DACL field which is set to [arg ACL]. If [arg ACL] is
the string [const \"null\"], the security descriptor is set to have no DACL.
If [arg ACL] is an empty list, the security descriptor has an empty DACL,
which is [emph not] the same not having a DACL.
The boolean argument [arg DEFAULTED] (default [const false]) should be
[const true] if the DACL is to marked
as having been set by a default mechanism as opposed to being explicitly set.

[call [cmd set_security_descriptor_group] [arg SECD] [arg ACCOUNT] [opt [arg DEFAULTED]]]
Returns a security descriptor identical to [arg SECD]
except for the group field which is set to the SID for account [arg ACCOUNT].
[arg ACCOUNT] may be the SID itself or the name of the group.
The boolean argument [arg DEFAULTED] (default [const false]) should be
[const true] if the group is to marked
as having been set by a default mechanism as opposed to being explicitly set.

[call [cmd set_security_descriptor_integrity] [arg SECD] [arg INTEGRITYLEVEL] [arg INTEGRITYPOLICY] [opt [arg options]]]
Returns a new security descriptor based on [arg SECD] with the specified
integrity. [arg INTEGRITYLEVEL] may be specified in one of the forms
described in [uri #get_resource_integrity [cmd get_resource_integrity]].
[arg INTEGRITYPOLICY] is a list of one or more of the following:
[const system_mandatory_label_no_write_up],
[const system_mandatory_label_no_read_up],
[const system_mandatory_label_no_execute_up]. These control the operations
allowed on the resource by processes with lower integrity levels. Refer
to the Windows SDK for more details.
[nl]
The following options may also be specified with the command:
[list_begin opt]
[opt_def [cmd -recursecontainers] [arg BOOLEAN]] If the specified value is 1,
the integrity is also applied to all descendents of the object to which the
security descriptor is attached that are themselves containers.
If 0 (default), the integrity is not inherited by descendents that are containers.
[opt_def [cmd -recurseobjects] [arg BOOLEAN]] If the specified value is 1,
the integrity is also applied to all descendents of the object to which the 
security descriptor is attached that 
are [emph not] themselves containers.
If 0 (default), the integrity does not apply to descendents that are not containers.
[list_end]

[call [cmd set_security_descriptor_owner] [arg SECD] [arg ACCOUNT] [opt [arg DEFAULTED]]]
Returns a security descriptor identical to [arg SECD]
except for the owner field which is set to the SID for account [arg ACCOUNT].
[arg ACCOUNT] may be the SID itself or the name of
the user or group.
The boolean argument [arg DEFAULTED] (default [const false]) should be
[const true] if the owner is to marked
as having been set by a default mechanism as opposed to being explicitly set.

[call [cmd set_security_descriptor_sacl] [arg SECD] [arg ACL] [opt [arg DEFAULTED]]]
Returns a security descriptor identical to [arg SECD]
except for the SACL field which is set to [arg ACL]. If [arg ACL] is
the string [const \"null\"], the security descriptor is set to have no SACL.
The boolean argument [arg DEFAULTED] (default [const false]) should be
[const true] if the SACL is to marked
as having been set by a default mechanism as opposed to being explicitly set.


[call [cmd set_thread_token] [arg TOKEN]]
Changes the security context of the current thread by associating it with
the given token. This is generally used in combination with
[uri \#open_thread_token [cmd open_thread_token]]
to implement an impersonation stack.

[call [cmd set_token_integrity] [arg TOKEN] [arg INTEGRITYLEVEL]]
On systems that support Windows UAC, this command sets the integrity
level for a token. [arg INTEGRITYLEVEL] must be an integer value 
between 0 and 65535, a
a SID value defined for a integrity level by Windows, or
one of
[const untrusted] (0), [const low] (4096),
[const medium] (8192), [const mediumplus] (8448), [const high] (12288) or
[const system].
[nl]
On systems that do not support integrity levels, this command will raise
an error unless the integrity level specified corresponds to medium integrity
[const 8192], [const medium] or [const S-1-16-8192].

To directly set the integrity level of a process, you can also
[uri process.html#set_process_integrity [cmd set_process_integrity]] command.

[call [cmd set_token_integrity_policy] [arg TOKEN] [opt [const no_write_up]] [opt [const new_process_min]]]
On systems that support Windows UAC, this command sets the integrity policy
associated with the token. See
[uri #get_token_integrity_policy [cmd get_token_integrity_policy]]
for the meaning of [const no_write_up] and [const new_process_min].

[call [cmd set_token_virtualization] [arg TOKEN] [arg ENABLE]]
Enables or disables virtualization of file and registry accesses depending
on the boolean value of [arg ENABLE]. If this call is made on a
system that does not support UAC, [arg ENABLE] must be [const false] else
an exception will be generated. The token must have been opened
with [const token_adjust_default] access.

[call [cmd sids_from_same_domain] [arg SIDA] [arg SIDB]]
Returns [const 1] if both specified SIDs belong to the same domain
and [const 0] otherwise. Generates an error if either is not a valid SID.

[call [cmd sort_aces] [arg ACELIST]]
Although not enforced by the operating system, Windows has a recommended
order in which multiple ACEs should be applied. The [cmd sort_aces] command
takes a list of ACE elements and returns a list that has been sorted
as per the Windows recommendation. This sort order places all inherited ACEs
after all non-inherited ACEs. Within each group, deny ACEs are placed before
allow ACEs.

[call [cmd sort_acl_aces] [arg ACL]]
Rearranges the ACEs in the given ACL to conform to the Windows recommended
order. Returns the modified ACL.

[call [cmd well_known_sid] [arg KNOWNSIDNAME] [opt "[cmd -domainsid] [arg DOMAINSID]"]]
Returns the well known SID corresponding to [arg KNOWNSIDNAME]. The
[cmd -domainsid] option specifies the domain to use to create the SID.
If unspecified, the local computer is used.

[arg KNOWNSIDNAME] may be one of the following :
[const null],
[const world],
[const local],
[const creatorowner],
[const creatorgroup],
[const creatorownerserver],
[const creatorgroupserver],
[const ntauthority],
[const dialup],
[const network],
[const batch],
[const interactive],
[const service],
[const anonymous],
[const proxy],
[const enterprisecontrollers],
[const self],
[const authenticateduser],
[const restrictedcode],
[const terminalserver],
[const remotelogonid],
[const logonids],
[const localsystem],
[const localservice],
[const networkservice],
[const builtindomain],
[const builtinadministrators],
[const builtinusers],
[const builtinguests],
[const builtinpowerusers],
[const builtinaccountoperators],
[const builtinsystemoperators],
[const builtinprintoperators],
[const builtinbackupoperators],
[const builtinreplicator],
[const builtinprewindows2000compatibleaccess],
[const builtinremotedesktopusers],
[const builtinnetworkconfigurationoperators],
[const accountadministrator],
[const accountguest],
[const accountkrbtgt],
[const accountdomainadmins],
[const accountdomainusers],
[const accountdomainguests],
[const accountcomputers],
[const accountcontrollers],
[const accountcertadmins],
[const accountschemaadmins],
[const accountenterpriseadmins],
[const accountpolicyadmins],
[const accountrasandiasservers],
[const ntlmauthentication],
[const digestauthentication],
[const schannelauthentication],
[const thisorganization],
[const otherorganization],
[const builtinincomingforesttrustbuilders],
[const builtinperfmonitoringusers],
[const builtinperfloggingusers],
[const builtinauthorizationaccess],
[const builtinterminalserverlicenseservers],
[const builtindcomusers],
[const builtiniusers],
[const iuser],
[const builtincryptooperators],
[const untrustedlabel],
[const lowlabel],
[const mediumlabel],
[const highlabel],
[const systemlabel],
[const writerestrictedcode],
[const creatorownerrights],
[const cacheableprincipalsgroup],
[const noncacheableprincipalsgroup],
[const enterprisereadonlycontrollers],
[const accountreadonlycontrollers],
[const builtineventlogreadersgroup],
[const newenterprisereadonlycontrollers],
[const builtincertsvcdcomaccessgroup],
[const mediumpluslabel],
[const locallogon],
[const consolelogon],
[const thisorganizationcertificate],
[const applicationpackageauthority],
[const builtinanypackage],
[const capabilityinternetclient],
[const capabilityinternetclientserver],
[const capabilityprivatenetworkclientserver],
[const capabilitypictureslibrary],
[const capabilityvideoslibrary],
[const capabilitymusiclibrary],
[const capabilitydocumentslibrary],
[const capabilitysharedusercertificates],
[const capabilityenterpriseauthentication],
[const capabilityremovablestorage].

See [uri http://msdn.microsoft.com/en-us/library/windows/desktop/aa379650(v=vs.85).aspx] for meaning of each name.

In many cases, the command will also accept the name of the account 
on US English systems
in lieu of the above tokens. For example, you can specify
[const Administrators] instead of [const builtinadministrators].


[list_end]

[keywords UUID "universally unique identifiers" \
     LUID "locally unique identifiers" \
     tokens "access tokens" privileges \
     ACE "access control entry" ACL "access control list" \
     "security descriptor" "process privileges" "process token" \
     impersonation SID DACL "discretionary access control list" \
     SACL "system access control list" security "access control" \
     session "logon session" SDDL UAC "User Account Control" \
     "elevation level" "integrity level" "virtualization"
     ]
[manpage_end]
