# Copyright (c) 2013 Amazon.com, Inc. or its affiliates.  All Rights Reserved
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish, dis-
# tribute, sublicense, and/or sell copies of the Software, and to permit
# persons to whom the Software is furnished to do so, subject to the fol-
# lowing conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABIL-
# ITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT
# SHALL THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
# IN THE SOFTWARE.
#
from tests.compat import unittest
from tests.unit import AWSMockServiceTestCase
from tests.unit import MockServiceWithConfigTestCase

from boto.s3.connection import S3Connection, HostRequiredError
from boto.s3.connection import S3ResponseError, Bucket


class TestSignatureAlteration(AWSMockServiceTestCase):
    connection_class = S3Connection

    def test_unchanged(self):
        self.assertEqual(
            self.service_connection._required_auth_capability(),
            ['hmac-v4-s3']
        )

    def test_switched(self):
        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.cn-north-1.amazonaws.com.cn'
        )
        self.assertEqual(
            conn._required_auth_capability(),
            ['hmac-v4-s3']
        )


class TestAnon(MockServiceWithConfigTestCase):
    connection_class = S3Connection

    def test_generate_url(self):
        conn = self.connection_class(
            anon=True,
            host='s3.amazonaws.com'
        )
        url = conn.generate_url(0, 'GET', bucket='examplebucket', key='test.txt')
        self.assertNotIn('Signature=', url)

    def test_anon_default_taken_from_config_opt(self):
        self.config = {
            's3': {
                # Value must be a string for `config.getbool` to not crash.
                'no_sign_request': 'True',
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com',
        )
        url = conn.generate_url(
            0, 'GET', bucket='examplebucket', key='test.txt')
        self.assertNotIn('Signature=', url)

    def test_explicit_anon_arg_overrides_config_value(self):
        self.config = {
            's3': {
                # Value must be a string for `config.getbool` to not crash.
                'no_sign_request': 'True',
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com',
            anon=False
        )
        url = conn.generate_url(
            0, 'GET', bucket='examplebucket', key='test.txt')
        self.assertIn('Signature=', url)


class TestPresigned(MockServiceWithConfigTestCase):
    connection_class = S3Connection

    def test_presign_respect_query_auth(self):
        self.config = {
            's3': {
                'use-sigv4': False,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com'
        )

        url_enabled = conn.generate_url(86400, 'GET', bucket='examplebucket',
                                        key='test.txt', query_auth=True)

        url_disabled = conn.generate_url(86400, 'GET', bucket='examplebucket',
                                         key='test.txt', query_auth=False)
        self.assertIn('Signature=', url_enabled)
        self.assertNotIn('Signature=', url_disabled)


class TestSigV4HostError(MockServiceWithConfigTestCase):
    connection_class = S3Connection

    def test_historical_behavior(self):
        self.assertEqual(
            self.service_connection._required_auth_capability(),
            ['hmac-v4-s3']
        )
        self.assertEqual(self.service_connection.host, 's3.amazonaws.com')

    def test_sigv4_opt_in(self):
        host_value = 's3.cn-north-1.amazonaws.com.cn'

        # Switch it at the config, so we can check to see how the host is
        # handled.
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        # Ensure passing a ``host`` in the connection args still works.
        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host=host_value
        )
        self.assertEqual(
            conn._required_auth_capability(),
            ['hmac-v4-s3']
        )
        self.assertEqual(
            conn.host,
            host_value
        )

        # Ensure that the host is populated from our config if one is not
        # provided when creating a connection.
        self.config = {
            's3': {
                'host': host_value,
                'use-sigv4': True,
            }
        }
        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more'
        )
        self.assertEqual(
            conn._required_auth_capability(),
            ['hmac-v4-s3']
        )
        self.assertEqual(
            conn.host,
            host_value
        )


class TestSigV4Presigned(MockServiceWithConfigTestCase):
    connection_class = S3Connection

    def test_sigv4_presign(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com'
        )

        # Here we force an input iso_date to ensure we always get the
        # same signature.
        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt',
                                      iso_date='20140625T000000Z')

        self.assertIn(
            'a937f5fbc125d98ac8f04c49e0204ea1526a7b8ca058000a54c192457be05b7d',
            url)

    def test_sigv4_presign_respects_is_secure(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com',
            is_secure=True,
        )

        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt')
        self.assertTrue(url.startswith(
            'https://examplebucket.s3.amazonaws.com/test.txt?'))

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com',
            is_secure=False,
        )

        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt')
        self.assertTrue(url.startswith(
            'http://examplebucket.s3.amazonaws.com/test.txt?'))

    def test_sigv4_presign_optional_params(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            security_token='token',
            host='s3.amazonaws.com'
        )

        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt', version_id=2)

        self.assertIn('VersionId=2', url)
        self.assertIn('X-Amz-Security-Token=token', url)

    def test_sigv4_presign_respect_query_auth(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com'
        )

        url_enabled = conn.generate_url(86400, 'GET', bucket='examplebucket',
                                        key='test.txt', query_auth=True)

        url_disabled = conn.generate_url(86400, 'GET', bucket='examplebucket',
                                         key='test.txt', query_auth=False)
        self.assertIn('Signature=', url_enabled)
        self.assertNotIn('Signature=', url_disabled)

    def test_sigv4_presign_headers(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com'
        )

        headers = {'x-amz-meta-key': 'val'}
        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt', headers=headers)

        self.assertIn('host', url)
        self.assertIn('x-amz-meta-key', url)

    def test_sigv4_presign_response_headers(self):
        self.config = {
            's3': {
                'use-sigv4': True,
            }
        }

        conn = self.connection_class(
            aws_access_key_id='less',
            aws_secret_access_key='more',
            host='s3.amazonaws.com'
        )

        response_headers = {'response-content-disposition': 'attachment; filename="file.ext"'}
        url = conn.generate_url_sigv4(86400, 'GET', bucket='examplebucket',
                                      key='test.txt', response_headers=response_headers)

        self.assertIn('host', url)
        self.assertIn('response-content-disposition', url)


class TestUnicodeCallingFormat(AWSMockServiceTestCase):
    connection_class = S3Connection

    def default_body(self):
        return """<?xml version="1.0" encoding="UTF-8"?>
<ListAllMyBucketsResult xmlns="http://doc.s3.amazonaws.com/2006-03-01">
  <Owner>
    <ID>bcaf1ffd86f461ca5fb16fd081034f</ID>
    <DisplayName>webfile</DisplayName>
  </Owner>
  <Buckets>
    <Bucket>
      <Name>quotes</Name>
      <CreationDate>2006-02-03T16:45:09.000Z</CreationDate>
    </Bucket>
    <Bucket>
      <Name>samples</Name>
      <CreationDate>2006-02-03T16:41:58.000Z</CreationDate>
    </Bucket>
  </Buckets>
</ListAllMyBucketsResult>"""

    def create_service_connection(self, **kwargs):
        kwargs['calling_format'] = u'boto.s3.connection.OrdinaryCallingFormat'
        return super(TestUnicodeCallingFormat,
                     self).create_service_connection(**kwargs)

    def test_unicode_calling_format(self):
        self.set_http_response(status_code=200)
        self.service_connection.get_all_buckets()


class TestHeadBucket(AWSMockServiceTestCase):
    connection_class = S3Connection

    def default_body(self):
        # HEAD requests always have an empty body.
        return ""

    def test_head_bucket_success(self):
        self.set_http_response(status_code=200)
        buck = self.service_connection.head_bucket('my-test-bucket')
        self.assertTrue(isinstance(buck, Bucket))
        self.assertEqual(buck.name, 'my-test-bucket')

    def test_head_bucket_forbidden(self):
        self.set_http_response(status_code=403)

        with self.assertRaises(S3ResponseError) as cm:
            self.service_connection.head_bucket('cant-touch-this')

        err = cm.exception
        self.assertEqual(err.status, 403)
        self.assertEqual(err.error_code, 'AccessDenied')
        self.assertEqual(err.message, 'Access Denied')

    def test_head_bucket_notfound(self):
        self.set_http_response(status_code=404)

        with self.assertRaises(S3ResponseError) as cm:
            self.service_connection.head_bucket('totally-doesnt-exist')

        err = cm.exception
        self.assertEqual(err.status, 404)
        self.assertEqual(err.error_code, 'NoSuchBucket')
        self.assertEqual(err.message, 'The specified bucket does not exist')

    def test_head_bucket_other(self):
        self.set_http_response(status_code=405)

        with self.assertRaises(S3ResponseError) as cm:
            self.service_connection.head_bucket('you-broke-it')

        err = cm.exception
        self.assertEqual(err.status, 405)
        # We don't have special-cases for this error status.
        self.assertEqual(err.error_code, None)
        self.assertEqual(err.message, '')


if __name__ == "__main__":
    unittest.main()
