﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/autoscaling/AutoScaling_EXPORTS.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace AutoScaling {
namespace Model {

/**
 * <p>Describes information used for one or more scheduled scaling action updates
 * in a <a
 * href="https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_BatchPutScheduledUpdateGroupAction.html">BatchPutScheduledUpdateGroupAction</a>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/autoscaling-2011-01-01/ScheduledUpdateGroupActionRequest">AWS
 * API Reference</a></p>
 */
class ScheduledUpdateGroupActionRequest {
 public:
  AWS_AUTOSCALING_API ScheduledUpdateGroupActionRequest() = default;
  AWS_AUTOSCALING_API ScheduledUpdateGroupActionRequest(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_AUTOSCALING_API ScheduledUpdateGroupActionRequest& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_AUTOSCALING_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name of the scaling action.</p>
   */
  inline const Aws::String& GetScheduledActionName() const { return m_scheduledActionName; }
  inline bool ScheduledActionNameHasBeenSet() const { return m_scheduledActionNameHasBeenSet; }
  template <typename ScheduledActionNameT = Aws::String>
  void SetScheduledActionName(ScheduledActionNameT&& value) {
    m_scheduledActionNameHasBeenSet = true;
    m_scheduledActionName = std::forward<ScheduledActionNameT>(value);
  }
  template <typename ScheduledActionNameT = Aws::String>
  ScheduledUpdateGroupActionRequest& WithScheduledActionName(ScheduledActionNameT&& value) {
    SetScheduledActionName(std::forward<ScheduledActionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time for the action to start, in YYYY-MM-DDThh:mm:ssZ format in
   * UTC/GMT only and in quotes (for example,
   * <code>"2019-06-01T00:00:00Z"</code>).</p> <p>If you specify
   * <code>Recurrence</code> and <code>StartTime</code>, Amazon EC2 Auto Scaling
   * performs the action at this time, and then performs the action based on the
   * specified recurrence.</p> <p>If you try to schedule the action in the past,
   * Amazon EC2 Auto Scaling returns an error message.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ScheduledUpdateGroupActionRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time for the recurring schedule to end, in UTC.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ScheduledUpdateGroupActionRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The recurring schedule for the action, in Unix cron syntax format. This
   * format consists of five fields separated by white spaces: [Minute] [Hour]
   * [Day_of_Month] [Month_of_Year] [Day_of_Week]. The value must be in quotes (for
   * example, <code>"30 0 1 1,6,12 *"</code>). For more information about this
   * format, see <a href="http://crontab.org">Crontab</a>.</p> <p>When
   * <code>StartTime</code> and <code>EndTime</code> are specified with
   * <code>Recurrence</code>, they form the boundaries of when the recurring action
   * starts and stops.</p> <p>Cron expressions use Universal Coordinated Time (UTC)
   * by default.</p>
   */
  inline const Aws::String& GetRecurrence() const { return m_recurrence; }
  inline bool RecurrenceHasBeenSet() const { return m_recurrenceHasBeenSet; }
  template <typename RecurrenceT = Aws::String>
  void SetRecurrence(RecurrenceT&& value) {
    m_recurrenceHasBeenSet = true;
    m_recurrence = std::forward<RecurrenceT>(value);
  }
  template <typename RecurrenceT = Aws::String>
  ScheduledUpdateGroupActionRequest& WithRecurrence(RecurrenceT&& value) {
    SetRecurrence(std::forward<RecurrenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The minimum size of the Auto Scaling group.</p>
   */
  inline int GetMinSize() const { return m_minSize; }
  inline bool MinSizeHasBeenSet() const { return m_minSizeHasBeenSet; }
  inline void SetMinSize(int value) {
    m_minSizeHasBeenSet = true;
    m_minSize = value;
  }
  inline ScheduledUpdateGroupActionRequest& WithMinSize(int value) {
    SetMinSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum size of the Auto Scaling group.</p>
   */
  inline int GetMaxSize() const { return m_maxSize; }
  inline bool MaxSizeHasBeenSet() const { return m_maxSizeHasBeenSet; }
  inline void SetMaxSize(int value) {
    m_maxSizeHasBeenSet = true;
    m_maxSize = value;
  }
  inline ScheduledUpdateGroupActionRequest& WithMaxSize(int value) {
    SetMaxSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The desired capacity is the initial capacity of the Auto Scaling group after
   * the scheduled action runs and the capacity it attempts to maintain.</p>
   */
  inline int GetDesiredCapacity() const { return m_desiredCapacity; }
  inline bool DesiredCapacityHasBeenSet() const { return m_desiredCapacityHasBeenSet; }
  inline void SetDesiredCapacity(int value) {
    m_desiredCapacityHasBeenSet = true;
    m_desiredCapacity = value;
  }
  inline ScheduledUpdateGroupActionRequest& WithDesiredCapacity(int value) {
    SetDesiredCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the time zone for a cron expression. If a time zone is not
   * provided, UTC is used by default. </p> <p>Valid values are the canonical names
   * of the IANA time zones, derived from the IANA Time Zone Database (such as
   * <code>Etc/GMT+9</code> or <code>Pacific/Tahiti</code>). For more information,
   * see <a
   * href="https://en.wikipedia.org/wiki/List_of_tz_database_time_zones">https://en.wikipedia.org/wiki/List_of_tz_database_time_zones</a>.</p>
   */
  inline const Aws::String& GetTimeZone() const { return m_timeZone; }
  inline bool TimeZoneHasBeenSet() const { return m_timeZoneHasBeenSet; }
  template <typename TimeZoneT = Aws::String>
  void SetTimeZone(TimeZoneT&& value) {
    m_timeZoneHasBeenSet = true;
    m_timeZone = std::forward<TimeZoneT>(value);
  }
  template <typename TimeZoneT = Aws::String>
  ScheduledUpdateGroupActionRequest& WithTimeZone(TimeZoneT&& value) {
    SetTimeZone(std::forward<TimeZoneT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_scheduledActionName;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::String m_recurrence;

  int m_minSize{0};

  int m_maxSize{0};

  int m_desiredCapacity{0};

  Aws::String m_timeZone;
  bool m_scheduledActionNameHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_recurrenceHasBeenSet = false;
  bool m_minSizeHasBeenSet = false;
  bool m_maxSizeHasBeenSet = false;
  bool m_desiredCapacityHasBeenSet = false;
  bool m_timeZoneHasBeenSet = false;
};

}  // namespace Model
}  // namespace AutoScaling
}  // namespace Aws
