// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/ui/webui/policy/policy_ui.h"

#include <memory>

#include "build/build_config.h"
#include "chrome/browser/profiles/profile.h"
#include "chrome/browser/ui/webui/policy/policy_ui_handler.h"
#include "chrome/browser/ui/webui/webui_util.h"
#include "chrome/common/url_constants.h"
#include "components/grit/dev_ui_components_resources.h"
#include "components/strings/grit/components_strings.h"
#include "content/public/browser/web_ui.h"
#include "services/network/public/mojom/content_security_policy.mojom.h"
#include "ui/base/webui/web_ui_util.h"

namespace {

content::WebUIDataSource* CreatePolicyUIHtmlSource() {
  content::WebUIDataSource* source =
      content::WebUIDataSource::Create(chrome::kChromeUIPolicyHost);
  PolicyUIHandler::AddCommonLocalizedStringsToSource(source);

  static constexpr webui::LocalizedString kStrings[] = {
    // Localized strings (alphabetical order).
    {"copyPoliciesJSON", IDS_COPY_POLICIES_JSON},
    {"exportPoliciesJSON", IDS_EXPORT_POLICIES_JSON},
    {"filterPlaceholder", IDS_POLICY_FILTER_PLACEHOLDER},
    {"hideExpandedStatus", IDS_POLICY_HIDE_EXPANDED_STATUS},
    {"isAffiliatedYes", IDS_POLICY_IS_AFFILIATED_YES},
    {"isAffiliatedNo", IDS_POLICY_IS_AFFILIATED_NO},
    {"labelAssetId", IDS_POLICY_LABEL_ASSET_ID},
    {"labelClientId", IDS_POLICY_LABEL_CLIENT_ID},
    {"labelDirectoryApiId", IDS_POLICY_LABEL_DIRECTORY_API_ID},
    {"labelGaiaId", IDS_POLICY_LABEL_GAIA_ID},
    {"labelIsAffiliated", IDS_POLICY_LABEL_IS_AFFILIATED},
    {"labelLocation", IDS_POLICY_LABEL_LOCATION},
    {"labelMachineEnrollmentDomain",
     IDS_POLICY_LABEL_MACHINE_ENROLLMENT_DOMAIN},
    {"labelMachineEnrollmentMachineName",
     IDS_POLICY_LABEL_MACHINE_ENROLLMENT_MACHINE_NAME},
    {"labelMachineEnrollmentToken", IDS_POLICY_LABEL_MACHINE_ENROLLMENT_TOKEN},
    {"labelMachineEntrollmentDeviceId",
     IDS_POLICY_LABEL_MACHINE_ENROLLMENT_DEVICE_ID},
    {"labelIsOffHoursActive", IDS_POLICY_LABEL_IS_OFFHOURS_ACTIVE},
    {"labelPoliciesPush", IDS_POLICY_LABEL_PUSH_POLICIES},
    {"labelPrecedence", IDS_POLICY_LABEL_PRECEDENCE},
    {"labelRefreshInterval", IDS_POLICY_LABEL_REFRESH_INTERVAL},
    {"labelStatus", IDS_POLICY_LABEL_STATUS},
    {"labelTimeSinceLastFetchAttempt",
     IDS_POLICY_LABEL_TIME_SINCE_LAST_FETCH_ATTEMPT},
    {"labelTimeSinceLastRefresh", IDS_POLICY_LABEL_TIME_SINCE_LAST_REFRESH},
    {"labelUsername", IDS_POLICY_LABEL_USERNAME},
    {"labelManagedBy", IDS_POLICY_LABEL_MANAGED_BY},
    {"labelVersion", IDS_POLICY_LABEL_VERSION},
    {"noPoliciesSet", IDS_POLICY_NO_POLICIES_SET},
    {"offHoursActive", IDS_POLICY_OFFHOURS_ACTIVE},
    {"offHoursNotActive", IDS_POLICY_OFFHOURS_NOT_ACTIVE},
    {"policyCopyValue", IDS_POLICY_COPY_VALUE},
    {"policiesPushOff", IDS_POLICY_PUSH_POLICIES_OFF},
    {"policiesPushOn", IDS_POLICY_PUSH_POLICIES_ON},
    {"policyLearnMore", IDS_POLICY_LEARN_MORE},
    {"reloadPolicies", IDS_POLICY_RELOAD_POLICIES},
    {"showExpandedStatus", IDS_POLICY_SHOW_EXPANDED_STATUS},
    {"showLess", IDS_POLICY_SHOW_LESS},
    {"showMore", IDS_POLICY_SHOW_MORE},
    {"showUnset", IDS_POLICY_SHOW_UNSET},
    {"signinProfile", IDS_POLICY_SIGNIN_PROFILE},
    {"status", IDS_POLICY_STATUS},
    {"statusDevice", IDS_POLICY_STATUS_DEVICE},
    {"statusMachine", IDS_POLICY_STATUS_MACHINE},
#if BUILDFLAG(IS_WIN) && BUILDFLAG(GOOGLE_CHROME_BRANDING)
    {"statusUpdater", IDS_POLICY_STATUS_UPDATER},
#endif
    {"statusUser", IDS_POLICY_STATUS_USER},
    {"labelLastCloudReportSentTimestamp",
     IDS_POLICY_LABEL_LAST_CLOUD_REPORT_SENT_TIMESTAMP},
  };
  source->AddLocalizedStrings(kStrings);

  source->AddResourcePath("policy.css", IDR_POLICY_CSS);
  source->AddResourcePath("policy_base.js", IDR_POLICY_BASE_JS);
  source->AddResourcePath("policy.js", IDR_POLICY_JS);
  source->AddResourcePath("policy_conflict.html.js",
                          IDR_POLICY_POLICY_CONFLICT_HTML_JS);
  source->AddResourcePath("policy_conflict.js", IDR_POLICY_POLICY_CONFLICT_JS);
  source->AddResourcePath("policy_row.html.js", IDR_POLICY_POLICY_ROW_HTML_JS);
  source->AddResourcePath("policy_row.js", IDR_POLICY_POLICY_ROW_JS);
  source->AddResourcePath("policy_precedence_row.html.js",
                          IDR_POLICY_POLICY_PRECEDENCE_ROW_HTML_JS);
  source->AddResourcePath("policy_precedence_row.js",
                          IDR_POLICY_POLICY_PRECEDENCE_ROW_JS);
  source->AddResourcePath("policy_table.html.js",
                          IDR_POLICY_POLICY_TABLE_HTML_JS);
  source->AddResourcePath("policy_table.js", IDR_POLICY_POLICY_TABLE_JS);
  source->AddResourcePath("status_box.html.js", IDR_POLICY_STATUS_BOX_HTML_JS);
  source->AddResourcePath("status_box.js", IDR_POLICY_STATUS_BOX_JS);
  source->SetDefaultResource(IDR_POLICY_HTML);

  source->EnableReplaceI18nInJS();
  source->OverrideContentSecurityPolicy(
      network::mojom::CSPDirectiveName::TrustedTypes,
      "trusted-types static-types;");
  return source;
}

}  // namespace

PolicyUI::PolicyUI(content::WebUI* web_ui) : WebUIController(web_ui) {
  web_ui->AddMessageHandler(std::make_unique<PolicyUIHandler>());
  content::WebUIDataSource::Add(Profile::FromWebUI(web_ui),
                                CreatePolicyUIHtmlSource());
}

PolicyUI::~PolicyUI() = default;
