/****************************************************************************
**
** Copyright (C) 2025 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** This file is part of the release tools of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:GPL-EXCEPT$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3 as published by the Free Software
** Foundation with exceptions as appearing in the file LICENSE.GPL3-EXCEPT
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/

// constructor

var ifwVersion;
var warningMessage;

function Component()
{
    ifwVersion = installer.value("FrameworkVersion");

    if (installer.versionMatches(ifwVersion, "<4.7.0") && systemInfo.productType != "macos" && systemInfo.productType != "windows") {
        var systemGlibcVersion =  glibcVersion();
        if (isTooOldGlibcVersion(systemGlibcVersion)) {
            warningMessage = ("Qt Maintenance Tool version 4.7.0 and above requires at least glibc version 2.28. "+
                                "Seems that your glibc version (" + systemGlibcVersion + ") is older.")

            if (installer.isInstaller()) {
                // Because model is not yet up to date (rootComponentList is empty) , we need to connect to finishAllComponentsReset() -signal
                // in order to deselect qt.tools.maintenance component
                warningMessage += " We advice not to install the Maintenance Tool."
                console.log(warningMessage);
                installer.finishAllComponentsReset.connect(Component.prototype.finishAllComponentsReset)
            } else if (installer.isUpdater()) {
                warningMessage += " Your Maintenance Tool will not be updated to a newer version."
                console.log(warningMessage);
                component.setValue("Unstable", true);
                component.setValue("Description", warningMessage);
            }
        }
    }
    installer.installationStarted.connect(this, Component.prototype.onInstallationStarted);
}

Component.prototype.finishAllComponentsReset = function()
{
    component.setValue("ForcedInstallation", false);
    component.setValue("Description", warningMessage);
    component.setValue("Checkable", true);
    installer.deselectComponent("qt.tools.maintenance");
}

isTooOldGlibcVersion = function(systemGlibcVersion) {
    return (systemGlibcVersion != 0 && installer.versionMatches(systemGlibcVersion, "<2.28"))
}

glibcVersion = function() {
    var systemGlibcVersion = 0;

    var lddOutput = installer.execute("ldd", "--version");
    var glicVersion = lddOutput[0];
    if (!glicVersion || glicVersion.length <= 0) {
        lddOutput = installer.execute("rpm", "-q glibc");
        glicVersion = lddOutput[0]
    }
    if (glicVersion && glicVersion.length > 0) {
        var glibcVersionArray = glicVersion.match("2\\.\\d*");
        if (glibcVersionArray.length > 0 ) {
            systemGlibcVersion = glibcVersionArray[0];
        }
    }
    if (systemGlibcVersion == 0)
        console.log("Not able to read system glibc version");

    return systemGlibcVersion;
}

Component.prototype.onInstallationStarted = function()
{
    if (component.updateRequested() || component.installationRequested()) {
        if (installer.value("os") == "win") {
            component.installerbaseBinaryPath = "@TargetDir@/installerbase.exe";
        } else if (installer.value("os") == "x11") {
            component.installerbaseBinaryPath = "@TargetDir@/installerbase";
        } else if (installer.value("os") == "mac") {
            // From IFW version 4.5.0 onwards the macOS maintenance tool can be the whole
            // app bundle.
            ifwVersion = installer.value("FrameworkVersion");
            if (installer.versionMatches(ifwVersion, "<4.5.0")) {
                component.installerbaseBinaryPath = "@TargetDir@/installerbase";
            } else if (installer.versionMatches(ifwVersion, "<=4.8.0")) {
                component.installerbaseBinaryPath = "@TargetDir@/MaintenanceTool.app";
            } else {
                component.installerbaseBinaryPath = "@TargetDir@/tmpMaintenanceToolApp/MaintenanceTool.app";
            }
        }
        installer.setInstallerBaseBinary(component.installerbaseBinaryPath);

        // update resource file from archive
        var updateResourceFilePath = "@TargetDir@/update.rcc";
        var normalizedUpdateResourceFilePath = updateResourceFilePath.replace(/@TargetDir@/, installer.value("TargetDir"));
        print("Updating resource file: " + normalizedUpdateResourceFilePath);
        installer.setValue("DefaultResourceReplacement", normalizedUpdateResourceFilePath);
    }
}

Component.prototype.createOperationsForArchive = function(archive)
{
    // Component contains two archives which both are maintenancetools - MaintenanceTool.app bundle
    // in archive <version>MaintenanceToolApp.7z, and maintenancetool executable in archive
    // maintenancetool_<platform>_<architecture>.7z. Installer supports app bundle extraction only from 4.5.0 onwards.
    // Two maintenancetool is needed so that the older versions of maintenance tool can be updated to
    // version 4.5.0, as old versions does not support the app bundle extraction yet.
    // Extract only other maintenancetool.
    // From 4.8.1 onwards maintenance tool is extracted to a folder so it does not interfere the
    // currently running maintenancetool.
    if (installer.value("os") == "mac") {
        if (installer.versionMatches(ifwVersion, ">=4.5.0")) {
            if (archive.endsWith("maintenancetool_linux_arm64.7z"))
                return;
            if (archive.endsWith("%MAINTENANCE_TOOL_APP_FILENAME%")) {
                console.log("Extracting macos maintenancetool.app bundle");
                component.addOperation("Extract", archive, "@TargetDir@/tmpMaintenanceToolApp");
            } else {
                component.createOperationsForArchive(archive);
            }
        } else {
            if (!archive.endsWith("%MAINTENANCE_TOOL_APP_FILENAME%")) {
                console.log("Extracting macos maintenance tool");
                component.addOperation("Extract", archive, "@TargetDir@");
            }
        }
    } else {
        component.createOperationsForArchive(archive);
    }
}

Component.prototype.createOperations = function()
{
    // Call the base createOperations and afterwards set some registry settings (unpacking ...)
    component.createOperations();

    var editionName = "Qt";
    if (!(installer.value("QT_EDITION_NAME") === ""))
        editionName = installer.value("QT_EDITION_NAME");

    // Create uninstall link only for windows
    if (installer.value("os") == "win")
    {
        // shortcut to uninstaller
        component.addOperation( "CreateShortcut",
                                "@TargetDir@/MaintenanceTool.exe",
                                "@StartMenuDir@/Uninstall " + editionName + ".lnk",
                                "--start-uninstaller");
    }
    // only for windows online installer
    if ( installer.value("os") == "win" && !installer.isOfflineOnly() )
    {
        // create shortcut
        component.addOperation( "CreateShortcut",
                                "@TargetDir@/MaintenanceTool.exe",
                                "@StartMenuDir@/" + editionName + " Maintenance Tool.lnk",
                                "workingDirectory=@TargetDir@",
                                "--start-package-manager");
    }
    if ( installer.value("os") == "x11" )
    {
        component.addOperation( "InstallIcons", "@TargetDir@/icons" );
        // only for online installer
        if (!installer.isOfflineOnly()) {
            component.addOperation( "CreateDesktopEntry",
                                    editionName + "-MaintenanceTool.desktop",
                                    "Type=Application\nExec=@TargetDir@/MaintenanceTool\nPath=@TargetDir@\nName=" + editionName + " Maintenance Tool\nGenericName=Install or uninstall Qt components.\nIcon=QtIcon\nTerminal=false\nCategories=Development;Qt;"
                                   );
        }
    }
    // Move the maintenancetool app bundle to root which was extracted to tmpMaintenanceToolApp. From IFW 4.8.1 versions
    // onwards this done in IFW code, below 4.5.0 versions does not have app bundle in repository.
    if (installer.value("os") == "mac"
            && installer.versionMatches(ifwVersion, ">=4.5.0")
            && installer.versionMatches(ifwVersion, "<=4.8.0")) {
        component.addOperation("Execute", "{0}","ditto", "@TargetDir@/tmpMaintenanceToolApp/MaintenanceTool.app/", "@TargetDir@/MaintenanceTool.app/");
        component.addOperation("Execute", "{0}","rm", "-r", "@TargetDir@/tmpMaintenanceToolApp");
    }
}


