// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CachedMetadataHandler_h
#define CachedMetadataHandler_h

#include <stdint.h>
#include "platform/heap/Handle.h"
#include "platform/wtf/Forward.h"

namespace blink {

class CachedMetadata;

// Handler class for caching operations.
class CachedMetadataHandler
    : public GarbageCollectedFinalized<CachedMetadataHandler> {
 public:
  enum CacheType {
    kSendToPlatform,  // send cache data to blink::Platform::cacheMetadata
    kCacheLocally     // cache only in Resource's member variables
  };
  virtual ~CachedMetadataHandler() = default;
  virtual void Trace(blink::Visitor* visitor) {}
  // Caches the given metadata in association with this resource and suggests
  // that the platform persist it. The dataTypeID is a pseudo-randomly chosen
  // identifier that is used to distinguish data generated by the caller.
  virtual void SetCachedMetadata(uint32_t data_type_id,
                                 const char*,
                                 size_t,
                                 CacheType = kSendToPlatform) = 0;
  // Reset existing metadata, to allow setting new data.
  virtual void ClearCachedMetadata(CacheType = kCacheLocally) = 0;
  // Returns cached metadata of the given type associated with this resource.
  // This cached metadata can be pruned at any time.
  virtual scoped_refptr<CachedMetadata> GetCachedMetadata(
      uint32_t data_type_id) const = 0;
  // Returns the encoding to which the cache is specific.
  virtual String Encoding() const = 0;

  virtual bool IsServedFromCacheStorage() const = 0;

 protected:
  CachedMetadataHandler() = default;
};
}  // namespace blink

#endif  // CachedMetadataHandler_h
