/*
 * We want a reentrant scanner.
 */
%option reentrant

/*
 * We don't use input, so don't generate code for it.
 */
%option noinput

/*
 * We don't use unput, so don't generate code for it.
 */
%option nounput

/*
 * We don't read interactively from the terminal.
 */
%option never-interactive

/*
 * We want to stop processing when we get to the end of the input.
 */
%option noyywrap

/*
 * The type for the state we keep for a scanner.
 */
%option extra-type="uat_load_scanner_state_t *"

/*
 * We have to override the memory allocators so that we don't get
 * "unused argument" warnings from the yyscanner argument (which
 * we don't use, as we have a global memory allocator).
 *
 * We provide, as macros, our own versions of the routines generated by Flex,
 * which just call malloc()/realloc()/free() (as the Flex versions do),
 * discarding the extra argument.
 */
%option noyyalloc
%option noyyrealloc
%option noyyfree

/*
 * Prefix scanner routines with "uat_load_" rather than "yy", so this scanner
 * can coexist with other scanners.
 */
%option prefix="uat_load_"

%{
	/*
	 * uat_load.l
	 *
	 *  User Accessible Tables
	 *  Maintain an array of user accessible data strucures
	 *  One parser to fit them all
	 *
	 * (c) 2007, Luis E. Garcia Ontanon <luis@ontanon.org>
	 *
	 * Wireshark - Network traffic analyzer
	 * By Gerald Combs <gerald@wireshark.org>
	 * Copyright 2001 Gerald Combs
	 *
	 * This program is free software; you can redistribute it and/or
	 * modify it under the terms of the GNU General Public License
	 * as published by the Free Software Foundation; either version 2
	 * of the License, or (at your option) any later version.
	 *
	 * This program is distributed in the hope that it will be useful,
	 * but WITHOUT ANY WARRANTY; without even the implied warranty of
	 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	 * GNU General Public License for more details.
	 *
	 * You should have received a copy of the GNU General Public License
	 * along with this program; if not, write to the Free Software
	 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
	 */
#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <glib.h>

#include "uat-int.h"
#include <wsutil/file_util.h>

#ifdef _WIN32
/* disable Windows VC compiler warning "signed/unsigned mismatch" associated  */
/* with YY_INPUT code generated by flex versions such as 2.5.35.              */
#pragma warning (disable:4018)
#endif

typedef struct {
	uat_t* uat;
	gchar *parse_str;

	gchar* error;
	gboolean valid_record;
	guint colnum;
	gchar* ptrx;
	guint len;
	void* record;
	guint linenum;
	size_t parse_str_pos;
} uat_load_scanner_state_t;

#define ERROR(fmtd) do { \
    char* fmt_str = g_strdup_printf fmtd; \
    yyextra->error = g_strdup_printf("%s:%d: %s",yyextra->uat->filename,yyextra->linenum,fmt_str); \
    g_free(fmt_str); \
    yyterminate(); \
} while(0)

#define SET_FIELD() \
	{ gchar* errx; \
	if (yyextra->uat->fields[yyextra->colnum].cb.chk) { \
		if ( ! yyextra->uat->fields[yyextra->colnum].cb.chk(yyextra->record, yyextra->ptrx, yyextra->len, yyextra->uat->fields[yyextra->colnum].cbdata.chk, yyextra->uat->fields[yyextra->colnum].fld_data, &errx) ) { \
			yyextra->error = g_strdup_printf("%s:%d: %s",yyextra->uat->filename,yyextra->linenum,errx); \
			g_free(errx); \
			yyextra->valid_record = FALSE; \
		}\
	}\
	yyextra->uat->fields[yyextra->colnum].cb.set(yyextra->record, yyextra->ptrx, yyextra->len, yyextra->uat->fields[yyextra->colnum].cbdata.chk, yyextra->uat->fields[yyextra->colnum].fld_data);\
	g_free(yyextra->ptrx);\
	yyextra->colnum++; \
	} while(0)

#ifdef DEBUG_UAT_LOAD
#define DUMP_FIELD(str) \
		{ guint i; printf("%s: %s='",str,yyextra->uat->fields[yyextra->colnum].name); for(i=0;i<yyextra->len;i++) if (yyextra->uat->fields[yyextra->colnum].mode == PT_TXTMOD_HEXBYTES) { printf("%.2x ",((guint8*)yyextra->ptrx)[i]); } else putc(yyextra->ptrx[i],stdout); printf("'[%d]\n",yyextra->len); }

#define DUMP(str) printf("%s\n",str)
#else
#define DUMP_FIELD(s)
#define DUMP(s)
#endif

/* Modified version of YY_INPUT generated by Flex 2.91 */
#define YY_INPUT(buf,result,max_size) \
	if ( yyextra->parse_str ) \
		{ \
		size_t n = 0; \
		size_t pslen = strlen(yyextra->parse_str); \
		if (yyextra->parse_str_pos < pslen) \
			{ \
			n = pslen - yyextra->parse_str_pos; \
			if (n > max_size) n = max_size; \
			memcpy(buf, yyextra->parse_str + yyextra->parse_str_pos, n); \
			yyextra->parse_str_pos += n; \
			} \
		result = n; \
		} \
	else \
		{ \
		errno=0; \
		while ( (result = fread(buf, 1, max_size, yyin))==0 && ferror(yyin)) \
			{ \
			if( errno != EINTR) \
				{ \
				YY_FATAL_ERROR( "input in flex scanner failed" ); \
				break; \
				} \
			errno=0; \
			clearerr(yyin); \
			} \
		}

		/*
		 * XXX
		 * quoted_string below fails badly on "...\\"
		 * workarround in uat_save(), using /x5c and /x22
		 */

#define YY_USER_INIT BEGIN START_OF_LINE;

/*
 * Sleazy hack to suppress compiler warnings in yy_fatal_error().
 */
#define YY_EXIT_FAILURE ((void)yyscanner, 2)

/*
 * Macros for the allocators, to discard the extra argument.
 */
#define uat_load_alloc(size, yyscanner)		(void *)malloc(size)
#define uat_load_realloc(ptr, size, yyscanner)	(void *)realloc((char *)(ptr), (size))
#define uat_load_free(ptr, yyscanner)		free((char *)ptr)

%}

quoted_string \042([^\042]|\134\042)*\042
binstring ([0-9a-zA-Z][0-9a-zA-Z])*
separator [ \t]*,
newline [ \t]*[\r]?\n
ws [ \t]+
comment #[^\n]*\n

%x START_OF_LINE NEXT_FIELD SEPARATOR END_OF_RECORD ERRORED
%%
<START_OF_LINE,NEXT_FIELD>{ws} ;
<START_OF_LINE>{newline} yyextra->linenum++;
<START_OF_LINE>{comment} yyextra->linenum++;

<START_OF_LINE,NEXT_FIELD>{separator} {
	yyextra->ptrx = g_strdup("");
	yyextra->len = 0;

	DUMP_FIELD("empty->next");

	SET_FIELD();

	if ( yyextra->colnum >= yyextra->uat->ncols ) {
		ERROR(("more fields than required"));
	}

	BEGIN NEXT_FIELD;
}

<START_OF_LINE,NEXT_FIELD>{newline}   {
	yyextra->ptrx = g_strdup("");
	yyextra->len = 0;

	BEGIN END_OF_RECORD;

	yyless((int) yyleng);
}

<START_OF_LINE,NEXT_FIELD>{quoted_string} {
	yyextra->ptrx = uat_undquote(yytext, (guint) yyleng, &yyextra->len);


	if (yyextra->colnum < yyextra->uat->ncols - 1) {
		DUMP("quoted_str->s");
		BEGIN SEPARATOR;
	} else {
		DUMP("quoted_str->eor");
		BEGIN END_OF_RECORD;
	}
}

<START_OF_LINE,NEXT_FIELD>{binstring} {
	yyextra->ptrx = uat_unbinstring(yytext,  (guint) yyleng, &yyextra->len);

	if (!yyextra->ptrx) {
		ERROR(("uneven hexstring for field %s",yyextra->uat->fields[yyextra->colnum].name));
	}

	if ( yyextra->colnum < yyextra->uat->ncols - 1 ) {
		DUMP("binstring->s");
		BEGIN SEPARATOR;
	} else {
		DUMP("binstring->eor");
		BEGIN END_OF_RECORD;
	}
}

<SEPARATOR>{separator} {

	DUMP_FIELD("separator->next");

	SET_FIELD();

	if ( yyextra->colnum >= yyextra->uat->ncols ) {
		ERROR(("more fields than required"));
	}

	BEGIN NEXT_FIELD;
}

<SEPARATOR>{newline} {
	yyextra->linenum++;
	ERROR(("expecting field %s in previous line",yyextra->uat->fields[yyextra->colnum].name));
}

<SEPARATOR>. {
	ERROR(("unexpected char '%s' while looking for field %s",yytext,yyextra->uat->fields[yyextra->colnum].name));
}

<END_OF_RECORD>{separator} {
	ERROR(("more fields than required"));
}

<END_OF_RECORD>{newline} {
	void* rec;
	char* err = NULL;

	yyextra->linenum++;

	DUMP_FIELD("newline->start");

	SET_FIELD();

	rec = uat_add_record(yyextra->uat, yyextra->record, yyextra->valid_record);

	if ((yyextra->uat->update_cb) && (rec != NULL)) {
		if (!yyextra->uat->update_cb(rec,&err)) {
			yyextra->error = err;
			yyterminate();
		}
	}

	yyextra->valid_record = TRUE;
	yyextra->colnum = 0;
	yyextra->ptrx = NULL;
	yyextra->len = 0;

	/* XXX is this necessary since we free it before reusing anyway? */
	memset(yyextra->record,0,yyextra->uat->record_size);

	BEGIN START_OF_LINE;
 }

<END_OF_RECORD>. {
	ERROR(("unexpected char while looking for end of line"));
}

<ERRORED>{newline} { yyextra->linenum++; BEGIN START_OF_LINE; }
<ERRORED>. ;

{newline} { yyextra->linenum++; ERROR(("incomplete record")); }
. { ERROR(("unexpected input")); }

%%




gboolean
uat_load(uat_t *uat, char **errx)
{
	gchar *fname = uat_get_actual_filename(uat, FALSE);
	FILE *in;
	yyscan_t scanner;
	uat_load_scanner_state_t state;

	if (!fname) {
		UAT_UPDATE(uat);

		if (uat->post_update_cb)
			uat->post_update_cb();

		return TRUE;
	}


	if (!(in = ws_fopen(fname,"r"))) {
		*errx = g_strdup(g_strerror(errno));
		g_free(fname);
		return FALSE;
	}

	if (uat_load_lex_init(&scanner) != 0) {
		*errx = g_strdup(g_strerror(errno));
		fclose(in);
		g_free(fname);
		return FALSE;
	}

	uat_load_set_in(in, scanner);

	state.uat = uat;
	state.parse_str = NULL;	/* we're reading from a file */

	state.error = NULL;
	state.valid_record = TRUE;
	state.colnum = 0;
	state.ptrx = NULL;
	state.len = 0;
	state.record = g_malloc0(uat->record_size);
	state.linenum = 1;
	state.parse_str_pos = 0;

	DUMP(fname);
	g_free(fname);	/* we're done with the file name now */

	/* Associate the state with the scanner */
	uat_load_set_extra(&state, scanner);

	uat_load_lex(scanner);

	uat_load_lex_destroy(scanner);
	g_free(state.record);
	fclose(in);

	uat->changed = FALSE;
	uat->loaded = TRUE;
	UAT_UPDATE(uat);

	if (state.error) {
		*errx = state.error;
		return FALSE;
	}

	if (uat->post_update_cb)
		uat->post_update_cb();

	*errx = NULL;
	return TRUE;
}

gboolean
uat_load_str(uat_t *uat, char *entry, char **err)
{
	yyscan_t scanner;
	uat_load_scanner_state_t state;

	state.uat = uat;
	state.parse_str = g_strdup_printf("%s\n", entry); /* Records must end with a newline */

	state.error = NULL;
	state.valid_record = TRUE;
	state.colnum = 0;
	state.ptrx = NULL;
	state.len = 0;
	state.record = g_malloc0(uat->record_size);
	state.linenum = 1;
	state.parse_str_pos = 0;

	if (uat_load_lex_init(&scanner) != 0) {
		*err = g_strdup(g_strerror(errno));
		g_free(state.parse_str);
		g_free(state.record);
		return FALSE;
	}

	DUMP(entry);

	/* Associate the state with the scanner */
	uat_load_set_extra(&state, scanner);

	uat_load_lex(scanner);

	uat_load_lex_destroy(scanner);
	g_free(state.record);
	g_free(state.parse_str);

	uat->changed = TRUE;
	uat->loaded = TRUE;
	UAT_UPDATE(uat);

	if (state.error) {
		*err = state.error;
		return FALSE;
	}

	if (uat->post_update_cb)
		uat->post_update_cb();

	*err = NULL;
	return TRUE;
}
