/* nmfilter.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2001-2019 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "nmfilter.hh"
#include "configheader.h"
#include "configparser.hh"
#include "wconfig.h"
#include "datei.h"
#include <aguix/lowlevelfunc.h>

NM_Filter::NM_Filter()
{
    pattern = NULL;
    _check = INACTIVE;
}

NM_Filter::~NM_Filter()
{
  if(pattern!=NULL) _freesafe(pattern);
}

NM_Filter::NM_Filter( const NM_Filter &other )
{
    pattern = NULL;
    _check = INACTIVE;
    setPattern( other.getPattern() );
    setCheck( other.getCheck() );
}

NM_Filter &NM_Filter::operator=( const NM_Filter &other )
{
    if ( this != &other ) {
        setPattern( other.getPattern() );
        setCheck( other.getCheck() );
    }
    return *this;
}

void NM_Filter::setPattern(const char *np)
{
  if(pattern!=NULL) _freesafe(pattern);
  if(np!=NULL) {
    pattern=dupstring(np);
  } else {
    pattern=NULL;
  }
}

const char *NM_Filter::getPattern() const
{
  return pattern;
}

void NM_Filter::setCheck( check_t mode )
{
    _check = mode;
}

NM_Filter::check_t NM_Filter::getCheck() const
{
    return _check;
}

int NM_Filter::load()
{
  int found_error = 0;

  setCheck( INACTIVE );
  for (;;) {
    if ( worker_token == PATTERN_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token == STRING_WCP ) {
        if ( pattern != NULL ) _freesafe( pattern );
        pattern = dupstring( yylval.strptr );
      } else {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else if ( worker_token == MODE_WCP ) {
      readtoken();

      if ( worker_token != '=' ) {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token == INCLUDE_WCP ) {
        setCheck( INCLUDE );
      } else if ( worker_token == EXCLUDE_WCP ) {
        setCheck( EXCLUDE );
      } else {
        found_error = 1;
        break;
      }      
      readtoken();

      if ( worker_token != ';' ) {
        found_error = 1;
        break;
      }
      readtoken();
    } else {
      break;
    }
  }
  
  return found_error;
}

int NM_Filter::save(Datei *fh)
{
  if ( fh == NULL ) return 1;

  if ( pattern != NULL ) fh->configPutPairString( "pattern", pattern );  
  switch ( _check ) {
    case INCLUDE:
      fh->configPutPair( "mode", "include" );
      break;
    case EXCLUDE:
      fh->configPutPair( "mode", "exclude" );
      break;
    default:
      break;
  }
  return 0;
}

NM_Filter *NM_Filter::duplicate()
{
    NM_Filter *fi = new NM_Filter();
    fi->setCheck( _check );
    fi->setPattern( pattern );
    return fi;
}

bool NM_Filter::operator==( const NM_Filter &other ) const
{
    if ( _check != other._check ) return false;

    if ( pattern == NULL && other.pattern == NULL ) return true;

    if ( pattern != other.pattern ) return false;

    if ( strcmp( pattern, other.pattern ) == 0 ) return true;

    return false;
}
